/* WMVSMWin.C - Window Maker Visual Sound Monitor Window(GUI)
 *
 *  wmvsm Window Maker Visual Sound Monitor
 * 
 *  Copyright (c) 1998-1999 Motoyasu Yamanaka
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include	<stdio.h>
#include	<string.h>
#include	<unistd.h>
#include	<X11/Xlib.h>
#include	<X11/Xutil.h>
#include	<X11/extensions/shape.h>
#include	<X11/xpm.h>

#include	<signal.h>

#include	"FFT.h"
#include	"WMVSMWin.h"

#include	"XPM/wmvsm.xpm"			/* wmvsm */
#include	"XPM/tile.xpm"			/* tile image */
#include	"XPM/work.xpm"			/* background image */
#include	"XPM/button_down.xpm"		/* button down image */
#include	"XPM/stopm.xpm"			/* stop mark */
#include	"XPM/vu_light.xpm"		/* VU light image */
#include	"XPM/spectrum_light.xpm"	/* spectrum light image */

/*
 size, position

 o WMVSM
        ___(DSIPLAY_X(6), DISPLAY_Y(6))
       /
   ___/______64____________
  /  /                     \
 +--/-----------------------+
 | +----------------------+ |\
 | |                      | | |
 | |                      | | |
 | |                      | | |
 | |     display area     | | |
 | |                      | | |
 | |                      | | 64
 | |                      | | |
 | |                      | | |
 | +----------------------+ | |
 |             +----++----+ | |
 |             |stop|| ch | | |
 |             |    ||    | | |
 |             +----++----+ |/
 +--------------------------+

*/
#define	DISPLAY_X	6
#define	DISPLAY_Y	6
/*

 o display area

     ____DISPLAY_W(52)___
    /                    \
   +----------------------+
   |                      |\
   |                      | |
   |                      | |
   |     display area     | DISPLAY_H(37)
   |                      | |
   |                      | |
   |                      | |
   |                      |/
   +----------------------+

*/
#define	DISPLAY_W	52
#define	DISPLAY_H	37
/*

 o button        _BUTTON_W(16)
                /  \ 
               +----++----+
               |stop|| ch |\BUTTON_H(13)
               |    ||    |/
               +----++----+

*/
#define	BUTTON_W	16
#define	BUTTON_H	13


/*
 * global
 */
static	int	cease;
static
void	sigview(int sig)
{
#if	0
	fprintf(stderr, "%s\n", strsignal(errno));
#endif
	cease = 1;

	return;
}

static	WINMGR	wmgr;

#define	SET_XSH_FLG(flg) \
(((flg)&DRAWN_MODE)? (PMinSize| PMaxSize| PPosition): \
(PMinSize| PMaxSize))
#define	SET_XWMH_FLG(flg) \
(((flg)&DRAWN_MODE)? (WindowGroupHint| StateHint| IconWindowHint): \
WindowGroupHint| StateHint)

static
int	hints_setting(int ac, char **av, int flag)
{
	XSizeHints	*xsh;
	XWMHints	*xwmh;
	XTextProperty	wname, iname;
	XClassHint	xch = {NAME, CLASS};

	char	*ptr = NAME;

	xsh = XAllocSizeHints();
	if (!xsh) {
		return	-1;
	}
	xsh->min_width = xsh->min_height = 64,
	xsh->max_width = xsh->max_height = 64;
	xsh->flags = SET_XSH_FLG(flag);
	xsh->x = xsh->y = 0;

	xwmh = XAllocWMHints();
	if (!xwmh) {
		return	-1;
	}
	xwmh->window_group = wmgr.w;
	xwmh->initial_state = (flag & DRAWN_MODE)? WithdrawnState: NormalState,
	xwmh->flags = SET_XWMH_FLG(flag);
	xwmh->icon_window = (flag & DRAWN_MODE)? wmgr.icon: 0;

	XStringListToTextProperty(&ptr, 1, &wname);
	memcpy(&iname, &wname, sizeof wname);
	/* inform window manager of 3 hints
	 */
	XSetWMProperties(wmgr.d, wmgr.w,
			 &wname, &iname, av, ac, xsh, xwmh, &xch);

	return	0;
}

static
int	xpm_setting(Window root, int flag)
{
	int	r;
	XpmAttributes	xpmatt;

	memset(&xpmatt, 0, sizeof xpmatt);
	xpmatt.exactColors = False;
	xpmatt.closeness = 40000;	/* 8 bit display */
	xpmatt.valuemask = XpmExactColors| XpmCloseness;
	/* wmvsm pixmap
	 */
	r = XpmCreatePixmapFromData(wmgr.d, root, wmvsm_xpm,
				    &wmgr.fore, &wmgr.mask, &xpmatt);
	if (r != XpmSuccess) {
		return	-1;
	}
	/* tile image
	 */
	r = XpmCreatePixmapFromData(wmgr.d, root, tile_xpm,
				    &wmgr.tile, NULL, &xpmatt);
	if (r != XpmSuccess) {
		return	-1;
	}
	/* button down image
	 */
	r = XpmCreatePixmapFromData(wmgr.d, root, button_down_xpm,
				    &wmgr.button, NULL, &xpmatt);
	if (r != XpmSuccess) {
		return	-1;
	}
	/* work pixmap
	 */
	r = XpmCreatePixmapFromData(wmgr.d, root, work_xpm,
				    &wmgr.work, NULL, &xpmatt);
	if (r != XpmSuccess) {
		return	-1;
	}
	/* VU light LOW, HIGHT pixmap
	 */
	r = XpmCreatePixmapFromData(wmgr.d, root, vu_light_xpm,
				    &wmgr.vulight, NULL, &xpmatt);
	if (r != XpmSuccess) {
		return	-1;
	}
	/* spectrum light LOW, HIGHT pixmap
	 */
	r = XpmCreatePixmapFromData(wmgr.d, root, spectrum_light_xpm,
				    &wmgr.splight, NULL, &xpmatt);
	if (r != XpmSuccess) {
		return	-1;
	}
	/* Stop mark (this position is work same position) pixmap
	 */
	r = XpmCreatePixmapFromData(wmgr.d, root, stopm_xpm,
				    &wmgr.stop, NULL, &xpmatt);
	if (r != XpmSuccess) {
		return	-1;
	}

	wmgr.roof = XCreatePixmap(wmgr.d, root, 64, 64,
		DefaultDepth(wmgr.d, DefaultScreen(wmgr.d))
	);

	if (flag & DRAWN_MODE || flag & SHAPE_MODE) {
		XShapeCombineMask(wmgr.d, *wmgr.activ,
				  ShapeBounding, 0, 0, wmgr.mask, ShapeSet);
	} else {
		XCopyArea(wmgr.d, wmgr.tile,
			  wmgr.roof, wmgr.gc, 0,0, 64,64, 0,0);
	}
	XSetClipMask(wmgr.d, wmgr.gc, wmgr.mask);
	XCopyArea(wmgr.d, wmgr.fore, wmgr.roof, wmgr.gc, 0,0, 64,64, 0,0);
	XSetClipMask(wmgr.d, wmgr.gc, None);

	return	0;
}

int	wmvsm_init(OPTMGR *optmgr)
{
	Window	root;
	long	ev_mask;
	XGCValues       xgcv;
	unsigned long	gc_mask;
	XColor	xcolor;
	XWindowAttributes	xwa;
	
	memset(&wmgr, 0, sizeof wmgr);
	
	/* ritual
	 */
	wmgr.d = XOpenDisplay(optmgr->display);
	if (!wmgr.d) {
		fprintf(stderr, "%s: can't open display named %s\n",
			optmgr->av[0],
			XDisplayName(optmgr->display));
		return	-1;
	}
	root = RootWindow(wmgr.d, DefaultScreen(wmgr.d));
	wmgr.w = XCreateSimpleWindow(
		wmgr.d,
		root, 0, 0, 64, 64, 0,
		WhitePixel(wmgr.d, DefaultScreen(wmgr.d)),
		BlackPixel(wmgr.d, DefaultScreen(wmgr.d)));
	wmgr.icon = (optmgr->flag & DRAWN_MODE)? XCreateSimpleWindow(
		wmgr.d, root, 0, 0, 64, 64, 0,
		WhitePixel(wmgr.d, DefaultScreen(wmgr.d)),
		BlackPixel(wmgr.d, DefaultScreen(wmgr.d))): 0L;

	wmgr.activ = (optmgr->flag & DRAWN_MODE)? &wmgr.icon: &wmgr.w;
	/* hints
	 */
	if (hints_setting(optmgr->ac, optmgr->av, optmgr->flag)) {
		return	-1;
	}
	/* for Window Maker
	 */
	wmgr.deleteAtom = XInternAtom(wmgr.d, "WM_DELETE_WINDOW", False);
	XSetWMProtocols(wmgr.d, *wmgr.activ, &wmgr.deleteAtom, 1);
	/* ritual (GC)
	 */
	gc_mask = GCForeground| GCBackground| GCGraphicsExposures;
		
	xgcv.graphics_exposures = False;
	xgcv.foreground = WhitePixel(wmgr.d, DefaultScreen(wmgr.d)),
	xgcv.background = BlackPixel(wmgr.d, DefaultScreen(wmgr.d));
	wmgr.gc = XCreateGC(wmgr.d, root, gc_mask, &xgcv);
	/* ritual (XPM)
	 */
	if (xpm_setting(root, optmgr->flag)) {
		return	-1;
	}
	ev_mask = ExposureMask| ButtonPressMask| ButtonReleaseMask;
	XSelectInput(wmgr.d, *wmgr.activ, ev_mask);
	XMapWindow(wmgr.d, wmgr.w);
 
	/* scope mode color (default: cyan)
	 */
	memset(&xcolor, 0, sizeof xcolor);
	XGetWindowAttributes(wmgr.d, root, &xwa);
	XParseColor(wmgr.d, xwa.colormap, "cyan", &xcolor);
	xcolor.flags = DoRed| DoGreen| DoBlue;
	XAllocColor(wmgr.d, xwa.colormap, &xcolor);
	XSetForeground(wmgr.d, wmgr.gc, xcolor.pixel);

	return	0;	/* make it */
}

void	wmvsm_exit(void)
{
	if (wmgr.d) {
		XFreeGC(wmgr.d, wmgr.gc);
		XFreePixmap(wmgr.d, wmgr.roof);
		XFreePixmap(wmgr.d, wmgr.tile);
		XFreePixmap(wmgr.d, wmgr.mask);
		XFreePixmap(wmgr.d, wmgr.fore);
		XFreePixmap(wmgr.d, wmgr.button);
		XFreePixmap(wmgr.d, wmgr.vulight);
		XFreePixmap(wmgr.d, wmgr.splight);
		XFreePixmap(wmgr.d, wmgr.work);
		XFreePixmap(wmgr.d, wmgr.stop);
		XDestroyWindow(wmgr.d, wmgr.w);
		if (wmgr.icon)
			XDestroyWindow(wmgr.d, wmgr.icon);
		XCloseDisplay(wmgr.d);
	}

	return;
}

static
void	redraw(void)
{
	XEvent	xe;

	XCopyArea(wmgr.d, wmgr.roof, *wmgr.activ, wmgr.gc,
		  0,0, 64,64, 0,0);
	while (XCheckTypedEvent(wmgr.d, Expose, &xe));

	return;
}

/* work.xpm

   7       
  / \ 
 +---+------------+--------------+
 |                |              |\
 | L                             | |
 |                |    cleare    | 37
 |                |     area     | |
 | R                             | |
 |                |              |/
 +---+------------+--------------+
  \                             /
    ~~~~~~~~~~~~ 109 ~~~~~~~~~~~

*/
static
void	display_clear(void)
{
	XCopyArea(wmgr.d, wmgr.work, wmgr.work, wmgr.gc,
		  7 +45 +7, 0, 109, 37, 7, 0);
	XCopyArea(wmgr.d, wmgr.work, wmgr.roof, wmgr.gc,
		  7 +45 +7, 0, 109, 37, DISPLAY_X, DISPLAY_Y);

	return;
}

#define	DEST_C_POS_X	43
#define	DEST_S_POS_X	27
#define	DEST_POS_Y	47
static
void	draw_change_button(int val)
{
	int	x, y, w, h, dx, dy;
	Pixmap	src_pix = (val == LOW)? wmgr.button: wmgr.fore;

	x = (val == LOW)? 16: 43,
	y = (val == LOW)? 0: 47,
	w = (val == LOW)? 32: 16,
	h = BUTTON_H,		/* (13) */
	dx = DEST_C_POS_X,	/* (43) */
	dy = DEST_POS_Y;	/* (47) */

	XCopyArea(wmgr.d, src_pix, wmgr.roof, wmgr.gc, x,y, w,h, dx,dy);

	return;
}

static
void	draw_stop_button(int val)
{
	int	x, y, w, h, dx, dy;
	Pixmap	src_pix = (val == LOW)? wmgr.button: wmgr.fore;

	x = (val == LOW)? 0: 27,
	y = (val == LOW)? 0: 47,
	w = BUTTON_W,		/* (16) */
	h = BUTTON_H,		/* (13) */
	dx = DEST_S_POS_X,	/* (27) */
	dy = DEST_POS_Y;	/* (47) */

	XCopyArea(wmgr.d, src_pix, wmgr.roof, wmgr.gc, x,y, w,h, dx,dy);

	return;
}


static
void	draw_stop_mark(void)
{
	XCopyArea(wmgr.d, wmgr.stop, wmgr.roof, wmgr.gc,
		  0,0, DISPLAY_W,DISPLAY_H, DISPLAY_X,DISPLAY_Y); 

	return;
}

#define	POS_JUDGE(X, Y, DX, DY)	\
(((X)>=(DX))&&((Y)>=(DY))&&((X)<=((DX)+BUTTON_W))&&((Y)<=((DY)+(BUTTON_H))))

#define	IGAIN_VAL(mod)	((mod) == MODE_SCOPE)?  80:\
((mod) == MODE_VUDISP)? 45: /* MODE_SPECTRUM */ 95

static
void	press(int x, int y, SOUND_DEV_MGR *sdmgr, OPTMGR *optmgr)
{
#if	1
	if (POS_JUDGE(x, y, DEST_C_POS_X, DEST_POS_Y) &&
	    !(wmgr.buttonmgr & STOP)) {
#else	/* TODO: equip oneself for wmvsm preference */
	if (POS_JUDGE(x, y, DEST_C_POS_X, DEST_POS_Y)) {
#endif
		draw_change_button(LOW);
		display_clear();
		wmgr.buttonmgr |= CBUTTON;	/* LOW (ON) */
		optmgr->mode = (++(optmgr->mode) > MODE_SCOPE)?
			MODE_SPECTRUM: optmgr->mode;
		optmgr->devopt.rate = (optmgr->mode == MODE_SCOPE)?
			8000L: 24000;
		sd_restart(sdmgr, &optmgr->devopt);
		sd_mix_igain(sdmgr, IGAIN_VAL(optmgr->mode));
	} else if (POS_JUDGE(x, y, DEST_S_POS_X, DEST_POS_Y)) {
		draw_stop_button((!(wmgr.buttonmgr & STOP))? LOW: HIGH);
	}
	redraw();

	return;
}

static
void	release(int x, int y, SOUND_DEV_MGR *sdmgr, OPTMGR *optmgr)
{
	if (wmgr.buttonmgr & CBUTTON) {
		wmgr.buttonmgr &= ~CBUTTON;		/* HIGH (OFF) */
		draw_change_button(HIGH);
	} else if (POS_JUDGE(x, y, DEST_S_POS_X, DEST_POS_Y)) {
		if (wmgr.buttonmgr & STOP) {
			wmgr.buttonmgr &= ~STOP;	/* HIGH (OFF) */
			sd_flush(sdmgr);
			if (sd_restart(sdmgr, &optmgr->devopt) == 0) {
				XCopyArea(wmgr.d, wmgr.work,
					  wmgr.work, wmgr.gc,
					  7 + 45, 0, 45, 104, 7, 0);
				XCopyArea(wmgr.d, wmgr.work,
					  wmgr.roof, wmgr.gc,
					  7 + 45,0,
					  45,104,
					  DISPLAY_X,DISPLAY_Y); 
			} else {
				wmgr.buttonmgr |= STOP;      /* LOW (ON) */
				draw_stop_button(LOW);
			}
		} else {
			wmgr.buttonmgr |= STOP;      /* LOW (ON) */
			sd_stop(sdmgr);
			draw_stop_mark();
		}
	} else {
		draw_stop_button((wmgr.buttonmgr & STOP)? LOW: HIGH);
	}
	redraw();

	return;
}


/*

 device raw data value 
    8 bit                 
     dev app(mono) (streo)
     --- --------- -------
     256        37     16  /\
              
     128        19      8  --

       0         0      0  \/

 scope ratio
  stereo
    8 bit  16 /   256 = 0.0625

  mono
    8 bit  37 /   256 = 0.014453125

*/
#define	MONO_RATIO_8	0.14453125
#define	STEREO_RATIO_8	0.0625
/*
 scope geometory (stereo)
      ____(13, 7)
     /   __________(47)
   \/  \/
   +------------
 R |}<--(16)

     _____(13, 25)
   \/
   +-------
 L |

        _______START_POS (8)
  +----/---------------+
  |   /                |\
  | L ---------------- | |
  |   \ SCOPE_DISP_W / | |<----DISPLAY_H(37)
  |        (44)        | |
  | R ---------------- | |
  |                    |/
  +--------------------+
   \__________________/
         /|
          +-------DISPLAY_W(52)
*/
#define	START_POS	7
#define	LABEL_W	7
#define	SCOPE_DISP_W	44
#define	SCOPE_DISP_SQUARE	86

static
void	scope_clear()
{
	XCopyArea(wmgr.d, wmgr.work, wmgr.work, wmgr.gc,
		  LABEL_W + 45, 0,
		  45, 104,
		  LABEL_W,0);
	return;
}

static
void	scope(SOUND_DEV_MGR *sdmgr, int channel)
{
	int	count = 0;
	int	i, j;
	if (channel == MONO) {	/* mono */
		XPoint	p[SCOPE_DISP_SQUARE];
		unsigned char	raw[SCOPE_DISP_W -1];
		while (1) {
			scope_clear();
#if	0
			sd_get_rawdata(sdmgr, raw, SCOPE_DISP_W -1);
#else
			read(sdmgr->d, raw, sizeof raw);
#endif
			for (i = 0, j; i < SCOPE_DISP_SQUARE -1; i += 2) {
				j = i >> 1L;
				p[i].x = LABEL_W +j,
				p[i].y = (short)raw[j] * MONO_RATIO_8,
				p[i +1].x = LABEL_W +j +1,
				p[i +1].y = (short)raw[j +1] * MONO_RATIO_8;
			}
			XDrawLines(wmgr.d, wmgr.work, wmgr.gc,
				   p, 85, CoordModeOrigin);
			XCopyArea(wmgr.d, wmgr.work, *wmgr.activ, wmgr.gc,
				  0, 0,
				  DISPLAY_W, DISPLAY_H,
				  DISPLAY_X, DISPLAY_Y);
			++count;
			if (count > 3) {
				break;
			}
		}
	} else {	/* stereo */
		unsigned char	raw[SCOPE_DISP_SQUARE];
		XPoint	pl[SCOPE_DISP_SQUARE],
			pr[SCOPE_DISP_SQUARE];
		while (1) {
			/* screen  clear */
			scope_clear();
#if	0
			sd_get_rawdata(sdmgr, raw, sizeof raw);
#else
			read(sdmgr->d, raw, sizeof raw);
#endif
			for (i = 0; i < SCOPE_DISP_SQUARE -1; i += 2) {
				j = i >> 1L;
				pl[i].x = 
				pr[i].x = LABEL_W +j,
				pl[i].y = (short)raw[i] * STEREO_RATIO_8,
				pr[i].y = (short)raw[i +1] * STEREO_RATIO_8 +18,
				pl[i +1].x = 
				pr[i +1].x = LABEL_W +j +1,
				pl[i +1].y = (short)raw[i +2] * STEREO_RATIO_8,
				pr[i +1].y = (short)raw[i +3] * STEREO_RATIO_8 + 18;
			}
			XDrawLines(wmgr.d, wmgr.work, wmgr.gc,
				   pl, 85, CoordModeOrigin);
			XDrawLines(wmgr.d, wmgr.work, wmgr.gc,
				   pr, 85, CoordModeOrigin);
			XCopyArea(wmgr.d, wmgr.work, *wmgr.activ, wmgr.gc,
				  0, 0,
				  DISPLAY_W, DISPLAY_H,
				  DISPLAY_X, DISPLAY_Y);
			++count;
			if (count > 3) {
				break;
			}
		}
	}

	return;
}

#define	SP_LIGHT_H	25
static
void	spectrum(SOUND_DEV_MGR *sdmgr, int channel)
{
	int	i, j, k,
		spv_max;

	static	int	spv_th[8] = {5, 32, 32, 32, 32, 32, 32, 20,};

	if (channel == MONO) {	/* mono */
		COMPLEX	freq[512 +1];	/* WMVSM_BUFSIZ(1024) / 2 +1 -> 512 +1 */
		unsigned char	raw[512];

		int	spv[8],
			a0;

		sd_get_rawdata(sdmgr, raw, sizeof raw);
		memset(freq, 0, sizeof freq);
		for (i = 0; i < fft_size(); i++)
			freq[i].real = (float)raw[i];

		fft(freq, 0);
		
		memset(spv, 0, sizeof spv);
		spv_max = spv_th[0] + spv_th[1] + spv_th[2] + spv_th[3]
			+ spv_th[4] + spv_th[5] + spv_th[6] + spv_th[7];
	
		a0 = (int)hypot(freq[0].imag, freq[0].real);

		for (k = i = 0; i < spv_max; i++) {
			for (j = 0; j < spv_th[k]; j++) {
				spv[k] += (int)hypot(freq[i +1 +j].imag,
						    freq[i +1 +j].real);
			}
			spv[k] = spv[k] * (SP_LIGHT_H) / a0;

			i += j;
			++k;
		}
		XCopyArea(wmgr.d, wmgr.splight,
			  wmgr.work, wmgr.gc, 0,0, DISPLAY_W,DISPLAY_H, 7,0);
		for (i = 0; i < 8; i++) {
			if (spv[i] > 0) spv[i] += 3;	/* astute */
			if (spv[i] > SP_LIGHT_H) spv[i] = SP_LIGHT_H;
			XCopyArea(wmgr.d, wmgr.splight, wmgr.work, wmgr.gc,
				  DISPLAY_W, 4,
				  54, SP_LIGHT_H,
				  8 + i * 3, 4);
			XCopyArea(wmgr.d, wmgr.splight, wmgr.work, wmgr.gc,
				  1, 4,
				  2, SP_LIGHT_H -spv[i],
				  8 + i * 3, 4);

		}
	} else {	/* stereo */
		COMPLEX	lfreq[512 +1],	/* WMVSM_BUFSIZ(1024) / 2 +1 -> 512 +1 */
			rfreq[512 +1];
		unsigned char	raw[WMVSM_BUFSIZ];
		unsigned char	left[512],
				right[512];
		int	lspv[16],
			*rspv;
		int	l_a0,
			r_a0;
		sd_get_rawdata(sdmgr, raw, sizeof raw);
		sd_split_buffer(raw, sizeof raw, left, right, SAMPLE_SIZE_8_BIT);
		/* left channel
		 */
		memset(lfreq, 0, sizeof lfreq);
		for (i = 0; i < fft_size(); i++)
			lfreq[i].real = (float)left[i];
		fft(lfreq, 0);
		/* right channel
		 */
		memset(rfreq, 0, sizeof rfreq);
		for (i = 0; i < fft_size(); i++)
			rfreq[i].real = (float)right[i];
		fft(rfreq, 0);

		memset(lspv, 0, sizeof lspv);
		rspv = lspv + 8;
		spv_max = spv_th[0] + spv_th[1] + spv_th[2] + spv_th[3]
			+ spv_th[4] + spv_th[5] + spv_th[6] + spv_th[7];
	
		l_a0 = (int)hypot(lfreq[0].imag, lfreq[0].real),
		r_a0 = (int)hypot(rfreq[0].imag, rfreq[0].real);
		for (k = i = 0; i < spv_max; i++) {
			for (j = 0; j < spv_th[k]; j++) {
				lspv[k] += (int)hypot(
					lfreq[i +1 +j].imag,
					lfreq[i +1 +j].real);
				rspv[k] += (int)hypot(
					rfreq[i +1 +j].imag,
					rfreq[i +1 +j].real);
			}
			lspv[k] = lspv[k] * (SP_LIGHT_H) / l_a0;
			rspv[k] = rspv[k] * (SP_LIGHT_H) / r_a0;

			i += j;
			++k;
		}
		XCopyArea(wmgr.d, wmgr.splight,
			  wmgr.work, wmgr.gc, 0,0, DISPLAY_W,DISPLAY_H, 7,0);
		for (i = 0; i < 8; i++) {
			if (lspv[i] > 0) lspv[i] += 3;	/* astute */
			if (rspv[i] > 0) rspv[i] += 3;	/* astute */
			if (lspv[i] > SP_LIGHT_H) lspv[i] = SP_LIGHT_H;
			if (rspv[i] > SP_LIGHT_H) rspv[i] = SP_LIGHT_H;
			XCopyArea(wmgr.d, wmgr.splight, wmgr.work, wmgr.gc,
				  DISPLAY_W, 4,
				  54, SP_LIGHT_H,
				  8 + i * 3, 4);
			XCopyArea(wmgr.d, wmgr.splight, wmgr.work, wmgr.gc,
				  1, 4,
				  2, SP_LIGHT_H -lspv[i],
				  8 + i * 3, 4);
			XCopyArea(wmgr.d, wmgr.splight, wmgr.work, wmgr.gc,
				  DISPLAY_W, 4,
				  54, SP_LIGHT_H,
				  27 + 8 + i * 3, 4);
			XCopyArea(wmgr.d, wmgr.splight, wmgr.work, wmgr.gc,
				  1, 4,
				  2, SP_LIGHT_H -rspv[i],
				  27 + 8 + i * 3, 4);
		}
	}
	XCopyArea(wmgr.d, wmgr.work, *wmgr.activ, wmgr.gc,
		  7,0, DISPLAY_W,DISPLAY_H, DISPLAY_X,DISPLAY_Y);

	return;
}

#define	ABS(n)	(((n)<0)? (-(n)): (n))
#define	VU_W	44
#define	VU_H	14
static
void	vu(SOUND_DEV_MGR *sdmgr, int channel)
{
	int	 i, n;

	if (channel == MONO) {	/* mono */
		int	val;
		unsigned char	raw[512];
		sd_get_rawdata(sdmgr, raw, sizeof raw);
		for (val = i = 0; i < 512; i++) {
			n = raw[i] -128;
			n = ABS(n);
			if (val < n)
				val = n;
		}
		val = val * VU_W >> 7L;
		XCopyArea(wmgr.d, wmgr.vulight, wmgr.work, wmgr.gc,
			  0, 0,
			  VU_W, VU_H,
			  START_POS, 3);
		XCopyArea(wmgr.d, wmgr.vulight, wmgr.work, wmgr.gc,
			  val, VU_H,
			  VU_W, VU_H << 1L,
			  val+ START_POS, 3);
	} else {	/* stereo */
		int lval, rval, size;
		unsigned char	raw[WMVSM_BUFSIZ];
		unsigned char	left[WMVSM_BUFSIZ], *right;
		right = left + 512;
		
		sd_get_rawdata(sdmgr, raw, sizeof raw);
		size = sd_split_buffer(raw, WMVSM_BUFSIZ, left, right,
				       SAMPLE_SIZE_8_BIT);

		for (lval = rval = i = 0; i < size; i++) {
			n = left[i] -128;
			n = ABS(n);
			if (lval < n)
				lval = n;
			n = right[i] -128;
			n = ABS(n);
			if (rval < n)
				rval = n;
		}
		XCopyArea(wmgr.d, wmgr.vulight, wmgr.work, wmgr.gc,
			  0, 0,
			  VU_W, VU_H,
			  START_POS, 3);
		XCopyArea(wmgr.d, wmgr.vulight, wmgr.work, wmgr.gc,
			  0, 0,
			  VU_W, VU_H,
			  START_POS, 20);
		lval = lval * VU_W >> 7L;
		XCopyArea(wmgr.d, wmgr.vulight, wmgr.work, wmgr.gc,
			  lval, VU_H,
			  VU_W, VU_H << 1L,
			  lval+ START_POS, 3);
		rval = rval * VU_W >> 7L;
		XCopyArea(wmgr.d, wmgr.vulight, wmgr.work, wmgr.gc,
			  rval, VU_H,
			  VU_W, VU_H << 1L,
			  rval+ START_POS, 20);
	}
	XCopyArea(wmgr.d, wmgr.work, *wmgr.activ, wmgr.gc,
		  0,0, DISPLAY_W,DISPLAY_H, DISPLAY_X,DISPLAY_Y);

	return;
}


int	wmvsm_run(OPTMGR *optmgr)
{
	SOUND_DEV_MGR	sdmgr;
	XEvent	xe;
	
	void	(*__mode[])(SOUND_DEV_MGR *, int) = {
		spectrum,
		vu,
		scope,
		0,
	};
	
#ifdef	__NO_DEV
	if (sd_init2(&sdmgr, "DATA")) {
#else
	if (sd_init(&sdmgr, &optmgr->devopt)) {
#endif	/* __NO_DEV */
		press(DEST_S_POS_X, DEST_POS_Y, &sdmgr, optmgr);
		release(DEST_S_POS_X, DEST_POS_Y, &sdmgr, optmgr);
		sleep(2);
	}

	signal(SIGINT, sigview);
	signal(SIGHUP, sigview);
	signal(SIGKILL, sigview);
	signal(SIGTERM, sigview);
	signal(SIGPIPE, sigview);
	signal(SIGCHLD, SIG_IGN);

	sd_mix_igain(&sdmgr, IGAIN_VAL(optmgr->mode));

	if (fft_init(9))	/* pow(2, 9) = 512 */
		return	-1;

	while (1) {
		if (!(wmgr.buttonmgr & STOP)) {
			(*__mode[optmgr->mode])(&sdmgr, optmgr->devopt.ch);
		}
		while (XPending(wmgr.d)) {
			XNextEvent(wmgr.d, &xe);
			switch (xe.type) {
			case	Expose:
				redraw();
				break;
			 
			case	ButtonPress:
				press(xe.xbutton.x, xe.xbutton.y, &sdmgr, optmgr);
				break;

			case	ButtonRelease:
				release(xe.xbutton.x, xe.xbutton.y, &sdmgr, optmgr);
				break;

			case	DestroyNotify:
fprintf(stderr, "DestroyNotify\n");
				break;

			case	ClientMessage:
				if (xe.xclient.data.l[0] == (long)wmgr.deleteAtom) {
					return	0;
				}

			default:
				break;
			}
		}
		if (cease)
			break;
		usleep(500L);
	}
	fft_exit();

	return	0;
}
