/* SoundDev.c - Sound Device (DSP, MIXER device)
 *
 *  wmvsm Window Maker Visual Sound Monitor
 * 
 *  Copyright (c) 1998-1999 Motoyasu Yamanaka
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	<stdio.h>
#include	"SoundDev.h"

#define	MKMSG(f, req)	#f "[" #req "]"
int	sd_init(SOUND_DEV_MGR *mgr, DEVOPT *devopt)
{
	int status;
	
	int	recsrc,
		recmask;
	
	memset(mgr, 0, sizeof *mgr);
	mgr->raw.size = WMVSM_BUFSIZ * 2;
	
  	/* open dsp device */
	mgr->d = open(devopt->device, O_RDONLY);
	if (mgr->d < 0) {
		perror(devopt->device);
		return	-1;
	}
  	/* sample size */
	/* sampling format (auto) with the proviso linux */
	status = ioctl(mgr->d, SOUND_PCM_WRITE_BITS, &devopt->bit);
	if (status < 0) {
		perror(MKMSG(ioctl, SOUND_PCM_WRITE_BITS));
		return	-1;
	}
	if (devopt->bit == SAMPLE_SIZE_8_BIT) {
		memset(mgr->raw.buffer, 128, sizeof mgr->raw.buffer);
	} else {
#if	__TODO
		memset(mgr->raw.buffer, 128, sizeof mgr->raw.buffer);
#endif
	}
  	/* channel */
	status = ioctl(mgr->d, SOUND_PCM_WRITE_CHANNELS, &devopt->ch);
	if (status < 0) {
		perror(MKMSG(ioctl, SOUND_PCM_WRITE_CHANNELS));
		return	-1;
	}
	/* sampling rate */
	status = ioctl(mgr->d, SOUND_PCM_WRITE_RATE, &devopt->rate);
	if (status < 0) {
		perror(MKMSG(ioctl, SOUND_PCM_WRITE_RATE));
	}
	ioctl(mgr->d, SOUND_PCM_READ_RATE, &devopt->rate);
	/* DMA buffer size */
	status = ioctl(mgr->d, SOUND_PCM_SUBDIVIDE, &devopt->dma);
	if (status < 0) {
		perror(MKMSG(ioctl, SOUND_PCM_SUBDIVIDE));
		return	-1;
	}
	/* Mixer recording source (enable CD device) */
	status = ioctl(mgr->d, SOUND_MIXER_READ_RECMASK, &recmask);
	if (status < 0) {
		perror(MKMSG(ioctl, SOUND_READ_RECMASK));
		return	-1;
	}
	status = ioctl(mgr->d, SOUND_MIXER_READ_RECSRC, &recsrc);
	if (status < 0) {
		perror(MKMSG(ioctl, SOUND_READ_RECSRC));
		return	-1;
	}
	if (((1L << SOUND_MIXER_CD) & recmask) &&
	   !((1L << SOUND_MIXER_CD) & recsrc)) {
		recsrc |= 1L << SOUND_MIXER_CD;
		status = ioctl(mgr->d, SOUND_MIXER_WRITE_RECSRC, &recsrc);
		if (status < 0) {
			perror(MKMSG(ioctl, SOUND_READ_RECSRC));
			return	-1;
		}
	}

	return	0;	/* make it */
}

#if	__NO_DEV
int	sd_init2(SOUND_DEV_MGR *mgr, const char *name)
{
	memset(mgr, 0, sizeof *mgr);
	mgr->raw.size = WMVSM_BUFSIZ * 2;
  	/* open dsp device or data file */
	mgr->d = open(name, O_RDONLY);
	if (mgr->d < 0) {
		perror(name);
		return	-1;
	}

	return	0;
}
#endif

int	sd_get_rawdata(SOUND_DEV_MGR *mgr, void *const dest, const int size)
{
	void	*buf;
	size_t	count,
		status;

	RAWDAT	*raw;

	raw = &mgr->raw;
	
	if (raw->current_size < size) {
		buf = (void *)raw->buffer + raw->current_size;
		count = (size_t)raw->size - raw->current_size;
		status = read(mgr->d, buf, count);
		if ((status != count) && (status <= 0)) {
			perror("read()");
			return	-1;
		}
		raw->current_size += status;
	}
	memcpy(dest, raw->buffer + raw->size -raw->current_size, size);
	raw->current_size -= size;

	return	0;
}

/*                     size
 *      _____________________________________
 *      /                                     \
 * src +---+---+---+---+---+---+-------+---+---+
 *     | L | R | L | R | L | R | o o o | L | R |
 *     +---+---+---+---+---+---+-------+---+---+
 *        |
 *        |
 *        | split
 *        |
 *        V    size / 2
 *      _______________________
 *     /                       \
 *   l +---+---+---+---+---+---+
 *     | L | L | L | L | L | L |
 *     +---+---+---+---+---+---+
 *   r +---+---+---+---+---+---+
 *     | R | R | R | R | R | R |
 *     +---+---+---+---+---+---+
 *
 * type: sample size
 *        8bit => unsigned char
 *       16bit => unsigned short
 */
int	sd_split_buffer(void *const src, int size, void *const l_dest, void *const r_dest, const int type)
{
	int i;
	if (type == SAMPLE_SIZE_8_BIT) {	/* sample size is 8 bit */
		unsigned char	*s, *l, *r;

		s = (unsigned char *)src,
		l = (unsigned char *)l_dest,
		r = (unsigned char *)r_dest;
		for (i = 0; i < size; i += 2) {
			*l++ = *s++;
			*r++ = *s++;
		}
	} else {	/* sample size is 16 bit */
		unsigned short	*s, *l, *r;

		s = (unsigned short *)src,
		l = (unsigned short *)l_dest,
		r = (unsigned short *)r_dest;
		for (i = 0; i < size; i += 2) {
			*l++ = *s++;
			*r++ = *s++;
		}
	}

	return	size >> 1L;
}

void	sd_flush(SOUND_DEV_MGR *const mgr)
{
	if (mgr->d) {
		ioctl(mgr->d, SNDCTL_DSP_SYNC, 0);
		mgr->raw.current_size = 0;
	}
	return;
}

void	sd_stop(SOUND_DEV_MGR *const mgr)
{
	if (mgr->d) {
		ioctl(mgr->d, SNDCTL_DSP_RESET, 0);
		mgr->raw.current_size = 0;
		close(mgr->d);
	}
	return;
}

int	sd_restart(SOUND_DEV_MGR *mgr, DEVOPT *devopt)
{
	sd_stop(mgr);
#ifdef	__NO_DEV
	return	sd_init2(mgr, "DATA");
#else
	return	sd_init(mgr, devopt);
#endif	/* __NO_DEV */
}

int	sd_mix_igain(SOUND_DEV_MGR *const mgr, const int val)
{
	int	level;
	level = (val << 8L) + val;
	if (ioctl(mgr->d, SOUND_MIXER_WRITE_IGAIN, &level) < 0) {
		return	-1;
	}

	return	0;
}

