/* OptMgr.c - Option analyze
 *
 *  wmvsm Window Maker Visual Sound Monitor
 * 
 *  Copyright (c) 1998-1999 Motoyasu Yamanaka
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	<string.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>

#include	"OptMgr.h"

static
void	usage(const char *name)
{
	fprintf(stderr,
		"%s Version. %s\n"
		"%s\n\n"
		"usage: %s  [option]\n"
		"  -h | --help        display this message\n"
		"  -v | --version     display version number\n"
		"       --display     target display (defalut: 0)\n"
		"  -s | --shape       non rect angular (shape) mode\n"
		"  -w | --withdraw    without drawn mode (for Window Maker)\n"
		"       --mode [spectum | VU | scope] starting mode\n"
		"\n"
		"  -d | --dev dspdev  dsp device (default: /dev/dsp)\n"
		"       --8bit        using  8 bit sample size (default)\n"
#if	0
		"       --16bit       using 16 bit sample size\n"
#endif	/* __TODO */
		"       --stereo      using stereo channel (default)\n"
		"       --mono        using monaural channel\n"
#if	__TODO
		"\n"
		"  -e | --exec [app]  app to start on click\n"
#endif	/* __TODO */
		,
		name,
		VERSION,
		DESIGN_BY,
		name
		);

	return;
}

void	opt_init(int argc, char **argv, OPTMGR *mgr)
{
	
	struct option opt[] = {
		{"help", 0, 0, 'h'},	/* 0 */
		{"version", 0, 0, 'v'},	/* 1 */
		{"display", 1, 0, 0},	/* 2 */
		{"device", 1, 0, 'd'},	/* 3 */
		{"shape", 0, 0, 's'},	/* 4 */
		{"withdraw", 0, 0, 'w'},/* 5 */
		{"8bit", 0, 0, 0},	/* 6 */
#if	__TODO
		{"16bit", 0, 0, 0},	/* X */
#endif	/* __TODO */
		{"stereo", 0, 0, 0},	/* 7 */
		{"mono", 0, 0, 0},	/* 8 */
		{"mode", 1, 0, 'm'},	/* 9 */
#if	__TODO
		{"exec", 1, 0, 'e'},	/*10 */
#endif	/* __TODO */
		{0, 0, 0, 0},
	};
	const	char	*optstr = "hvswid:b:";
	char	*ptr;

	int	c,
		ind = 0;

	memset(mgr, 0, sizeof *mgr);
	strcpy(mgr->display, ":0");
	mgr->mode = MODE_SPECTRUM;
	mgr->flag = OTHER_WM_MODE;
	strcpy(mgr->devopt.device, DEV );	/* default /dev/dsp */
	mgr->devopt.bit = SAMPLE_SIZE_8_BIT;	/* sample size 8 bit */
	mgr->devopt.ch = STEREO;		/* stereo channel */
	mgr->devopt.rate = 24000;		/* sampling rate (24 KHz) */
	mgr->devopt.dma = 4;			/* dma */
	mgr->ac = argc;
	mgr->av = argv;

	while ((c = getopt_long(argc, argv, optstr, opt, &ind)) != -1) switch (c) {
	case	0:
		ptr = (optarg)? optarg: "";
		/* --display */
		if (!strcmp(opt[ind].name, opt[2].name)) {
			if (strlen(ptr) > 127){
				fprintf(stderr, "invalid display name: %s\n",
					ptr
					);
				exit(0);
			}
			strcpy(mgr->display, ptr);
		}
		/* --8bit */
		if (!strcmp(opt[ind].name, opt[6].name)) {
			mgr->devopt.bit = SAMPLE_SIZE_8_BIT;
		}
#if	__TODO
		/* --16bit */
		if (!strcmp(opt[ind].name, opt[7].name)) {
			mgr->devopt.bit = SAMPLE_SIZE_16_BIT;
		}
#endif	/* __TODO */
		/* --stereo */
		if (!strcmp(opt[ind].name, opt[7].name)) {
			mgr->devopt.ch = STEREO;
		}
		/* --mono */
		if (!strcmp(opt[ind].name, opt[8].name)) {
			mgr->devopt.ch = MONO;
		}
		break;
		
	case	'h':	/* --help */
		usage(argv[0]);
		exit(0);
		
	case	'v':	/* --version */
		fprintf(stderr, "%s Ver. %s\n%s\n", argv[0], VERSION, DESIGN_BY);
		exit(0);
		
	case	'd':	/* --device */
		ptr = (optarg)? optarg: "";
		if (strlen(ptr) > 127){
			fprintf(stderr, "invalid device name: %s\n", ptr);
			exit(1);
		}
		strcpy(mgr->devopt.device, ptr);
		break;
		
	case	's':	/* --shape */
		mgr->flag |= SHAPE_MODE;
		break;

	case	'w':	/* --withdraw */
		mgr->flag |= DRAWN_MODE;
		break;

	case	'm':	/* --mode */
		ptr = (optarg)? optarg: "";
		if (!strcmp(optarg, "spectrum")) {
			mgr->mode = MODE_SPECTRUM;
			mgr->devopt.rate = 24000;
		} else if (!strcmp(optarg, "VU")) {
			mgr->mode = MODE_VUDISP;
			mgr->devopt.rate = 24000;
		} else if (!strcmp(optarg, "scope")) {
			mgr->mode = MODE_SCOPE;
			mgr->devopt.rate = 8000L;
		}
		break;

#if	__TODO
	case	'e':	/* --exec */
		ptr = (optarg)? optarg: "";
		if (strlen(ptr) > 127 -1){
			fprintf(stderr, "invalid app name: %s\n", ptr);
			exit(1);
		}
		sprintf(mgr->app, "%s&", ptr);
		break;
#endif	/* __TODO */

	default:
		exit(1);
	}

	return;
}

void	opt_stat(OPTMGR *mgr)
{
	fprintf(stderr,
		"display: %s\n"
		"device : %s\n"
		"bit    : %d\n"
		"rate   : %d\n"
		"channel: %d\n"
		"flag   : %d\n"
		"mode   : %d\n",
		mgr->display,
		mgr->devopt.device,
		mgr->devopt.bit,
		mgr->devopt.rate,
		mgr->devopt.dma,
		mgr->flag,
		mgr->mode
		);
		
	return;
}
