/* FFT.c - fast Fourier transformation
 *
 *  wmvsm Window Maker Visual Sound Monitor
 * 
 *  Copyright (c) 1998-1999 Motoyasu Yamanaka
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<math.h>
#include	"FFT.h"

static	FFTMGR	fftmgr;

int	fft_size(void)
{
	return	1L << fftmgr.exp;
}

static
void make_sintbl(void)
{
	int i, n2, n4, n8, n;
	double c, s, dc, ds, t;

	n = 1L << fftmgr.exp;
	n2 = n / 2, n4 = n / 4, n8 = n / 8;
	t = sin(M_PI / n);
	dc = 2 * t * t;	 ds = sqrt(dc * (2 - dc));
	t = 2 * dc, c = fftmgr.sin_tbl[n4] = 1,  s = fftmgr.sin_tbl[0] = 0;
	for (i = 1; i < n8; i++) {
		c -= dc, dc += t * c;
		s += ds, ds -= t * s;
		fftmgr.sin_tbl[i] = s, fftmgr.sin_tbl[n4 -i] = c;
	}
	if (n8 != 0) fftmgr.sin_tbl[n8] = sqrt(0.5);
	for (i = 0; i < n4; i++)
		fftmgr.sin_tbl[n2 - i] = fftmgr.sin_tbl[i];
	for (i = 0; i < n2 + n4; i++)
		fftmgr.sin_tbl[i + n2] = - fftmgr.sin_tbl[i];

	return;
}

static
void	make_bitrev(void)
{
	int	i, j, k, n2, n;

	n = 1L << fftmgr.exp;
	n2 = n / 2, i = j = 0;
	while (1) {
		fftmgr.bit_rev[i] = j;
		if (++i >= n) break;
		k = n2;
		while (k <= j)
			j -= k, k /= 2;
		j += k;
	}

	return;
}

int	fft_init(unsigned long exp)
{
	unsigned long	bufsize;

	memset(&fftmgr, 0, sizeof fftmgr);
	bufsize  = 1L << exp;
	fftmgr.sin_tbl = (double*)calloc(
		 bufsize +bufsize / 4 +1, sizeof(double));
	if (!fftmgr.sin_tbl) {
		return	-1;
	}
	fftmgr.bit_rev = (int*)calloc(bufsize +1, sizeof(int));
	if (!fftmgr.bit_rev) {
		free(fftmgr.sin_tbl);
		return	-1;
	}

	fftmgr.exp = exp;

	make_sintbl();
	make_bitrev();

	return	0;
}

void	fft_exit(void)
{
	free(fftmgr.sin_tbl),
	free(fftmgr.bit_rev);
	fftmgr.sin_tbl = 0,
	fftmgr.bit_rev = 0,
	fftmgr.exp = 0;

	return;
}

void	fft(COMPLEX *const x, const unsigned int rev)
{
	unsigned long	n = 1L << fftmgr.exp;
	unsigned long	i, j, k, ik, h, d, k2, n4;
	double	s, c, dx, dy;
	n4 = n / 4;
	for (i = 0; i < n; i++) {
		COMPLEX *t;
		j = fftmgr.bit_rev[i];
		if (i < j)
			t = &x[i], x[i] = x[j], x[j] = *t;
	}

	for (k = 1; k < n; k = k2) {
		h = 0, k2 = k + k, d = n / k2;
		for (j = 0; j < k; j++) {
			c = fftmgr.sin_tbl[h + n4];
			if (rev) {
				s = -fftmgr.sin_tbl[h];
			} else {
				s = fftmgr.sin_tbl[h];
			}
			for (i = j; i < n; i += k2) {
				ik = i + k;
				dx = s * x[ik].imag + c * x[ik].real;
				dy = c * x[ik].imag - s * x[ik].real;
				x[ik].real = x[i].real -dx, x[i].real += dx;
				x[ik].imag = x[i].imag -dy, x[i].imag += dy;
			}
			h += d;
		}
	}
	if (!rev) {
		for (i = 0; i < n; i++)
			x[i].imag /= n, x[i].real /= n;
	}

	return;
}
