/*
 wmtz.c
 Shows local time in different timezones + JD.
 Jan Lindblom <jlin@algonet.se> 1999-04-15

 wmtz.c was derived from:

   wminet.c
   Multi-function system monitor
   Dave Clark (clarkd@skynet.ca) (http://www.neotokyo.org/illusion)
   Martijn Pieterse (pieterse@xs4all.nl) (http://windowmaker.mezaway.org)
   and Antoine Nulle (warp@xs4all.nl) (http://windowmaker.mezaway.org)

 This software is licensed through the GNU General Public Licence.
 Read the COPYING file for details.

*/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>
#include <math.h>

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include "../wmgeneral/wmgeneral.h"
#include "../wmgeneral/misc.h"

#include "wmtz.xpm"
char wmtz_mask_bits[64*64];
int  wmtz_mask_width = 64;
int  wmtz_mask_height = 64;

#define WMTZ_VERSION "0.3"
#define CHAR_WIDTH 5
#define CHAR_HEIGHT 7

typedef struct
{
    char label[10]; /* Time zone designation */
    int  diff;      /* Time zone diff. from UT in hours OR when label="JDN"...  */
} timezone_t;       /* ...numbers of days to subtract from JD */

timezone_t zone[6]; 
static struct tm *clk;
char uconfig_file[256];

void usage(void);
void printversion(void);
void BlitString(char *name, int x, int y);
void wmtz_routine(int, char **);
int ReadConfigInt(FILE *fp, char *setting, int *value);
int ReadConfigString(FILE *fp, char *setting, char *value);
int Read_Config_File( char *filename );

char *month[12];
char *week_day[7];

int main(int argc, char *argv[]) {

	int		i;

    uconfig_file[0] = 0;
    
	/* Parse Command Line */	
	for (i=1; i<argc; i++) {
		char *arg = argv[i];

		if (*arg=='-') {
			switch (arg[1]) {
			case 'd' :
				if (strcmp(arg+1, "display")) {
					usage();
					exit(1);
				}
				break;
			case 'g' :
				if (strcmp(arg+1, "geometry")) {
					usage();
					exit(1);
				}
				break;
			case 'v' :
				printversion();
				exit(0);
				break;
            case 'c' :
                if (argc > (i+1))
                {
                    strcpy(uconfig_file, argv[i+1]);
                    i++;
                }
                break;
            default:
				usage();
				exit(0);
				break;
			}
		}
	}

	month[0] = "JAN\0";  month[1] = "FEB\0";  month[2] = "MAR\0";
	month[3] = "APR\0";  month[4] = "MAY\0";  month[5] = "JUN\0";
	month[6] = "JUL\0";  month[7] = "AUG\0";  month[8] = "OCT\0";
	month[9] = "SEP\0";  month[10] = "NOV\0";  month[11] = "DEC\0";

	week_day[0] = "SUNDAY   \0";
	week_day[1] = "MONDAY   \0";
	week_day[2] = "TUESDAY  \0";
	week_day[3] = "WEDNESDAY\0";
	week_day[4] = "THURSDAY \0";
	week_day[5] = "FRIDAY   \0";
	week_day[6] = "SATURDAY \0";

	wmtz_routine(argc, argv);

	return 0;
}


/*******************************************************************************\
|* wmtz_routine								       *|
\*******************************************************************************/
void wmtz_routine(int argc, char **argv)
{
    int			j=0,k=0, hour=0, len=0, pos=0;
    double              jd;
    int                 swatch_beats;
    time_t		curtime;
    time_t		prevtime;
    char config_file[512];
    char buf[64];
    char blitstr[32];
    char* fracstr;
 
    createXBMfromXPM(wmtz_mask_bits, wmtz_master_xpm, wmtz_mask_width, wmtz_mask_height);
    
    openXwindow(argc, argv, wmtz_master_xpm, wmtz_mask_bits, wmtz_mask_width, wmtz_mask_height);

    memset(&zone, 0, sizeof(zone));

    // Read config file 
    if (uconfig_file[0] != 0)
    {
        // user-specified config file
        fprintf(stderr, "Using user-specified config file '%s'.\n", uconfig_file);
        Read_Config_File(uconfig_file);
    }
    else
    {
        sprintf(config_file, "%s/.wmtzrc", getenv("HOME"));

        if (!Read_Config_File(config_file))
        {
            // Fall back to /etc/wmtzrc
            sprintf(config_file, "/etc/wmtzrc");
        
            Read_Config_File(config_file);
        }
    }

    RedrawWindow();
    prevtime = time(0) - 1;
    

    while (1)
    {
        curtime = time(0);
        if ( curtime > prevtime)
        {
	  prevtime = curtime;
          clk = gmtime(&curtime);
          /* Update the display */
          for (j=1; j<6; j++)
          {
            if (strncmp( zone[j].label, "xxx", 3) == 0 )
	    {
              BlitString("        ", 5, (11*(j-1)) + 5); 
	    }
           /* Display Internet time */
            else if (strncmp( zone[j].label,"BEATS", 5) == 0 )
            {
              /* Calculate Internet time */
              swatch_beats = (int)(((curtime+3600)%86400)/86.4);
              /* Print label */
              BlitString ("BEATS:", 5, (11*(j-1)) + 5);
              /* Print time */
              sprintf (blitstr, "%3i", swatch_beats);
              BlitString (blitstr, 39, (11*(j-1)) + 5);
            }
            else if (strncmp( zone[j].label,"Y2K", 3) == 0 )
	    { 
              /* Calculate number of days until 2000.0 */
	      jd = curtime/86400.0 - 10957.0 - (zone[j].diff * .04167);

              if (jd > 0)
	      {
                /* Print warning */
                BlitString ("TOO LATE", 5, (11*(j-1)) + 5);
	      }
              else
	      {
                /* Print label */
                BlitString ("Y2K:", 5, (11*(j-1)) + 5);

	        /* Display integer part */
                sprintf(blitstr, "%d", (int)jd );              
                BlitString(blitstr, 23, (11*(j-1)) + 5);  
                len = strlen(blitstr);

	        /* point */
                copyXPMArea(77, 64, 2, 8, 48, (11*(j-1)) + 5);
	  
	        /* Display the rest... */     
                sprintf(blitstr, "%f", jd ); 
                for (k=0; k<(strlen(blitstr)); k++)
                {
                  if (blitstr[k] == '.')
                    pos = k;
                }

                fracstr = blitstr + pos + 1;
                fracstr[1] = 0;
                BlitString(fracstr, 51, (11*(j-1)) + 5); 
	      }
	    }
            else if (strncmp( zone[j].label, "JDN", 3) == 0 )
	    {
	      /* Calculate Julin Day Number */
	      jd = curtime/86400.0 + 2440587.5 - zone[j].diff;

	      /* Display integer part */
              sprintf(blitstr, "%d", (int)jd );              
              BlitString(blitstr, 5, (11*(j-1)) + 5);  
              len = strlen(blitstr);

	      /* "floating" point */
              copyXPMArea(77, 64, 2, 8, (len*6 + 6), (11*(j-1)) + 5);
	  
	      /* Display the rest... */     
              sprintf(blitstr, "%f", jd ); 

              for (k=0; k<(strlen(blitstr)); k++)
              {
                if (blitstr[k] == '.')
                  pos = k;
              }

              fracstr = blitstr + pos + 1;
              fracstr[8-len] = 0;
              BlitString(fracstr, (len*6 + 9), (11*(j-1)) + 5);        
	    }
            else if (strncmp( zone[j].label, "DATE", 4) == 0 )
	    {
	      clk = localtime(&curtime);
	      snprintf(blitstr,4,"%s",month[clk->tm_mon]);
              BlitString(blitstr, 5, (11*(j-1)) + 5);  

	      snprintf(blitstr,3,"%02d",clk->tm_mday);
              BlitString(blitstr, 5+5+(6*3), (11*(j-1)) + 5);  

	      while(clk->tm_year>99) clk->tm_year-=100;
	      snprintf(blitstr,3,"%02d",clk->tm_year);
              BlitString(blitstr, 5+5+5+(6*5), (11*(j-1)) + 5);  

	      copyXPMArea(77, 64, 2, 8, (3*6 + 6), (11*(j-1)) + 5);
	      copyXPMArea(77, 64, 2, 8, (6*6 + 4), (11*(j-1)) + 5);
	      clk = gmtime(&curtime);
	    }
            else if (strncmp( zone[j].label, "WDAY", 4) == 0 )
	    {
	      clk = localtime(&curtime);
	      snprintf(blitstr,10,"%s",week_day[clk->tm_wday]);
              BlitString(blitstr, 4, (11*(j-1)) + 5);  
	      clk = gmtime(&curtime);
	    }
            else 
            {
              strncpy(buf, zone[j].label, 3);
                       
              for (k=0; k<3; k++)
              {
                if (buf[k] == 0)
                  buf[k] = ' ';
              }
              buf[3] = ':';
              buf[4] = 0;

              hour = clk->tm_hour - zone[j].diff;
              if (hour > 23 )
                hour -= 24;
              if (hour < 0 )
                hour += 24;

	      // Print Label
              sprintf(blitstr, "%s",buf );
              BlitString(blitstr, 5, (11*(j-1)) + 5);

	      // Print timestring
              sprintf(blitstr, "%02i", hour );
              BlitString(blitstr, 29, (11*(j-1)) + 5);

	      /* "floating" point */
              copyXPMArea(77, 64, 2, 8, 42, (11*(j-1)) + 5);

              sprintf(blitstr, "%02i", clk->tm_min );
              BlitString(blitstr, 45, (11*(j-1)) + 5);
	    }
          }                                     
        }
        RedrawWindow();
        usleep(100000);
    }        
}


// ReadConfigSetting
int ReadConfigString(FILE *fp, char *setting, char *value)
{
    char str[1024];
    char buf[1024];
    int i;
    int len;
    int slen;
    char *p=NULL;


    if (!fp)
    {
        return 0;
    }

    sprintf(str, "%s=", setting);
    slen = strlen(str);
    
    fseek(fp, 0, SEEK_SET);

    while ( !feof(fp) )
    {
        
        if (!fgets(buf, 512, fp))
            break;
        
        len = strlen(buf);

        // strip linefeed
        for (i=0; i!=len; i++)
        {
            if (buf[i] == '\n')
            {
                buf[i] = 0;
            }
        }

        //printf("Scanning '%s'...\n", buf);
        if ( strncmp(buf, str, strlen(str)) == 0)
        {
            // found our setting
            
            for(i=0; i!=slen; i++)
            {
                if ( buf[i] == '=' )
                {
                    p=buf+i+1;
                    strcpy(value, p);
                    return 1;
                }
            }
    
        }
    }    
        return 0;
}

int ReadConfigInt(FILE *fp, char *setting, int *value)
{
    char buf[1024];

    if (ReadConfigString(fp, setting, (char *) &buf))
    {
        *value = atoi(buf);
        return 1;
    }

    return 0;
}

int Read_Config_File( char *filename )
{
    FILE *fp;

    fp = fopen(filename, "r");
    if (fp)
    {
        ReadConfigString(fp, "time1", zone[1].label);
        ReadConfigString(fp, "time2", zone[2].label);
        ReadConfigString(fp, "time3", zone[3].label);
        ReadConfigString(fp, "time4", zone[4].label);
        ReadConfigString(fp, "time5", zone[5].label);
        ReadConfigInt(fp, "utdiff1", &zone[1].diff);
        ReadConfigInt(fp, "utdiff2", &zone[2].diff);
        ReadConfigInt(fp, "utdiff3", &zone[3].diff);
        ReadConfigInt(fp, "utdiff4", &zone[4].diff);
        ReadConfigInt(fp, "utdiff5", &zone[5].diff);

        fclose(fp);
        return 1;
    }
    else
    {
        perror("Read_Config_File");
        fprintf(stderr, "Unable to open %s, no settings read.\n", filename);
        return 0;
    }

}

// Blits a string at given co-ordinates
void BlitString(char *name, int x, int y)
{
    int		i;
    int		c;
    int		k;

    k = x;
    for (i=0; name[i]; i++)
    {

        c = toupper(name[i]); 
        if (c >= 'A' && c <= 'Z')
        {   // its a letter
			c -= 'A';
			copyXPMArea(c * 6, 74, 6, 8, k, y);
			k += 6;
        }
        else
        {   // its a number or symbol
			c -= '0';
			copyXPMArea(c * 6, 64, 6, 8, k, y);
			k += 6;
		}
	}
}



/*******************************************************************************\
|* usage								       *|
\*******************************************************************************/
void usage(void)
{
    fprintf(stderr, "\nwmtz - shows local time around the world and JD.\n");
    fprintf(stderr, "See ~/.wmtzrc or /etc/wmtzrc for configuration.\n\n");
    fprintf(stderr, "usage:\n");
    fprintf(stderr, "    -display <display name>\n");
    fprintf(stderr, "    -geometry +XPOS+YPOS      initial window position\n");
    fprintf(stderr, "    -c <filename>             use specified config file\n");
    fprintf(stderr, "    -h                        this help screen\n");
    fprintf(stderr, "    -v                        print the version number\n");
    fprintf(stderr, "\n");
}



/*******************************************************************************\
|* printversion								       *|
\*******************************************************************************/
void printversion(void)
{
	fprintf(stderr, "wmtz v%s\n", WMTZ_VERSION);
}

