/* Copyright (c) 1998 Soren (soren@linuxwarez.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 ****************************************************************
 */

#if !defined Linux
#	if (defined NetBSD || defined OpenBSD)
#		warning Compiling for NetBSD/OpenBSD using /dev/io style IO.
#		include <i386/pio.h>
#	else
#		if defined FreeBSD
#			warning Compiling for FreeBSD using /dev/io style IO.
#			include <machine/cpufunc.h>
#		endif
#	endif
#else
#	warning Compiling for Linux using ioperm/iopl style IO.
#	include <asm/io.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include <X11/xpm.h>

#include <time.h>

#include "wmgeneral/wmgeneral.h"
#include "wmtune-master.xpm"
#include "wmtune-mask.xbm"

// Defines
#define VERSION "1.0"
#define RCFILE "wmtunerc"
#define LIMBO 3
#define ON 1
#define OFF 0
#define TRUE 1
#define FALSE 0
#define FCODE(f)    ((int)(((float)(f)-88.0)*40)+0xf6c)

#if (defined FreeBSD || defined NetBSD || defined OpenBSD)
	#define OUTW(word,port)  outw(port,word)
	#define OUTB(byte,port)  outb(port,byte)
#else
	#define OUTW(word,port)  outw(word,port)
	#define OUTB(byte,port)  outb(byte,port)
#endif

// Data Types
#if (defined FreeBSD || defined NetBSD || defined OpenBSD)
	int fpiopl;
#endif

	unsigned int max_presets;
	unsigned int hour=0,minute=0;

	struct tm *time_struct;
	long current_time;

	unsigned int rport=0x000;
	double freqf=0.0;

	char *myname;
	char temp[128];

	int i;

	XEvent Event;

	int but_stat=-1;
	int but_timer=0;
	
	int auto_radio_on = OFF;
	int auto_alarm_start = OFF;
	int cmdline_preset = OFF;
	int preset_count=1;
	int radio_status=OFF;
	int alarm_state=0;

struct { double freq; } *presets;

// Functions
void ParseCMDLine(int argc, char *argv[]);
int ParseRCFile(char *);
void CheckIOPerms(void);
void RadioOff(void);
void RadioOn(void);
void DrawDigitalFreq(void);
void DrawDigitalPreset(void);
void DrawDigitalTime(int, int);
void ButtonUp(int);
void ButtonDown(int);
void TimeUp(void);
void TimeDown(void);
void VolumeUp(void);
void VolumeDown(void);
void TuneUp(void);
void TuneDown(void);
void RadioOut(int, int);
void PresetHandler(int);
void ScanUp(void);
void ScanDown(void);
void OnPreset(void);
void GeneralFreqUpdate(void);
void FastFreqUpdate(void);
void TuneRadio(void);
int TestTune(void);
void TestFreq(void);

// Main
void main(int argc,char *argv[])
{
	myname = argv[0];
	ParseCMDLine(argc,argv);

	strcpy(temp,(char*)getenv("HOME"));
	strcat(temp,"/.");
	strcat(temp,RCFILE);
	if (ParseRCFile(temp) == 1)
	{
		strcpy(temp, "/etc/");
		strcat(temp, RCFILE);
		if (ParseRCFile(temp) == 1)
		{
			fprintf(stderr,"User(~/.%s) or global(/etc/%s) configuration files not found.\n",RCFILE,RCFILE);
			exit(1);
		}
	}

	CheckIOPerms();

	if (cmdline_preset == ON)
	{
		if (!(preset_count <= max_presets && preset_count >= 1))
		{
			fprintf(stderr,"Selected preset number is not valid in your %s file.\n",RCFILE);
			exit(1);
		}
	}

	freqf = presets[preset_count].freq;

	openXwindow(argc,argv,wmtune_master_xpm,wmtune_mask_bits,
				wmtune_mask_width,wmtune_mask_height);

	AddMouseRegion(0,47,48,59,59);		// ON/OFF BUTTON
	AddMouseRegion(1,5,35,16,44);		// FINE TUNE UP (+)
	AddMouseRegion(2,16,35,27,44);		// FINE TUNE DOWN (-)
	AddMouseRegion(3,27,35,38,44);		// SCAN UP
	AddMouseRegion(4,38,35,49,44);		// SCAN DOWN
	AddMouseRegion(5,49,35,59,44);		// ALARM BUTTON
	AddMouseRegion(6,23,48,35,59);		// LEFT PRESET SCAN BUTTON
	AddMouseRegion(7,35,48,47,59);		// RIGHT PRESET SCAN BUTTON

	if (auto_alarm_start == ON)
	{
		alarm_state = ON;
		DrawDigitalTime(hour,minute);
		copyXPMArea(117, 70, 5, 6, 6, 23); // Light On
		usleep(300000L);
		RedrawWindow();
	}

	if (auto_radio_on == ON)
	{
		RadioOn();
		RedrawWindow();
	}

while (1)
{
	while (XPending(display))
	{
		XNextEvent(display, &Event);
		switch (Event.type) 
		{
			case Expose:
				RedrawWindow();
			break;
			case DestroyNotify:
				XCloseDisplay(display);
				#if (defined FreeBSD || defined NetBSD || defined OpenBSD)
				close(fpiopl);
				#else
				ioperm(rport,2,0);
				#endif
				exit(0);
			break;
			case ButtonPress:
				i = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);
					switch (i)
					{
					case 0:			// ON/OFF BUTTON
						ButtonDown(0);
					break;
					case 1:			// FINE TUNE UP (+)
						ButtonDown(1);
						but_timer = 0;
					break;
					case 2:			// FINE TUNE DOWN (-)
						ButtonDown(2);
						but_timer = 0;
					break;
					case 3:			// SCAN UP
						ButtonDown(3);
						but_timer = 0;
					break;
					case 4:			// SCAN DOWN
						ButtonDown(4);
						but_timer = 0;
					break;
					case 5:			// ALARM BUTTON
						ButtonDown(5);
					break;
					case 6:			// LEFT PRESET SCAN BUTTON
						ButtonDown(6);
					break;
					case 7:			// RIGHT PRESET SCAN BUTTON
						ButtonDown(7);
					break;
				}
				but_stat = i;
			break;
			case ButtonRelease:
				switch (but_stat) 
				{
					case 0:			// ON/OFF BUTTON
						ButtonUp(0);
					break;
					case 1:			// FINE TUNE UP (+)
						ButtonUp(1);
					break;
					case 2:			// FINE TUNE DOWN (-)
						ButtonUp(2);
					break;
					case 3:			// SCAN UP
						ButtonUp(3);
					break;
					case 4:			// SCAN DOWN
						ButtonUp(4);
					break;
					case 5:			// ALARM BUTTON
						ButtonUp(5);
					break;
					case 6:			// LEFT PRESET SCAN BUTTON
						ButtonUp(6);
					break;
					case 7:			// RIGHT PRESET SCAN BUTTON
						ButtonUp(7);
					break;
				}
				if (i == but_stat && but_stat >= 0)
				{
			if  (radio_status == OFF || alarm_state == LIMBO)
			{
				if (alarm_state == LIMBO)
				{
					switch (i)
					{
						case 3:			// TIME UP
							if (!(but_timer >= 5))
							{
								TimeUp();
							}
						break;
						case 4:			// TIME DOWN
							if (!(but_timer >= 5))
							{
								TimeDown();
							}
						break;
						case 5:			// ALARM BUTTON SET
							alarm_state = ON;
							copyXPMArea(117, 70, 5, 6, 6, 23); // Light On
							RedrawWindowXYWH(6, 23, 5, 6);
						break;
					}
				}
				else
				{
					switch (i)
					{
						case 0:
							RadioOn();
						break;
					}
				}
				but_stat = -1;
			}
			else
			{	
					switch (i)
					{
						case 0:			// ON/OFF BUTTON
								RadioOff();
						break;
						case 1:			// FINE TUNE UP (+)
							if (!(but_timer >= 5))
							{
								TuneUp();
							}
							OnPreset();
							FastFreqUpdate();
							TestFreq();
						break;
						case 2:			// FINE TUNE DOWN (-)
							if (!(but_timer >= 5))
							{
								TuneDown();
							}
							OnPreset();
							FastFreqUpdate();
							TestFreq();
						break;
						case 3:			// SCAN UP
							switch (Event.xbutton.button)
							{
								case 1:
									ScanUp();
									OnPreset();
								break;
								case 3:
									VolumeUp();
								break;
							}
						break;
						case 4:			// SCAN DOWN
							switch (Event.xbutton.button)
							{
								case 1:
									ScanDown();
									OnPreset();
								break;
								case 3:
									VolumeDown();
								break;
							}
						break;
						case 5:			// ALARM BUTTON
							if (alarm_state == ON)
							{
								alarm_state = OFF;
								copyXPMArea(76, 55, 34, 7, 6, 22);
								RedrawWindowXYWH(6, 22, 34, 7);
							}
							else
							{
								alarm_state = LIMBO;
								DrawDigitalTime(hour,minute);
							}
						break;
						case 6:			// LEFT PRESET SCAN BUTTON
							PresetHandler(-1);
						break;
						case 7:			// RIGHT PRESET SCAN BUTTON
							PresetHandler(1);
						break;
						}
					}
				but_stat = -1;
			}
		}
	} 
	if (((i == but_stat && but_stat >= 0)) && (radio_status == ON))
	{
		but_timer++;
		if(but_timer >= 5)
		{
			if (alarm_state == LIMBO)
			{
				switch (i)
				{
					case 3:			// TIME UP, AUTO-REPEAT
						TimeUp();
					break;
					case 4:			// TIME DOWN, AUTO-REPEAT
						TimeDown();
					break;
				}
			}
			else
			{
				switch (i)
				{
					case 1:			// FINE TUNE UP (+) AUTO-REPEAT
						TuneUp();
					break;
					case 2:			// FINE TUNE DOWN (-) AUTO-REPEAT
						TuneDown();
					break;
				}
			}
		}	
	}
	usleep(5000);
	if (alarm_state == ON)
	{
		current_time = time(0);
		time_struct = localtime(&current_time);
		if(hour == time_struct->tm_hour);
		{
			if (minute == time_struct->tm_min)
			{
				alarm_state = OFF;
				copyXPMArea(76, 55, 34, 7, 6, 22);
				RedrawWindowXYWH(6, 22, 34, 7);
				if (radio_status == ON)
				{
					RadioOff();
				}
				else
				{
					RadioOn();
				}
			}
		}		
	}
} // while
} // main

void RadioOn(void)
{
	radio_status = ON;
	copyXPMArea(93, 90, 13, 5, 44, 9); 
	RedrawWindowXYWH(44, 9, 13, 5); 	// Mhz/Khz
	copyXPMArea(96, 79, 11, 7, 45, 22);
	RedrawWindowXYWH(45, 22, 11, 7); 	// FM/AM
	GeneralFreqUpdate();
}

void GeneralFreqUpdate(void)
{
	TuneRadio();
	TestFreq();
	DrawDigitalPreset();
	DrawDigitalFreq();
}

void FastFreqUpdate(void)
{
	RadioOut(FCODE(freqf),16);
	RadioOut(0xa0,8);
}

void DrawDigitalTime(int hr, int min) 
{
    char    temp[10];
    char    *p = temp;
    int     i,j,k=13;

    sprintf(temp, "%02d:%02d", hr, min);

    for (i=0; i<2; i++)
	{
        for (j=0; j<2; j++)
		{
            copyXPMArea((*p-'0')*6 + 1, 79, 6, 7, k, 22);
            k += 6;
            p++;
        }
        if (*p == ':')
		{
            copyXPMArea(61, 79, 2, 7, k, 22);
            k += 4;
            p++;
        }
    }
	RedrawWindowXYWH(13, 22, 27, 7);
}

void PresetHandler(int preset_hint)
{
	if (preset_hint < 0)
	{
		if (preset_count == 1)
		{
			preset_count = max_presets;
		}
		else
		{
			preset_count--;
		}
	} 
	else
	{
		if (preset_count == max_presets)
		{
			preset_count = 1;
		}
		else
		{
			preset_count++;
		}
	}
	
	freqf = presets[preset_count].freq;

	GeneralFreqUpdate();
}

void DrawDigitalPreset(void)
{
	char temp[10];
	char *p = temp;
	int j=0,k=6;

	sprintf(temp,"%02d",preset_count);

	if (*p == '0')
	{
		copyXPMArea(66, 79, 5, 7, k, 50);
		k += 6;
		j++;
		p++;
	}
	while (j<2) 
	{
		copyXPMArea((*p-'0')*6 + 1, 79, 5, 7, k, 50);
		k += 6;
		p++;
		j++;
	}
	RedrawWindowXYWH(6, 50, 11, 7);
}
	
void DrawDigitalFreq(void)
{
	char temp[10];
	char *p = temp;
	int i=5,j=0,k=6;

	sprintf(temp,"%0f",freqf);

	while (j<i) 
	{
		if (j == 0 && *p != '1')
		{
			copyXPMArea(76, 40, 6, 9, k, 7);
			k += 7;
			i--;
		}
		copyXPMArea((*p-'0')*7 + 1, 66, 7, 9, k, 7);
		k += 7;
		p++;
		if (*p == '.')
		{
			copyXPMArea(71, 66, 3, 9, k, 7);
			k += 3;
			p++;
		}
		j++;
	}
	RedrawWindowXYWH(6, 7, 37, 9);
}

void ScanUp(void)
{
	copyXPMArea(114, 49, 13, 4, 44, 16);	// POSSIBLE SIGNAL LOSS (0)
	RedrawWindowXYWH(44, 16, 13, 4); 		// Test Freq Field Only
	while (1)
	{
		if ((float)freqf != 108.0)
		{
			freqf += .01;
		}
		else
		{
			freqf = 88.0;
		}
		FastFreqUpdate();
		DrawDigitalFreq();
		if ((i = TestTune()) != 0)
		{
			if (i == 1)
			{
				copyXPMArea(92, 100, 13, 4, 44, 16);	// SIGNAL & STEREO (1)
			}
			else
			{
			copyXPMArea(92, 105, 13, 4, 44, 16);		// POSSIBLY MONO :)	(2)
			}
			RedrawWindowXYWH(44, 16, 13, 4); 			// Test Freq Field Only
			return;
		}
	}
}

void ScanDown(void)
{
	copyXPMArea(114, 49, 13, 4, 44, 16);	// POSSIBLE SIGNAL LOSS (0)
	RedrawWindowXYWH(44, 16, 13, 4); 		// Test Freq Field Only
	while (1)
	{
		if ((float)freqf != 88.0)
		{
			freqf -= .01;
		}
		else
		{
			freqf = 108.0;
		}
		FastFreqUpdate();
		DrawDigitalFreq();
		if ((i = TestTune()) != 0)
		{
			if (i == 1)
			{
				copyXPMArea(92, 100, 13, 4, 44, 16);	// SIGNAL & STEREO (1)
			}
			else
			{
			copyXPMArea(92, 105, 13, 4, 44, 16);		// POSSIBLY MONO :)	(2)
			}
			RedrawWindowXYWH(44, 16, 13, 4); 			// Test Freq Field Only
			return;
		}
	}
}

void CheckIOPerms(void)
{
	#if (defined FreeBSD || defined NetBSD || defined OpenBSD)
	if ((fpiopl = open( "/dev/io", O_RDONLY ) < 0) )
	{
		fprintf(stderr, "Failed to gain IO privledges.  Am I setuid root?  Is /dev/io accessable by me?\n");
		exit(1);
	}
	#else
	if (ioperm(rport,2,1) < 0)
	{
		fprintf(stderr, "Failed to gain IO privledges.  Am I setuid root?\n");
		exit(1);
	}
	#endif
}

void ParseCMDLine(int argc,char *argv[])
{
	char *cmdline;
	int i;

	for (i = 1; i < argc; i++) 
	{
		cmdline = argv[i];
		if (cmdline[0] == '-') 
		{
			switch(cmdline[1]) 
			{
			case 'n':
				auto_radio_on = ON;
			break;
			case 't':
				auto_alarm_start = ON;
				sscanf(cmdline+3,"%d",&hour);
				if ((cmdline = strchr(cmdline+3,':')) != NULL)
				{
					sscanf(cmdline+1,"%d",&minute);
				}
				if (!((hour >= 0 && hour <= 23) 
					&& (minute >= 0 && minute <= 59)))
				{
					fprintf(stderr,"Invalid timer setting, valid is ##:@@ where ## is 0-23 and @@ is 0-59.\n");
					exit(1);
				}
			break;
			case 'p':
				sscanf(cmdline+3,"%d",&preset_count);
				cmdline_preset = ON;
			break;
			case 'd':
			break;
			default:
				printf("\nwmtune v%s, by soren@linuxwarez.com, design & art by warp@xs4all.nl\n",VERSION);
				printf("http://linuxwarez.com/wmtune & http://windowmaker.mezaway.org\n\n");
				printf("usage:\n");
				printf("\t-n\t\tturn radio on initially on startup\n");
				printf("\t-t <##:##>\tset timer on startup, military time\n");
				printf("\t-p <##>\t\tset startup preset # listed in %s file\n",RCFILE);
				printf("\t-d <display>\tselects target display\n");
				printf("\t-h\t\tdisplay this screen\n");
				printf("\n");
				printf("examples:\n");
				printf("\t\tturn on radio on startup using preset 5\n");
				printf("\twmtune -n -p 5\n");
				printf("\t\tturn radio on at 2:35pm using preset 2\n");
				printf("\twmtune -p 2 -t 14:35\n");
				printf("\t\tturn on radio on startup and turn radio off at midnight\n");
				printf("\twmtune -n -t 0:0\n");
				printf("\n");
				exit(0);
			}
		}
	}
}

void RadioOff(void)
{
	OUTB(0,rport);
	OUTB(0xc0,rport);
	radio_status = OFF;	
	copyXPMArea(76, 40, 51, 13, 6, 7); 
	copyXPMArea(115, 55, 11, 7, 45, 22);
	copyXPMArea(83, 55, 11, 7, 6, 50);
	RedrawWindowXYWH(6, 50, 11, 7); // Full Preset Field
	RedrawWindowXYWH(6, 7, 51, 22); // Full Upper Field
}

void TuneRadio(void)
{
	RadioOut(FCODE(freqf),16);
	RadioOut(0xa0,8);
	usleep(1000);
	OUTB(0,rport);
	usleep(50000);
	OUTB(0xc8,rport);
}

void RadioOut(int v,int n)
{
    while (n--) 
	{
		if (v&1) 
		{
			OUTW (5,rport);
			OUTW (5,rport);
			OUTW (7,rport);
			OUTW (7,rport);
		}
		else 
		{
			OUTW (1,rport);
			OUTW (1,rport);
			OUTW (3,rport);
			OUTW (3,rport);
	    }
	v>>=1;
	}
}

void ButtonDown(int button)
{
	switch (button)
	{
		case 0:			// ON/OFF BUTTON
			copyXPMArea(79, 100, 12, 11, 47, 48);
			RedrawWindowXYWH(47, 48, 12, 11);
		break;
		case 1:			// FINE TUNE UP (+)
			copyXPMArea(0, 98, 11, 9, 5, 35);
			RedrawWindowXYWH(5, 35, 11, 9);
		break;
		case 2:			// FINE TUNE DOWN (-)
			copyXPMArea(11, 98, 11, 9, 16, 35);
			RedrawWindowXYWH(16, 35, 11, 9);
		break;
		case 3:			// SCAN UP & TIME UP
			copyXPMArea(22, 98, 11, 9, 27, 35);
			RedrawWindowXYWH(27, 35, 11, 9);
		break;
		case 4:			// SCAN DOWN & TIME DOWN
			copyXPMArea(33, 98, 11, 9, 38, 35);
			RedrawWindowXYWH(38, 35, 11, 9);
		break;
		case 5:			// ALARM BUTTON
			copyXPMArea(44, 98, 10, 9, 49, 35);
			RedrawWindowXYWH(49, 35, 10, 9);
		break;
		case 6:			// LEFT PRESET SCAN BUTTON
			copyXPMArea(55, 100, 12, 11, 23, 48);
			RedrawWindowXYWH(23, 48, 12, 11);
		break;
		case 7:			// RIGHT PRESET SCAN BUTTON
			copyXPMArea(67, 100, 12, 11, 35, 48);
			RedrawWindowXYWH(35, 48, 12, 11);
		break;
	}
}

void ButtonUp(int button)
{
	switch (button)
	{
		case 0:			// ON/OFF BUTTON
			copyXPMArea(79, 88, 12, 11, 47, 48);
			RedrawWindowXYWH(47, 48, 12, 11);
		break;
		case 1:			// FINE TUNE UP (+)
			copyXPMArea(0, 88, 11, 9, 5, 35);
			RedrawWindowXYWH(5, 35, 11, 9);
		break;
		case 2:			// FINE TUNE DOWN (-)
			copyXPMArea(11, 88, 11, 9, 16, 35);
			RedrawWindowXYWH(16, 35, 11, 9);
		break;
		case 3:			// SCAN UP
			copyXPMArea(22, 88, 11, 9, 27, 35);
			RedrawWindowXYWH(27, 35, 11, 9);
		break;
		case 4:			// SCAN DOWN
			copyXPMArea(33, 88, 11, 9, 38, 35);
			RedrawWindowXYWH(38, 35, 11, 9);
		break;
		case 5:			// ALARM BUTTON
			copyXPMArea(44, 88, 10, 9, 49, 35);
			RedrawWindowXYWH(49, 35, 10, 9);
		break;
		case 6:			// LEFT PRESET SCAN BUTTON
			copyXPMArea(55, 88, 12, 11, 23, 48);
			RedrawWindowXYWH(23, 48, 12, 11);
		break;
		case 7:			// RIGHT PRESET SCAN BUTTON
			copyXPMArea(67, 88, 12, 11, 35, 48);
			RedrawWindowXYWH(35, 48, 12, 11);
		break;
	}
}

void TimeUp(void)
{
	if (minute == 59)
	{
		if (hour == 23)
		{
			hour = 0;
		}
		else
		{
			hour++;
		}
		minute = 0;
	}
	else 
	{
		minute++;
	}
	DrawDigitalTime(hour,minute);
}

void TimeDown(void)
{
	if (minute == 0)
	{
		if (hour == 0)
		{
			hour = 23;
		}
		else
		{
			hour--;
		}
		minute = 59;
	}
	else
	{
		minute--;
	}
	DrawDigitalTime(hour,minute);
}

void VolumeUp(void)
{
	OUTB(0x88,rport);
	usleep(200000);
	OUTB(0xc8,rport);
}

void VolumeDown(void)
{
	OUTB(0x48,rport);
	usleep(200000);
	OUTB(0xc8,rport);
}

void TuneUp(void)
{
	if ((float)freqf != 108.0)
	{
		freqf += .01;
	}
	else
	{
		freqf = 88.0;
	}
	DrawDigitalFreq();
}

void TuneDown(void)
{
	if ((float)freqf != 88.0)
	{
		freqf -= .01;
	}
	else
	{
		freqf = 108.0;
	}
	DrawDigitalFreq();
}

void TestFreq(void)
{
	switch (TestTune())
	{
		case 0:
			copyXPMArea(114, 49, 13, 4, 44, 16);	// POSSIBLE SIGNAL LOSS (0)
		break;
		case 1:
			copyXPMArea(92, 100, 13, 4, 44, 16);	// SIGNAL & STEREO 		(1)
		break;
		case 2:
			copyXPMArea(92, 105, 13, 4, 44, 16);	// POSSIBLY MONO :) 	(2)
		break;
	}
	RedrawWindowXYWH(44, 16, 13, 4);
}

void OnPreset(void)
{
	int count=1;

	while (count < max_presets+1)
	{
		if ((float)freqf == (float)presets[count].freq)
		{
			preset_count = count;
			DrawDigitalPreset();
			return;
		}
		count++;
	}
}

int TestTune(void)
{
	int res;

	OUTB(0xf8,rport);
	usleep(150000);
	res = (int)inb(rport);
	if (res == 0x0fd)
	{
		return 1;				// SIGNAL & STEREO 		(1)
	}
	else if (res != 0xff)
	{
		return 2;				// POSSIBLY MONO :) 	(2)
	}
	return 0;					// POSSIBLE SIGNAL LOSS (0)
}

int ParseRCFile(char *filename)
{
	char temp[128];
	char *p;
	FILE *fp;
	int ln=1;
	int count=-1;
	char *tokens = " \t\n";

	if ((fp = fopen(filename,"r")) == NULL)
	{
		return 1;
	}
	while(fgets(temp, 128, fp)) 
	{
		if ((p = strtok(temp,tokens)) != NULL)
		{
			if ((p = strchr(temp,'#')) != NULL) 
			{
				*p = '\0';
			}
			if ((strlen(temp)) != 0)
			{
				if (count == -1)
				{
					sscanf(temp,"%x",&rport);
				}
				if (count == 0)
				{
					sscanf(temp,"%d",&max_presets);
					presets = malloc(sizeof(*presets) * (max_presets+1));
					if (!(max_presets >= 1 && max_presets <= 99))
					{
						fprintf(stderr,"Invalid number of presets (1-99) in %s line %d.\n",filename,ln);
						exit(1);
					}
				}
				else
				{
					if (!(count > max_presets))
					{
						sscanf(temp,"%lf",&presets[count].freq);
						if (!(presets[count].freq >= 88.0
							&& presets[count].freq <= 108.0))
						{
							fprintf(stderr,"Invalid preset in %s line %d.\n",filename,ln);
							exit(1);
						}
					}
				}
				count++;
			}
		}
		ln++;
	}
	fclose(fp);
	return 0;
}
