/*  wmsvencd - a CD player designed for Window Maker
    May 8 1999   version 0.5
    Copyright (C) 1999 Steven Cook <sven@linuxfreak.com>
    http://www.linuxfreak.com/~wmsvencd
    
    This software comes with ABSOLUTELY NO WARRANTY
    This software is free software, and you are welcome to redistribute it
    under certain conditions
    See the README file for a more complete notice.

*/

// Based upon wmcdplay
// wmcdplay - A cd player designed for WindowMaker
// 05/09/98  Release 1.0 Beta1
// Copyright (C) 1998  Sam Hawker <shawkie@geocities.com>

// Contains CDDB code based upon parts of XfreeCD v0.7.8
// Copyright (C) 1998 Brian C. Lane
// nexus@tatoosh.com
// http://www.tatoosh.com/nexus


// Defines, includes and global variables
// --------------------------------------

// User defines - standard
#define NORMSIZE    64
#define ASTEPSIZE   56
#define NAME        "wmsvencd"
#define VERSION     "0.5.0"
#define CLASS       "WMSVENCD"

// User defines - custom
#define CDDB_DIR    "~/.cddb"

#define CDDEV       "/dev/cdrom"
#define BACKCOLOUR  "#202020"
#define LEDCOLOUR   "#20b2ae"
#define UINTERVAL_N 1          // 20ths of a second
#define UINTERVAL_E 20         // 20ths of a second

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <ctype.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>
#include <fcntl.h>

#include "cdctl.h"

// Pixmaps - standard
Pixmap pm_tile;
Pixmap pm_disp;
Pixmap pm_mask;

// Pixmaps - artwork
Pixmap pm_cd;
Pixmap pm_cdmask;
Pixmap pm_numbers;
Pixmap pm_charset;

// Xpm images
#include "tile.xpm"
#include "wmsvencd.xpm"
#include "charset.xpm"
#include "numbers.xpm"

// Variables for command-line arguments - standard
bool wmaker=false;
bool ushape=false;
bool astep=false;
char display[256]="";
char position[256]="";
int winsize;
char cddev[256]=CDDEV;
char local_cddb_path[256]=CDDB_DIR;
char backcolour[256]=BACKCOLOUR;
char ledcolour[256]=LEDCOLOUR;

// X-Windows basics - standard
Atom _XA_GNUSTEP_WM_FUNC;
Atom deleteWin;
Display *d_display;
Window w_icon;
Window w_main;
Window w_root;
Window w_activewin;

// X-Windows basics - custom
GC gc_gc, gc_bitgc;
unsigned long color[4];


// Misc custom global variables 
// ----------------------------


int mode=-1, track=-1, pos=-1;
int ucount=0;
char trackstr[8]="";
char timestr[8]="";

int timeCounter = 0;
char *discTitle;
char *songTitle;
int  discTitleLength;
int  songTitleLength;
char timeDisplay = 0;

bool scrollsong = false, scrolldisc = false;

char *titles[100];

CDCtl *cdctl;


// Procedures and functions
// ------------------------

// Procedures and functions - standard
void initXWin(int argc, char **argv);
void freeXWin();
void createWin(Window *win, int x, int y);
unsigned long mixColor(char *colorname1, int prop1, char *colorname2, int prop2);

// Procedures and functions - custom
void scanArgs(int argc, char **argv);
void checkStatus(bool forced);
void pressEvent(XButtonEvent *xev);
void repaint();
void update();
void drawText(int x, int y, char *text);

// Procedures and functions - artwork specials
void createPixmap(char **data, Pixmap *image, Pixmap *mask);

unsigned long cddb_discid();
bool inPoly(int x, int y, int ulx, int uly, int brx, int bry);
void drawTrackNum(int tracknum);
void drawTime(int min, int sec);
void drawText();
int read_cddb();
int read_cddb_file( char *fname, char *tmpinfo[99] );
int find_discid( char *fname, unsigned long id, char *path);

// Implementation
// --------------

int main(int argc, char **argv) {

	scanArgs(argc, argv);
	initXWin(argc, argv);

	color[0] = mixColor(ledcolour, 0, backcolour, 3);
	color[1] = mixColor(ledcolour, 1, backcolour, 2);
	color[2] = mixColor(ledcolour, 2, backcolour, 1);
	color[3] = mixColor(ledcolour, 3, backcolour, 0);

	XpmAttributes xpmattr;
	XpmColorSymbol xpmcsym[4] = {{"back_colour", NULL, color[0]}, 
	                             {"led_colour",  NULL, color[3]},
				     {"un_colour",   NULL, color[1]},
				     {"semi_colour", NULL, color[2]}};
	xpmattr.numsymbols = 4;
	xpmattr.colorsymbols = xpmcsym;
	xpmattr.exactColors = false;
	xpmattr.closeness = 40000;
	xpmattr.valuemask = XpmColorSymbols | XpmExactColors | XpmCloseness;
	XpmCreatePixmapFromData(d_display, w_root, wmsvencd_xpm, &pm_cd, &pm_cdmask, &xpmattr);
	XpmCreatePixmapFromData(d_display, w_root, numbers_xpm, &pm_numbers, NULL, &xpmattr);
	XpmCreatePixmapFromData(d_display, w_root, charset_xpm, &pm_charset, NULL, &xpmattr);

	for (int i = 0; i < 100; i++)
		titles[i] = NULL;
   
	discTitle = (char *)malloc(1024 * sizeof(char));
	songTitle = (char *)malloc(1024 * sizeof(char));
   
	createPixmap(tile_xpm, &pm_tile, NULL);
	pm_disp = XCreatePixmap(d_display, w_root, 64, 64, DefaultDepth(d_display, DefaultScreen(d_display)));
	pm_mask = XCreatePixmap(d_display, w_root, 64, 64, 1);

	XGCValues gcv;
	unsigned long gcm;
	gcm=GCGraphicsExposures;
	gcv.graphics_exposures=false;
	gc_gc=XCreateGC(d_display, w_root, gcm, &gcv);
	gc_bitgc=XCreateGC(d_display, pm_mask, gcm, &gcv);

	cdctl=new CDCtl(cddev);

	if(!cdctl->openOK())
		fprintf(stderr, "%s : Unable to open cdrom device '%s'.\n", NAME, cddev);
	else {
		// This can be  tsNone, tsNext, tsRepeat, tsRepeatCD, tsRandom 
		cdctl->setTrackSelection(tsNext);

		checkStatus(true);

		XEvent xev;
		XSelectInput(d_display, w_activewin, ButtonPress | ExposureMask);
		XMapWindow(d_display, w_main);

		bool done=false;
		while(!done) {
			while(XPending(d_display)) {
				XNextEvent(d_display, &xev);
				switch(xev.type) {
				case Expose:
					repaint();
					break;
				case ButtonPress:
					pressEvent(&xev.xbutton);
					break;
				case ClientMessage:
					if(xev.xclient.data.l[0]==deleteWin)
						done=true;
					break;
				}
			}
			ucount++;
			if(ucount>=((mode==ssNoCD || mode==ssTrayOpen) ? UINTERVAL_E : UINTERVAL_N))
				checkStatus(false);
			XFlush(d_display);
			usleep(300000);
		}
	}
	XFreeGC(d_display, gc_gc);
	XFreeGC(d_display, gc_bitgc);
	XFreePixmap(d_display, pm_tile);
	XFreePixmap(d_display, pm_disp);
	XFreePixmap(d_display, pm_mask);
	XFreePixmap(d_display, pm_cd);
	XFreePixmap(d_display, pm_cdmask);
	XFreePixmap(d_display, pm_numbers);
	XFreePixmap(d_display, pm_charset);
	freeXWin();
	free(discTitle);
	free(songTitle);
//	free(local_cddb_path);

	delete cdctl;
	return 0;
}

void initXWin(int argc, char **argv){
	winsize=astep ? ASTEPSIZE : NORMSIZE;

	if((d_display=XOpenDisplay(display))==NULL){
		fprintf(stderr,"%s : Unable to open X display '%s'.\n", NAME, XDisplayName(display));
		exit(1);
	}
	_XA_GNUSTEP_WM_FUNC=XInternAtom(d_display, "_GNUSTEP_WM_FUNCTION", false);
	deleteWin=XInternAtom(d_display, "WM_DELETE_WINDOW", false);

	w_root=DefaultRootWindow(d_display);

	XWMHints wmhints;
	XSizeHints shints;
	shints.x=0;
	shints.y=0;
	shints.flags=0;
	bool pos=(XWMGeometry(d_display, DefaultScreen(d_display), position, NULL, 0, &shints, &shints.x, &shints.y,
		&shints.width, &shints.height, &shints.win_gravity) & (XValue | YValue));
	shints.min_width=winsize;
	shints.min_height=winsize;
	shints.max_width=winsize;
	shints.max_height=winsize;
	shints.base_width=winsize;
	shints.base_height=winsize;
	shints.flags=PMinSize | PMaxSize | PBaseSize;

	createWin(&w_main, shints.x, shints.y);

	if (wmaker || astep || pos)
		shints.flags |= USPosition;
	if (wmaker) {
		wmhints.initial_state=WithdrawnState;
		wmhints.flags=WindowGroupHint | StateHint | IconWindowHint;
		createWin(&w_icon, shints.x, shints.y);
		w_activewin=w_icon;
		wmhints.icon_window=w_icon;
	} else {
		wmhints.initial_state=NormalState;
		wmhints.flags=WindowGroupHint | StateHint;
		w_activewin=w_main;
	}
	wmhints.window_group=w_main;
	XSetWMHints(d_display, w_main, &wmhints);
	XSetWMNormalHints(d_display, w_main, &shints);
	XSetCommand(d_display, w_main, argv, argc);
	XStoreName(d_display, w_main, NAME);
	XSetIconName(d_display, w_main, NAME);
	XSetWMProtocols(d_display, w_activewin, &deleteWin, 1);
}

void freeXWin(){
	XDestroyWindow(d_display, w_main);
	if (wmaker)
		XDestroyWindow(d_display, w_icon);
	XCloseDisplay(d_display);
}

void createWin(Window *win, int x, int y){
	XClassHint classHint;
	*win=XCreateSimpleWindow(d_display, w_root, x, y, winsize, winsize, 0, 0, 0);
	classHint.res_name=NAME;
	classHint.res_class=CLASS;
	XSetClassHint(d_display, *win, &classHint);
}

unsigned long mixColor(char *colorname1, int prop1, char *colorname2, int prop2){
	XColor color, color1, color2;
	XWindowAttributes winattr;
	XGetWindowAttributes(d_display, w_root, &winattr);
	XParseColor(d_display, winattr.colormap, colorname1, &color1);
	XParseColor(d_display, winattr.colormap, colorname2, &color2);
	color.pixel=0;
	color.red=(color1.red*prop1+color2.red*prop2)/(prop1+prop2);
	color.green=(color1.green*prop1+color2.green*prop2)/(prop1+prop2);
	color.blue=(color1.blue*prop1+color2.blue*prop2)/(prop1+prop2);
	color.flags=DoRed | DoGreen | DoBlue;
	XAllocColor(d_display, winattr.colormap, &color);
	return color.pixel;
}

void scanArgs(int argc, char **argv){
	for(int i=1;i<argc;i++) {
		if(strcmp(argv[i], "-h")==0 || strcmp(argv[i], "-help")==0 || strcmp(argv[i], "--help")==0) {
			printf("wmsvencd - A cd player designed for Window Maker\n");
			printf("May 8 1999  Version %s\n",VERSION);
			printf("Copyright (C) 1999  Steven Cook <sven@linuxfreak.com>\n");
			printf("This software comes with ABSOLUTELY NO WARRANTY\n");
			printf("This software is free software, and you are welcome to redistribute it\n");
			printf("under certain conditions\n");
			printf("See the README file for a more complete notice.\n\n");
			printf("usage:\n\n   %s [options]\n\noptions:\n\n",argv[0]);
			printf("   -h | -help | --help    display this help screen\n");
			printf("   -w                     use WithdrawnState    (for WindowMaker)\n");
			printf("   -s                     shaped window\n");
			printf("   -a                     use smaller window    (for AfterStep Wharf)\n");
			printf("   -r                     display track time remaining\n");
			printf("   -R                     display disc time remaining\n");
			printf("   -c cddb_dir            use specified local CDDB directory\n");
			printf("   -l led_colour          use the specified colour for led displays\n");
			printf("   -b back_colour         use the specified colour for the background\n");
			printf("   -d cd_device           use specified device  (rather than /dev/cdrom)\n");
			printf("   -position position     set window position   (see X manual pages)\n");
			printf("   -display display       select target display (see X manual pages)\n\n");
			exit(0);
		}
		if(strcmp(argv[i], "-w")==0)
			wmaker=!wmaker;
		if(strcmp(argv[i], "-s")==0)
			ushape=!ushape;
		if(strcmp(argv[i], "-a")==0)
			astep=!astep;
		if (strcmp(argv[i], "-r") == 0)
			timeDisplay = 1;
		if (strcmp (argv[i],"-R") == 0)
			timeDisplay = 2;
		if (strcmp(argv[i], "-c")==0) {
			if (i<argc-1) {
				i++;
				sprintf(local_cddb_path, "%s", argv[i]);
			}
			continue;
		}
		if (strcmp(argv[i], "-d")==0) {
			if (i<argc-1) {
				i++;
				sprintf(cddev, "%s", argv[i]);
			}
			continue;
		}
		if(strcmp(argv[i], "-l")==0) {
			if(i<argc-1) {
				i++;
				sprintf(ledcolour, "%s", argv[i]);
			}
			continue;
		}
		if(strcmp(argv[i], "-b")==0) {
			if(i<argc-1) {
				i++;
				sprintf(backcolour, "%s", argv[i]);
			}
			continue;
		}
		if(strcmp(argv[i], "-position")==0){
			if(i<argc-1){
				i++;
				sprintf(position, "%s", argv[i]);
			}
			continue;
		}
		if(strcmp(argv[i], "-display")==0){
			if(i<argc-1){
				i++;
				sprintf(display, "%s", argv[i]);
			}
			continue;
		}
	}
}


char *get_string_piece(FILE * file, int delim)
{
  /* gets one complete row from 'file' and save it in 'buffer'.
     buffer's memory will be freed and allocated to fit the stringsize
     automatically. */

  char *buffer1 = (char *) malloc(1), *buffer2 = (char *) malloc(1), *tmp = (char *) malloc(1024);
  char **active, **inactive;
  int i = 0;

  strcpy(buffer1, "");
  strcpy(buffer2, "");
  strcpy(tmp, "");
  do {
    /*switch the frames */
    if (inactive == &buffer1) {
      active = &buffer1;
      inactive = &buffer2;
    }
    else {
      active = &buffer2;
      inactive = &buffer1;
    }
    /*get the next part, and handle EOF */
    if (fgets(tmp, 1024, file) == NULL) {	/* ok, so we reached the end of the
						   file w/o finding the delimiter */
      free(*active);
      return NULL;
    }

    free(*active);
    *active = (char *) malloc((++i) * 1024);
    sprintf(*active, "%s%s", *inactive, tmp);

  } while (strchr(*active, delim) == NULL);

  free(*inactive);
  return *active;
}

void strip_trailing_space(char **string)
{
  int i = strlen(*string) - 1;
  char *return_string;

  if (string == NULL || *string == NULL)
    return;
  while (isspace((*string)[i]))
    i--;
  i++;
  return_string = (char *) malloc(i + 1);
  strncpy(return_string, *string, i);
  return_string[i] = 0;
  free(*string);
  *string = return_string;
}

void strip_leading_space(char **string)
{
  char *tmp = *string, *return_string;

  if (string == NULL || *string == NULL)
    return;

  while (isspace(*tmp))
    tmp++;

  return_string = strdup(tmp);
  free(*string);
  *string = return_string;
}

void checkStatus(bool forced){
	ucount=0;
	int oldmode=mode;
	int oldpos=pos;
	int oldtrack=track;

	cdctl->doStatus();
	mode=cdctl->getStatusState();
	track=cdctl->getStatusTrack();
	
	if(mode==ssStopped){
		if (timeDisplay == 0)
			pos = 0;
		else 
			pos=cdctl->getTrackStart(track);
	}

	if(mode==ssPlaying || mode==ssPaused) {
		if ((timeDisplay == 0) || (timeDisplay == 1))
			pos=cdctl->getStatusPosRel();
		else
			pos=cdctl->getStatusPosAbs();
	}

	bool umode=mode!=oldmode || forced;
	bool utrack=umode || (!(mode==ssNoCD || mode==ssTrayOpen) && track!=oldtrack);
	bool utimer=utrack || ((mode==ssPlaying || mode==ssPaused || mode==ssStopped) && (int)(pos/75)!=(int)(oldpos/75));


	if(utimer){
		if(umode)
			update();
		if(utrack)
			if (mode == ssNoCD || mode == ssTrayOpen || mode == ssData) 
				drawTrackNum(0);
			else
				drawTrackNum(cdctl->getStatusTrack());

		if(mode==ssPlaying || mode==ssPaused || mode==ssStopped) {
			int t;
			
			if (timeDisplay == 0)
				t = pos;
			else if (timeDisplay == 1)
				t = cdctl->getTrackLen(cdctl->getStatusTrack())-pos;
			else if (timeDisplay == 2)
				t=cdctl->getCDLen()-pos;
			
			drawTime(t/4500, (t/75)%60);

			scrolldisc = strlen(titles[99]) > 9;
			scrollsong = strlen(titles[cdctl->getStatusTrack()-1]) > 9;

			if (scrollsong)
				sprintf(songTitle,"%s  **  ",titles[cdctl->getStatusTrack()-1]);
			else
				sprintf(songTitle,"%s         ",titles[cdctl->getStatusTrack()-1]);
			if (scrolldisc)
				sprintf(discTitle,"%s  **  ",titles[99]);
			else
				sprintf(discTitle,"%s         ",titles[99]);
			songTitleLength = strlen(songTitle);
			discTitleLength = strlen(discTitle);
		} else
			drawTime(0,0);
	}
	drawText();
	timeCounter++;
	repaint();
}

void pressEvent(XButtonEvent *xev){
	int x=xev->x-(winsize/2-32);
	int y=xev->y-(winsize/2-32);
	int btn=-1;
	int acmd = 0;
   
	if (inPoly(x,y,16,48,24,57)) {
	   	btn = 0;                        // play/pause
		if (mode == ssPlaying) 
			acmd = acPause;
		else if (mode == ssPaused)
			acmd = acResume;
		else if (mode == ssStopped)
			acmd = acPlay;
	} else if (inPoly(x,y,37,48,46,57)) {
		btn = 5;                         // stop
		acmd = acStop;
	} else if (inPoly(x,y,48,48,57,57)) {
		btn = 6;                         // eject
		if (mode == ssTrayOpen)
			acmd = acClose;
		else
			acmd = acEject;
	} else if (inPoly(x,y,26,48,35,57)) {
		btn = 4;                         // next track
		acmd = acNext;
	} else if (inPoly(x,y,6,48,14,57)) {
		btn = 3;                         // previous track
		acmd = acPrev;
	} else if (inPoly(x,y,24,4,58,15)) {
		timeDisplay = (timeDisplay + 1) % 3;    
		if (mode == ssNoCD || mode == ssTrayOpen || mode == ssData)
			drawTime(0,0);
		else {
			int t;
			if (timeDisplay == 0)
				t = pos;
			else if (timeDisplay == 1)
				t = cdctl->getTrackLen(cdctl->getStatusTrack())-pos;
			else if (timeDisplay == 2)
				t=cdctl->getCDLen()-pos;
		
			drawTime(t/4500, (t/75)%60);
		}
	}
   
	if (btn >= 0) {
		cdctl->doAudioCommand(acmd);
		checkStatus(false);
	}
}

bool inPoly(int x, int y, int ulx, int uly, int brx, int bry) {
	return ((x >= ulx) && (x <= brx) && (y >= uly) && (y <= bry));
}

void repaint(){
	XCopyArea(d_display, pm_disp, w_activewin, gc_gc, 0, 0, 64, 64, winsize/2-32, winsize/2-32);
	XEvent xev;
	while(XCheckTypedEvent(d_display, Expose, &xev));
}

void update(){
//	printf("update\n");
	
	if(mode==ssData) {
		sprintf(discTitle, "Data Disc");
		sprintf(songTitle, "         ");
		timeCounter = 0;
		scrollsong = false;
		scrolldisc = false;
	} else if(mode==ssNoCD) {
		sprintf(discTitle, "No Disc  ");
		sprintf(songTitle, "         ");
		timeCounter = 0;
		scrollsong = false;
		scrolldisc = false;
	} else if(mode==ssTrayOpen) {
		sprintf(discTitle, "Tray Open");
		sprintf(songTitle, "         ");
		timeCounter = 0;
		scrollsong = false;
		scrolldisc = false;
	} else 
		if (read_cddb() < 0) {
			sprintf(songTitle,"Unknown");
			sprintf(discTitle,"Unknown");
			scrollsong = false;
			scrolldisc = false;
		}
		
	discTitleLength = strlen(discTitle);
	songTitleLength = strlen(songTitle);

	if(pm_cdmask!=None){
		XSetForeground(d_display, gc_bitgc, 0);
		XCopyArea(d_display, pm_cdmask, pm_mask, gc_bitgc, 0, 0, 64, 64, 0, 0);

		if(!(wmaker || ushape || astep)){
			XCopyArea(d_display, pm_tile, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);
			XSetClipMask(d_display, gc_gc, pm_mask);
		}
	}
	
	XCopyArea(d_display, pm_cd, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);

	XSetForeground(d_display, gc_bitgc, 1);

	if(wmaker || ushape || astep)
		XShapeCombineMask(d_display, w_activewin, ShapeBounding, winsize/2-32, winsize/2-32, pm_mask, ShapeSet);
	XSetClipOrigin(d_display, gc_gc, 0, 0);
	XSetClipOrigin(d_display, gc_bitgc, 0, 0);
	XSetClipMask(d_display, gc_gc, None);
	XSetClipMask(d_display, gc_bitgc, None);

}

void drawTime(int min, int sec) {
	
	if (min > 9)
		XCopyArea(d_display, pm_numbers, pm_disp, gc_gc, (min/10)*7+1, 0, 7, 9, 26, 6);
	else 
		XCopyArea(d_display, pm_numbers, pm_disp, gc_gc, 71, 0, 7, 9, 26, 6);
	XCopyArea(d_display, pm_numbers, pm_disp, gc_gc, (min%10)*7+1, 0, 7, 9, 33, 6);
	XCopyArea(d_display, pm_numbers, pm_disp, gc_gc, (sec/10)*7+1, 0, 7, 9, 44, 6);
	XCopyArea(d_display, pm_numbers, pm_disp, gc_gc, (sec%10)*7+1, 0, 7, 9, 51, 6);

}

void drawTrackNum(int tracknum) {
	
	if (tracknum > 9)
		XCopyArea(d_display, pm_numbers, pm_disp, gc_gc, (tracknum/10)*7+1, 0, 7, 9, 6, 6);
	else
		XCopyArea(d_display, pm_numbers, pm_disp, gc_gc, 71, 0, 7, 9, 6, 6);
	XCopyArea(d_display, pm_numbers, pm_disp, gc_gc, (tracknum%10)*7+1, 0, 7, 9, 13, 6);
	
}

void drawText() {

	int j = timeCounter % discTitleLength;

	if (scrolldisc) {
		while (strlen(discTitle) < 18)
			sprintf(discTitle,"%s%s",discTitle,discTitle);

		for (int i = 0; i < 9; i++) 
			if (discTitle[(i+j)%discTitleLength] >= 0x20 && discTitle[(i+j)%discTitleLength] <= 0x7E) 
				XCopyArea(d_display, pm_charset, pm_disp, gc_gc, (discTitle[(i+j)%discTitleLength]-32)*6,0,6,7,6*i+4,21);
			else if (discTitle[(i+j)%discTitleLength] >= (-96) && discTitle[(i+j)%discTitleLength] <= 0) 
				XCopyArea(d_display, pm_charset, pm_disp, gc_gc, (discTitle[(i+j)%discTitleLength]+0x5F)*6,7,6,7,6*i+4,21);
	} else
		for (int i = 0; i < 9; i++) 
			if (discTitle[i] >= 0x20 && discTitle[i] <= 0x7E) 
				XCopyArea(d_display, pm_charset, pm_disp, gc_gc, (discTitle[i]-32)*6,0,6,7,6*i+4,21);
			else if (discTitle[i] >= (-96) && discTitle[i] <= 0) 
				XCopyArea(d_display, pm_charset, pm_disp, gc_gc, (discTitle[i]+0x5F)*6,7,6,7,6*i+4,21);

	j = timeCounter % songTitleLength;

	if (scrollsong) {		
		while (strlen(songTitle) < 18)
			sprintf(songTitle,"%s%s",songTitle,songTitle);

		for (int i = 0; i < 9; i++)
			if (songTitle[(i+j)%songTitleLength] >= 0x20 && songTitle[(i+j)%songTitleLength] <= 0x7E)
				XCopyArea(d_display, pm_charset, pm_disp, gc_gc, (songTitle[(i+j)%songTitleLength]-32)*6,0,6,7,6*i+4,35);
			else if (songTitle[(i+j)%songTitleLength] >= (-96) && songTitle[(i+j)%songTitleLength] <= 0) 
				XCopyArea(d_display, pm_charset, pm_disp, gc_gc, (songTitle[(i+j)%songTitleLength]+0x5F)*6,7,6,7,6*i+4,35);
	} else
		for (int i = 0; i < 9; i++)
			if (songTitle[i] >= 0x20 && songTitle[i] <= 0x7E)
				XCopyArea(d_display, pm_charset, pm_disp, gc_gc, (songTitle[i]-32)*6,0,6,7,6*i+4,35);
			else if (songTitle[i] >= (-96) && songTitle[i] <= 0) 
				XCopyArea(d_display, pm_charset, pm_disp, gc_gc, (songTitle[i]+0x5F)*6,7,6,7,6*i+4,35);

}

void createPixmap(char **data, Pixmap *image, Pixmap *mask) {
	XpmAttributes xpmattr;
	XpmColorSymbol xpmcsym[4]={{"back_color",     NULL, color[0]},
				   {"led_color_high", NULL, color[1]},
                                   {"led_color_med",  NULL, color[2]},
                                   {"led_color_low",  NULL, color[3]}};
	xpmattr.numsymbols=4;
	xpmattr.colorsymbols=xpmcsym;
	xpmattr.exactColors=false;
	xpmattr.closeness=40000;
	xpmattr.valuemask=XpmColorSymbols | XpmExactColors | XpmCloseness | XpmSize;
	XpmCreatePixmapFromData(d_display, w_root, data, image, mask, &xpmattr);
}


/* -----------------------------------------------------------------------
   Search all sub-directories below CDDB_PATH for the file to read
   Fills in the category with the name of the trailing directory that
   it is found in.


   Return 0 on file found
   Return -1 on an error
   Return -2 on no file found
   ----------------------------------------------------------------------- */
int find_discid( char *fname,                /* Path to local database */
		 unsigned long id,           /* discid to search for   */
		 char *path)                 /* Return full path       */
{
	DIR   *dp;
	struct dirent *dirp;
	char  idstr[9], tmp_fname[1024], *p;
	int   fp;

	/* Convert a leading ~ into the user's HOME directory */
	if( fname[0] == '~' ) {
		/* Copy the rest of the path/filename to tmp_fname */
		strncpy( tmp_fname, &fname[1], 1023 );

		if ((p = (char *) getenv("HOME")) == NULL ) 
			return(-2);

		strncpy( fname, p, 1023 );

//	printf("find_discid()\n");
		/* Make sure there is a slash inbetween the two */
		if( (fname[strlen(fname)-1] != '/') && (tmp_fname[0] != '/') ) 
			strcat( fname, "/" );

		strncat( fname, tmp_fname, 1023-strlen( p ) );
	}

//	printf("find_discid( %s )\n", fname );

	sprintf( idstr, "%08lx", id );

	if( ( dp = opendir( fname ) ) == NULL )
		return(-1);

	while( ( dirp = readdir( dp ) ) != NULL ) {
		if( (strcmp(dirp->d_name,"." )==0) || (strcmp(dirp->d_name, "..")==0) )
			continue;

		strcpy( path, fname );
		if( path[strlen(path)] != '/' )
			strcat( path, "/" );
		strcat( path, dirp->d_name );
		if( path[strlen(path)] != '/' )
			strcat( path, "/" );
		strcat( path, idstr );

//		printf("Checking %s\n", path );

		/* Does this file exist? */
		if( ( fp = open( path, O_RDONLY ) ) != -1 ) {
			close( fp );
			closedir( dp );
			return(0);
		}
	} 

	closedir( dp );
	return(-2);
}


/* -----------------------------------------------------------------------
   Read the CDDB info from a filename into a cdinfo structure
   Allocate all needed string storage using gtk's g_string functions

   return -2  = failed to find HOME enviornmental variable
   ----------------------------------------------------------------------- */
int read_cddb_file( char *fname, char *tmpinfo[99] ) {
	int   gotid, i, x, f;
	FILE  *fp;
	char  line[255], discid[9], *p;
  
	if( ( fp = fopen( fname, "r" ) ) == NULL ) {
		return(-1);
	}

	/* Read the cddb file, placing data into tmpinfo */
	if( fgets( line, 255, fp ) == NULL ) {
		fclose( fp );
		return(-1);
	}

	/* Check the file to make sure its a cddb file */
	if( strncmp( line, "# xmcd", 6 ) != 0 ) {
		fclose( fp );
		return(-1);
	}

//	printf("%s", line );

	/* Find the track offsets, abort search at the end of comments */
	while( ( line[0] == '#' ) && ( strncmp( line, "# Track frame offsets:",22 ) != 0 )  ) 
		if( fgets( line, 255, fp ) == NULL ) {
			fclose( fp );
			return(-1);
		}

	/* Skip all the reset of the comments up to Revision: */
	while( line[0] == '#' && (strncmp( line, "# Revision:", 11)!=0) )
		if( fgets( line, 255, fp ) == NULL ) {
			fclose( fp );
			return(-1);
		}

	/* If we got the Revision line, get the revision # */
	if( strncmp( line, "# Revision:", 11)==0 ) {
//		sscanf( line, "# Revision: %d", &tmpinfo->revision );

		/* Now read the rest of the comments */
		while( line[0] == '#' ) 
			if( fgets( line, 255, fp ) == NULL ) {
				fclose( fp );
				return(-1);
			}
	}

	/* Read discid lines */
	/*
	   How should this be handled? check for our id, and discard all others
           that may be present?

           DISCID= lines are comma seperated and can be multiple lines
	*/
	gotid = 0;
	sprintf( discid, "%08lx", cdctl->getDiscID() );
	while( strncmp( line, "DISCID=", 7 ) == 0 ) {
		if( strstr( line, discid ) != NULL )
			gotid = 1;
		if( fgets( line, 255, fp ) == NULL ) {
			fclose( fp );
			return(-1);
		}
	}

	/* Process multiple DTITLE lines and concatanate them */
	i = 0;
	f = 0;
	while( strncmp( line, "DTITLE", 6 ) == 0 ) {
		p = strtok( line, "=\n" );
		p = strtok( NULL, "=\n" );

		/* Add the title to the tmpinfo.title string */
		if( f == 0 ) {
			tmpinfo[99] = strdup( p );
			f = 1;
		} else 
			tmpinfo[99] = strcat(tmpinfo[99], p);

		/* Keep reading DTITLE no matter what */
		if( fgets( line, 255, fp ) == NULL ) {
			fclose( fp );
			return(-1);
		}
	}

	/*
           Copy the tmpinfo from the TTITLE strings

           This has to:
             Get the track # from the TTITLEx
             strcat split title lines up to the limit of storage (255)
        */
	x = -1;
	f = 0;
	while( strncmp( line, "TTITLE", 6 ) == 0 ) {
		/* Get the track # */
		p = strtok( &line[6], "=\n" );

		/* Is it a new track? */
		if( atoi(p) != x ) {
			/* Yes, reset the length counter and track name */
			i = 0;
			f = 0;
			tmpinfo[atoi(p)] = NULL;
		}

		/* Get the track number and make sure its not too big. */
		if( ( x = atoi( p ) ) < 99 ) {
			/* Get the track name */
			p = strtok( NULL, "=\n" );

			/* If its blank, then insert default track name */
			if( p == NULL ) 
				tmpinfo[x] = strdup(" ");
			else {
				if (f == 0) {
					tmpinfo[x] = strdup(p);
					f = 1;
				} else
					tmpinfo[x] = strcat(tmpinfo[x], p);
			}
		}

		/* Read the next line */
		if( fgets( line, 255, fp ) == NULL ) {
			fclose( fp );
			return(-1);
		}
	}

	fclose( fp );
	return(0);
}


/* -----------------------------------------------------------------------
   Read a cddb entry for the current CD

   This needs to search through the sub-directories in CDDB_PATH to find
   the discid of the current CD.

   We then read the file, filling in the track names, etc.
   We should also compare the frame count to make sure it is the
   correct CD.

   Returns -1 if there was an error
   Returns -2 if it cannot find the file
   ----------------------------------------------------------------------- */
int read_cddb() {
	int   x;
	char  fname[255];
	char  *tmp[100];

	char  *cddbdir = (char *)malloc(1024 * sizeof(char));
	
	sprintf(cddbdir,local_cddb_path);

	/* Find out where this ID lives and fill in fname with full name */
	if (find_discid(cddbdir, cdctl->getDiscID(), fname) < 0 ) {
		for (x = 0; x < 99; x++) {
			sprintf(cddbdir,"Track %d",x+1);
			titles[x] = strdup(cddbdir);
		}
		titles[99] = strdup("Unknown");
		free(cddbdir);
		return(-2);
	}

	if (read_cddb_file(fname, tmp) == 0 ) 
		for( x = 0; x < 100; x++ )
			titles[x] = tmp[x];

	free(cddbdir);
//	printf("read_cddb() succeeded\n");

	return(0);
}


