// CDCtl provides easy control of cd audio functions.
// Modified March 1999 by Steven Cook <sven@linuxfreak.com>
// Changes:
//    - CDDB disc ID calculation
//    - converted .h file to .cc file, and added a separate .h file
//      for better OO (IMHO)
//
// All changes Copyright (c)1999 Steven Cook

// cdctl.h - CDCtl class provides easy control of cd audio functions
// 05/09/98  Release 1.0 Beta1
// Copyright (C) 1998  Sam Hawker <shawkie@geocities.com>

// This software comes with ABSOLUTELY NO WARRANTY
// This software is free software, and you are welcome to redistribute it
// under certain conditions
// See the README file for a more complete notice.

// Although cdctl.h is an integral part of wmcdplay, it may also be distributed seperately.

// Change this define to alter the size of forward and backward skips (in frames)
// Yes, I know this should really be a method of CDCtl
#define _CDCTL_SKIP_SIZE 1125

// Try defining some of these. They may improve performance or reliability
// (or just plain make it work)
// #define _CDCTL_STOP_BEFORE_PLAY
// #define _CDCTL_START_BEFORE_PLAY
// #define _CDCTL_SOFT_STOP

// Define this if it stops after each track
#define _CDCTL_SENSITIVE_EOT
// If it still stops for a while between tracks, increase this (0-75 is a sensible range)
#define _CDCTL_SENSITIVITY 0

#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <linux/cdrom.h>

#include "cdctl.h"

// Track selection values (what to do when I've played the requested track)
// Note:      Track selection is not perfect - so use tsNone if you want to avoid trouble.
//            Basically, if we receive a CDROM_AUDIO_COMPLETED status, then we have to decide what to do.
//            If we think the last play command was ours (Next/Prev/FFwd/Rewd don't count), then we do something,
//            depending on the current track selection mode.
// Failures:  Sometimes we may think we sent the last play command when we did not (if we didn't see play stop in
//            in between).
//            If another application is polling the status, it may receive the CDROM_AUDIO_COMPLETED we are looking
//            for, and we will not, so will think play was stopped manually.
//            Similarly, we may read the CDROM_AUDIO_COMPLETED status when we don't want it, such that the other
//            application never sees it.
// Verdict:   Linux audio cdrom handling is broken.
// Update:    New define _CDCTL_SENSITIVE_EOT may help in cases where CDROM_AUDIO_COMPLETED is not being returned
//            correctly. It may, however, interfere with other running cd players.

// Update:    I think this works like a dream now. Even with many cd players sharing a cdrom. Let me know if not!!


/*  PUBLIC:  */


CDCtl :: CDCtl(char *dname) {
	device=(char *)malloc(sizeof(char)*(strlen(dname)+1));
	strcpy(device,dname);
	srand(getpid());
	tracksel=tsRandom;
	tskOurPlay=false;

	if(cdfdopen=(cdfd=open(device,O_RDONLY | O_NONBLOCK))!=-1){
		status_state=ssNoCD;
		status_track=0;
		status_pos=0;
		cd_trklist=NULL;
		doStatus();
	}
}

CDCtl :: ~CDCtl() {
	if(cdfdopen) {
		close(cdfd);
		if(device!=NULL)
			free(device);
		if(cd_trklist!=NULL)
			free(cd_trklist);
	}
}

bool CDCtl :: openOK() {
	return cdfdopen;
}

void CDCtl :: doAudioCommand(int cmd) {
	if(cdfdopen) {
		int newtrk=status_track;
		switch(cmd) {
		case acStop:
			#ifdef _CDCTL_SOFT_STOP
			ioctl(cdfd,CDROMSTART);
			#endif
			#ifndef _CDCTL_SOFT_STOP
			ioctl(cdfd,CDROMSTOP);
			#endif
			tskOurPlay=false;
			break;
		case acPlay:
			status_state=ssPlaying;
			select(status_track);
			tskOurPlay=true;
			break;
		case acPause:
			ioctl(cdfd,CDROMPAUSE);
			break;
		case acResume:
			ioctl(cdfd,CDROMRESUME);
			break;
		case acPrev:
			newtrk--;
			if(newtrk<0)
				newtrk=cd_tracks-1;
			select(newtrk);
			break;
		case acNext:
			newtrk++;
			if(newtrk>cd_tracks-1)
				newtrk=0;
			select(newtrk);
			break;
		case acRewd:
			if(status_pos>cd_trklist[status_track].track_start+_CDCTL_SKIP_SIZE){
				status_pos-=_CDCTL_SKIP_SIZE;
				play();
			}
			break;
		case acFFwd:
			if(status_pos<cd_trklist[status_track].track_start+cd_trklist[status_track].track_len-_CDCTL_SKIP_SIZE){
				status_pos+=_CDCTL_SKIP_SIZE;
				play();
			}
			break;
		case acEject:
			if(ioctl(cdfd,CDROMEJECT))
				status_state=ssNoCD;
			else
				status_state=ssTrayOpen;
			break;
		case acClose:
			ioctl(cdfd,CDROMCLOSETRAY);
			status_state=ssNoCD;
			break;
		}
		doStatus();
	}
}

void CDCtl :: doStatus() {
	if(cdfdopen) {
		struct cdrom_subchnl sc;
		sc.cdsc_format=CDROM_MSF;
		if(ioctl(cdfd, CDROMSUBCHNL, &sc)) {
			if(status_state!=ssNoCD)
				status_state=ssTrayOpen;
			status_track=0;
			status_pos=0;
			tskOurPlay=false;
		} else {
			if(status_state==ssNoCD || status_state==ssTrayOpen)
				readTOC();
			int start,now,stop;
			switch(sc.cdsc_audiostatus){
			case CDROM_AUDIO_PLAY:
				if(status_state==ssStopped)
				tskOurPlay=false;
				status_state=ssPlaying;
				break;
			case CDROM_AUDIO_PAUSED:
				if(status_state==ssStopped)
					tskOurPlay=false;
				status_state=ssPaused;
				break;
			case CDROM_AUDIO_COMPLETED:
				if(tskOurPlay) {
					status_state=ssPlaying;
					selecttrack();
					doStatus();
					return;
				} else
					status_state=ssStopped;
				break;
			default:
				#ifdef _CDCTL_SENSITIVE_EOT
				if(tskOurPlay){
					start = cd_trklist[status_track].track_start;
					stop = start + cd_trklist[status_track].track_len - _CDCTL_SENSITIVITY;
					now = ((sc.cdsc_absaddr.msf.minute) * 60 + sc.cdsc_absaddr.msf.second) * 75 + sc.cdsc_absaddr.msf.frame - CD_MSF_OFFSET;
					if(now>0 && (now<start || now>=stop)){
						status_state=ssPlaying;
						selecttrack();
						doStatus();
						return;
					} else
						status_state=ssStopped;
				} else
					#endif
					status_state=ssStopped;
			}
			trackinfo(&sc);
			if(cd_trklist[status_track].track_data)
				status_state=ssData;
		}
	}
}

void CDCtl :: setTrackSelection(int ts) {
	tracksel=ts;
}

int CDCtl :: getTrackSelection() {
	return tracksel;
}

char * CDCtl :: getDevName() {
	return device;
}

int CDCtl :: getCDTracks() {
	return cd_tracks;
}

int CDCtl :: getCDLen() {
	return cd_len;
}

int CDCtl :: getTrackStart(int trk) {
	return cd_trklist[trk-1].track_start;
}

int CDCtl :: getTrackLen(int trk) {
	return cd_trklist[trk-1].track_len;
}

bool CDCtl :: getTrackData(int trk) {
	return cd_trklist[trk-1].track_data;
}

int CDCtl :: getStatusState() {
	return status_state;
}

int CDCtl :: getStatusTrack() {
	return status_track+1;
}

int CDCtl :: getStatusPosAbs() {
	return status_pos-cd_trklist[0].track_start;
}

int CDCtl :: getStatusPosRel() {
	return status_pos-cd_trklist[status_track].track_start;
}

int CDCtl :: getLeadout() {
	return leadout;
}

unsigned long CDCtl :: getDiscID() {
	return cddb_discid;
}


/*  PRIVATE:  */


int CDCtl :: cddb_sum(int n) {
	char	buf[12], *p;
	int	ret = 0;

	sprintf(buf, "%lu", n);
	for (p = buf; *p != '\0'; p++)
		ret += (*p - '0');

	return (ret);
}

void CDCtl :: readTOC() {
	struct cdrom_tochdr hdr;
	struct cdrom_tocentry te;
	if(cd_trklist!=NULL)
		free(cd_trklist);
	ioctl(cdfd, CDROMREADTOCHDR, &hdr);
	cd_tracks=hdr.cdth_trk1;
	cd_trklist=(struct CDTrack *)malloc(cd_tracks*sizeof(struct CDTrack));
	int prev_addr=0;
	int t = 0, n = 0, k;                             /* ADDED */
	for(int i=0;i<=cd_tracks;i++) {
		if(i==cd_tracks)
			te.cdte_track=CDROM_LEADOUT;
		else
			te.cdte_track=i+1;
		te.cdte_format=CDROM_MSF;    // I think it is ok to read this as LBA, but for a quiet life...
		ioctl(cdfd, CDROMREADTOCENTRY, &te);
		if (i != cd_tracks) {
			n += cddb_sum((te.cdte_addr.msf.minute * 60) + te.cdte_addr.msf.second); /* ADDED */
			if (i == 0)              /* ADDED */
				k = te.cdte_addr.msf.minute * 60 + te.cdte_addr.msf.second;   /* ADDED */
		} else
			t = ((te.cdte_addr.msf.minute * 60) + te.cdte_addr.msf.second) - k;
		int this_addr=((te.cdte_addr.msf.minute * 60) + te.cdte_addr.msf.second) * 75 + te.cdte_addr.msf.frame - CD_MSF_OFFSET;
		if(i>0)
			cd_trklist[i-1].track_len = this_addr - prev_addr - 1;
		prev_addr=this_addr;
		if(i<cd_tracks) {
			cd_trklist[i].track_data = te.cdte_ctrl & CDROM_DATA_TRACK ? true : false;
			cd_trklist[i].track_start = this_addr;
		} else
			cd_len = this_addr;
	}
	cddb_discid = ((n % 0xff) << 24 | t << 8 | cd_tracks);    /* ADDED */
	leadout = ((te.cdte_addr.msf.minute * 60) + te.cdte_addr.msf.second) * 75 + te.cdte_addr.msf.frame - CD_MSF_OFFSET;
}

void CDCtl :: trackinfo(struct cdrom_subchnl *subchnl) {
	if(status_state==ssPlaying || status_state==ssPaused) {
		status_pos=((subchnl->cdsc_absaddr.msf.minute) * 60 + subchnl->cdsc_absaddr.msf.second) * 75 + subchnl->cdsc_absaddr.msf.frame - CD_MSF_OFFSET;
		for(status_track=0;status_track<cd_tracks;status_track++)
			if(status_pos<cd_trklist[status_track].track_start+cd_trklist[status_track].track_len)
				break;
	}
}

void CDCtl :: play() {
	struct cdrom_msf pmsf;
	int abs0=status_pos + CD_MSF_OFFSET;
	int abs1=cd_trklist[status_track].track_start + cd_trklist[status_track].track_len - 1 + CD_MSF_OFFSET;
	pmsf.cdmsf_min0=abs0/(75*60);
	pmsf.cdmsf_min1=abs1/(75*60);
	pmsf.cdmsf_sec0=(abs0%(75*60))/75;
	pmsf.cdmsf_sec1=(abs1%(75*60))/75;
	pmsf.cdmsf_frame0=abs0%75;
	pmsf.cdmsf_frame1=abs1%75;

	#ifdef _CDCTL_STOP_BEFORE_PLAY
	ioctl(cdfd,CDROMSTOP);
	#endif
	#ifdef _CDCTL_START_BEFORE_PLAY
	ioctl(cdfd,CDROMSTART);
	#endif

	ioctl(cdfd,CDROMPLAYMSF,&pmsf);
}

void CDCtl :: select(int trk) {
	status_track=trk;
	status_pos=cd_trklist[status_track].track_start;
	if(status_state==ssPlaying) {
		if(cd_trklist[status_track].track_data) {

			#ifdef _CDCTL_HARD_STOP
			ioctl(cdfd,CDROMSTOP);
			#endif
			#ifndef _CDCTL_HARD_STOP
			ioctl(cdfd,CDROMSTART);
			#endif
			tskOurPlay=false;

		} else
			play();
	}
}

void CDCtl :: selecttrack() {
	int newtrk=status_track;
	do {
		switch(tracksel) {
		case tsNone:
			tskOurPlay=false;
			return;
			break;
		case tsRepeat:
			// do nothing
			break;
		case tsNext:
			newtrk++;
			if (newtrk >= cd_tracks) {
				tskOurPlay=false;
				return;
			}
			break;
		case tsRepeatCD:
			newtrk++;
			if(newtrk>=cd_tracks)
				newtrk=0;
			break;
		case tsRandom:
			newtrk+=(int)((cd_tracks-1)*(float)rand()/RAND_MAX+1);
			if(newtrk>=cd_tracks)
				newtrk-=cd_tracks;
			break;
		}
	} while(cd_trklist[newtrk].track_data);
	select(newtrk);
	play();
}
