/******************************************************************************
*   WMSound     WindowMaker Sound Server                                      *
*   Copyright (C) 1999  Anthony Quinn <southgat@frontiernet.net>              *
*   			Pascal Hofstee <daeron@shadowmere.student.utwente.nl> *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with this program; if not, write to the Free Software               *
*   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
*******************************************************************************/

#include <assert.h>
#include "wmsnd.h"



void
wAbort()
{
    exit(1);
}

Bool wmsndPlaySound (char *sound_file)
{
    if (!wmsndFindSoundServer()) {
	return False;
    }
    else {
	if (!WMSoundDBLoaded) {
	    wmsndLoadConfig();
	}
	wmsndSetStringForKey(sound_file, WMSND_USERDEF);
	PLSave(WMSoundDB, YES);
	WMSoundDBLoaded = False;
	PLRelease(WMSoundDB);
	wmsndSendSound();
    }
    return True;
}


Bool wmsndPlaySoundCore (char *sound_file)
{
    int		retCode = 0;
    char	*devAudio;
    
    devAudio = wmsndGetStringForKey("Device");
    if (!devAudio) {
      return False;
    }
    retCode = sPlayFile(sound_file, devAudio);
    return ((retCode == 0)? True : False);
}


Bool wmsndFindSoundServer (void)
{
    Window	*lstChildren;
    Window	retRoot;
    Window	retParent;
    int		indexCount;
    u_int	numChildren;
    XClassHint	*retHint;
    
    dpy = XOpenDisplay("");
    
    if (!dpy) {
	return False;
    }
    
    _XA_WINDOWMAKER_WM_FUNCTION = XInternAtom(dpy, "_WINDOWMAKER_WM_FUNCTION", False);
    if (XQueryTree(dpy, DefaultRootWindow(dpy), &retRoot, &retParent, &lstChildren, &numChildren)) {
	for (indexCount = 1; indexCount < numChildren; indexCount++) {
	    retHint = XAllocClassHint();
	    if (!retHint) {
		XFree(lstChildren);
		fprintf(stderr, "No Memory\n");
		return False;
	    }
	    
	    XGetClassHint (dpy, lstChildren[indexCount], retHint);
	    if (retHint->res_class) {
		if (strcmp("WMSoundServer", retHint->res_class) == 0) {
		    wmsnd = lstChildren[indexCount];
		    XFree(lstChildren);
		    if(retHint) {
			XFree(retHint);
		    }
		    return True;
		}
	    }
	    XFree(retHint);
	    retHint = 0;
	}
	XFree(lstChildren);
    }
    XCloseDisplay(dpy);
    return False;
}


Bool wmsndSendSound (void)
{
    XEvent	SoundEvent;
    
    SoundEvent.xclient.type		= ClientMessage;
    SoundEvent.xclient.message_type	= _XA_WINDOWMAKER_WM_FUNCTION;
    SoundEvent.xclient.format		= 32;
    SoundEvent.xclient.display		= dpy;
    SoundEvent.xclient.window		= wmsnd;
    SoundEvent.xclient.data.l[0]	= 1011;
    XSendEvent(dpy, wmsnd, False, NoEventMask, &SoundEvent);
    XFlush(dpy);
    XCloseDisplay(dpy);
    return True;
}


Bool wmsndPlayEvent (long wsEventID)
{
  char *sound_file;
  Bool blCode = False;;

  if (!WMSoundDBLoaded) {
    wmsndLoadConfig();
  }
  
  if (wsEventID == 1011) {
    sound_file = strdup(wmsndGetStringForKey(wmsndMapEventID(wsEventID)));
  }
  else {
    sound_file = wmsndGetSoundFile(wmsndMapEventID(wsEventID));
  }
  
  if (sound_file) {
    blCode = wmsndPlaySoundCore(sound_file);
    free(sound_file);
  }
  WMSoundDBLoaded = False;
  PLRelease(WMSoundDB);
  return blCode;
}


char *wmsndGetSoundFile (char *plKey)
{
    proplist_t array, val;
    int nr_elem = 0, i = 0;
    char *strsndfile, *strpath, *strbuffer;

    strsndfile = wmsndGetStringForKey(plKey);
    if (!strsndfile) {
      return NULL;
    }

    array = wmsndGetObjectForKey("SoundPath");
    nr_elem = PLGetNumberOfElements(array);

    while (i < nr_elem) {
	val = PLGetArrayElement(array, i);
	strpath = expandPath(PLGetString(val));
	if (!strpath) {
	  return NULL;
	}
	strbuffer = (char *)malloc(255);
	if (!strbuffer) {
	  if (strpath) 
	    free(strpath);
	  return NULL;
	}
	strcpy(strbuffer,strpath);
	strcat(strbuffer,"/");
	strcat(strbuffer,strsndfile);
	if (checkForFile(strbuffer) == 0) {
	  if (val) {
	    PLRelease(val);
	  }
	  if (array) {
	    PLRelease(array);
	  }
	  if (strpath){ 
	    free(strpath);
	  }
	  return (strbuffer);
	}
	i++;
	free(strpath);
	strpath = NULL;
	PLRelease(val);
	val = NULL;
	free(strbuffer);
	strbuffer = NULL;
    }
    if (strpath){
      free(strpath);
    }
    if (val) {
      PLRelease(val);
    }	  
    if (array) {
      PLRelease(array);
    }
    return NULL;
}

char *wmsndMapEventID (long wsEventID)
{
    switch(wsEventID) {
	case	1000: {
	  return(WMSND_STARTUP);
	}
	case	1001: {
	    return(WMSND_SHADE);
	}
	case	1002: {
	    return(WMSND_UNSHADE);
	}
	case	1003: {
	    return(WMSND_MAXIMIZE);
	}
	case	1004: {
	    return(WMSND_UNMAXIMIZE);
	}
	case	1005: {
	    return(WMSND_ICONIFY);
	}
	case	1006: {
	    return(WMSND_DEICONIFY);
	}
	case	1007: {
	    return(WMSND_HIDE);
	}
	case	1008: {
	    return(WMSND_UNHIDE);
	}
	case	1009: {
	    return(WMSND_APPSTART);
	}
	case	1010: {
	    return(WMSND_APPEXIT);
	}
	case	1011: {
	    return(WMSND_USERDEF);
	}
	default: {
	    return(NULL);
	}
    }
}

void
wmsndLoadConfig () {
  
  WMSoundDB = loadLocalDomain();  
  
  if (!WMSoundDB) {
    makeLocalDomain();
    WMSoundDB = (proplist_t)loadLocalDomain();
    if (!WMSoundDB) {
      fprintf(stderr,"Problem loading WMSound Config file!\n");
      exit(1);
    }
  }
  WMSoundDBLoaded = True;
}

proplist_t
loadLocalDomain () {
  proplist_t db;
  char *path;
  
  path = getLocalDefaultDomain();
  db = PLGetProplistWithPath(path);
  if (db) {                           
    if (!PLIsDictionary(db)) {
      PLRelease(db);
      db = NULL;
      free(path);
      return(db);
    }
  } 
  else {
    return NULL;
  }
  free(path);
  return(db);
}

char *
getLocalDefaultDomain() {
  char *retbuffer, *strbuffer;
  int iSize = 255;
  
  strbuffer = (char *)malloc(iSize);
  
  if (!strbuffer) {
    return NULL;
  }
  
  strcpy(strbuffer, getenv("HOME"));
  strcat(strbuffer, GSTEP_DIR);
  strcat(strbuffer, DEFAULTS_DIR);
  strcat(strbuffer, WMSOUND_DOMAIN);
  
  retbuffer = strdup(strbuffer);
  free(strbuffer);
  return (retbuffer);
}

char *
getGlobalDefaultDomain() {
  FILE *fstream;
  char *retbuffer, *strbuffer;
  int iSize = 255;
 
  strbuffer = (char *)malloc(iSize);
  
  if (!strbuffer) {
    return NULL;
  }

  fstream = popen("wmaker --global_defaults_path", "r");
  if ((!fstream) || (!fgets(strbuffer,iSize,fstream))){ 
    fprintf(stderr, "wmaker --global_defaults_path failed!\n");
    if(strbuffer) {
      free(strbuffer);
    }
    return NULL;
  }
  if (fstream) {
    pclose(fstream);
  }
  strbuffer[strlen(strbuffer) - strlen(WMAKER_DOMAIN)] = '\0';  
  strcat(strbuffer,WMSOUND_DOMAIN);
  retbuffer = strdup(strbuffer);
  free(strbuffer);
  return(retbuffer);
}

Bool
copyDomain(char *insrcpath, char *indestpath, int dwsize) {
  FILE *fstream, *nwdomain;
  char *domainbuffer;
  
  fstream = fopen(insrcpath,"r");
  if (!fstream) {
    return False;
  }
  nwdomain = fopen(indestpath, "w+");
  if (!nwdomain) {
    fclose(fstream);
    return False;
  }

  domainbuffer = (char *)malloc(dwsize);
  if (!domainbuffer) {
    fclose(nwdomain);
    fclose(fstream);
    return False;
   }
   if (!fread(domainbuffer,dwsize,1, fstream)) {
     fclose(nwdomain);
     fclose(fstream);
     if (domainbuffer) {
       free(domainbuffer);
     }
     return False;
   }
   if (!fwrite(domainbuffer, dwsize, 1, nwdomain)) {
     fclose(nwdomain);
     fclose(fstream);
     if (domainbuffer){
       free(domainbuffer);
     }
     return False;
   }
   if (domainbuffer){
     free(domainbuffer);
   }
   fclose(nwdomain);
   fclose(fstream);
   return True;
}

Bool
makeLocalDomain () {
    char *glblpath, *homepath;
    int iSize = 255, flen = 0;
    Bool blret = False;
    
    glblpath = getGlobalDefaultDomain();
    homepath = getLocalDefaultDomain();
    
    if ((checkForFile(homepath) < 0) && (checkForFile(glblpath) == 0)) {
      flen = getDomainSize(glblpath);
      if (flen) {
        blret = copyDomain(glblpath, homepath,flen);
      }
    }
    else {
      blret = True;
    }
    free(glblpath);
    free(homepath);
    return blret;
}

int 
checkForFile(char *instrpath) {
  return(access(instrpath,F_OK));
}

int
getDomainSize(char *instrpath){
  struct stat flstat;
  int chk_exist = 0;
  
  chk_exist = stat(instrpath,&flstat);
  if (chk_exist > 0) {
    return -1;
  }
  return ((int) flstat.st_size); 
}

char *
expandPath(char *strpath) {
  char *retbuffer, *srchbuffer, *storebuff;
  int iSize = 255;
  
  retbuffer = (char *)malloc(iSize);
  if (!retbuffer) {
    return NULL;
  }
  switch(strpath[0]) {
    case '$': {
      srchbuffer = strstr(strpath,"/");
      storebuff = strdup(srchbuffer);
      strpath[strlen(strpath) - strlen(srchbuffer)] = '\0';
      strpath++;
      strcpy(retbuffer,getenv(strpath));
      strcat(retbuffer,storebuff);
      free(storebuff);
      return (retbuffer);
    }
    case '~': {
      strcpy(retbuffer,getenv("HOME"));
      strpath++;
      strcat(retbuffer,strpath);
      return (retbuffer);
    }
    default : {
      strcpy(retbuffer,strpath);
      return (retbuffer);
    }
  }
  return NULL;
}






