/******************************************************************************
*   WMSound     WindowMaker Sound Server                                      *
*   Copyright (C) 1999  Anthony Quinn <southgat@frontiernet.net>              *
*   			Pascal Hofstee <daeron@shadowmere.student.utwente.nl> *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with this program; if not, write to the Free Software               *
*   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/stat.h>
#include <sys/ioctl.h>

#ifdef __NetBSD__
  #include <soundcard.h>
#endif
#ifdef __FreeBSD__
  #include <machine/soundcard.h>
#endif
#ifdef __linux__
  #include <linux/soundcard.h>
#endif

#ifdef _HPUX_SOURCE
  #include <Alib.h>
#endif

#ifdef SVR4
  #include <sys/audioio.h>
  #include <errno.h>
#endif

#include "Sound.h"

#ifdef ESD
#include <esd.h>
#endif


int sPlayFile (char *sound_file, char *audiodev)
{
#ifdef _HPUX_SOURCE
  Audio           *audio;         /* the audio structure             */
   AGainEntry      gainEntry[4];   /* used to set playback parameters */
   AGainMatrix     *gainMatrix;     /* for getting default matrix      */
   ATransID        xid;            /* the transaction ID              */
   AFileFormat     fileFormat;     /* receives the audio file format  */
   AudioAttrMask   AttribsMask;    /* attribute mask                  */
   AudioAttributes Attribs;        /* audio attributes                */
   SBucket         *pSBucket;      /* sound bucket structure          */
   SBPlayParams    playParams;     /* play parameters                 */
   char            *ServerName;    /* to identify audio server        */
   AEvent          event;          /* an audio event */
   long            status;
   int             done = 0;

   if(!sound_file) {
     printf("Could not find sound '%s'",sound_file);
   }

   fileFormat     = AFFUnknown;
   ServerName     = "";                      /* play on same system   */
   AttribsMask = 0;                          /* audio attributes mask */
 
   /*  Open the audio connection
    */
   audio = AOpenAudio( ServerName, NULL );
   
   /*  Load the audio file into a sound bucket.
    */
     pSBucket = ALoadAFile( audio, sound_file, fileFormat, AttribsMask,
                           &Attribs, NULL );
   /*
    * Setup the playback parameters.
    * Use ASimplePlayer() to get the simplest output gain matrix
    * supported by the audio hardware.
    */
   memset(&playParams, 0, sizeof(playParams));
   playParams.gain_matrix = *ASimplePlayer(audio);
   playParams.priority = APriorityNormal;       /* normal priority          */
   playParams.play_volume = AUnityGain;         /* no change to play volume */
   playParams.duration.type = ATTFullLength;    /* play entire sample       */
   playParams.event_mask = ATransCompletedMask;  /* let us know when we're done */
   
   /*  Play the sound bucket
    */
   xid = APlaySBucket( audio, pSBucket, &playParams, 0 );
   
   while (!done) {
   	ANextEvent(audio, &event, &status);
 	switch (event.any_event.type) {
 		case AETTransCompleted:
 			done = 1;
 			break;
 	}
   }
   
   /*  Set close mode to prevent playback from stopping 
    *  when we close the audio connection.
    */
   ASetCloseDownMode( audio, AKeepTransactions, NULL );
   
   /*  That's all, folks!
    */
   ADestroySBucket( audio, pSBucket, NULL );
   ACloseAudio( audio, NULL );
 #else

    int	snd_typ = 0;
    
    if (!sound_file) {
	fprintf(stderr, "Could not find sound '%s'", sound_file);
    }
    
    snd_typ = Find_SndFmt(sound_file);
    switch(snd_typ) {
	case 1: {return(sCoreAu(sound_file, audiodev));}
	case 2: {return(sCoreRiff(sound_file, audiodev));}
	default: {return(ERR_UNSUPPORTED);}
    }
 #endif   
}


int sCoreAu (char *sound_file, char *audiodev)
{
    FILE	*fstream;
    int		audiofd = 0,
		wrbytes,
		index,
		work_int,
		chk_st = 0;
    dword	datalen = 0,
		work_ulong,
		hdrincr = 0;
    unsigned char *inbuff;
    struct stat	flstat;
    au_info	auheader;
#ifdef SVR4
    audio_info_t sun_audio_info;
#endif

#ifdef ESD
    esd_format_t esdformat;
#endif
    
    chk_st = stat(sound_file, &flstat);
    if (chk_st > 0) {
	return ERR_ONSTAT;
    }
    
    fstream = fopen(sound_file, "rb");
    if (!fstream) {
	return ERR_NOFSTREAM;
    }
    
    auheader.magic	= Read_BE_DWORD(fstream);
    auheader.hdr_size	= Read_BE_DWORD(fstream);
    auheader.data_size	= Read_BE_DWORD(fstream);
    auheader.encoding	= Read_BE_DWORD(fstream);
    auheader.sample_rate = Read_BE_DWORD(fstream);
    auheader.channels	= Read_BE_DWORD(fstream);
    if (auheader.hdr_size > 24) {
	hdrincr = Read_BE_DWORD(fstream);
	hdrincr = Read_BE_DWORD(fstream);
    }
    datalen = (dword)flstat.st_size - auheader.hdr_size;

#ifdef SVR4
    audiofd = open(audiodev, O_WRONLY);
    if (audiofd == -1) {
	fclose(fstream);
#ifdef DEBUG
      perror("open audiodev failed in sCoreAu, Errno");
#endif
	return NO_SOUND_DEV;
    }

    if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO FORMAT failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }
    
#ifdef DEBUG
    fprintf(stderr, "Encoding: %d\n", auheader.encoding);
#endif
/*    sun_audio_info.play.encoding = auheader.encoding; */
    sun_audio_info.play.encoding = AUDIO_ENCODING_ULAW;
    sun_audio_info.play.precision = 8;
    
    if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO FORMAT failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }


    if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO SAMPLE RATE failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }
    
    sun_audio_info.play.sample_rate = auheader.sample_rate;
    
    if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO SAMPLE RATE failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }


    if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO CHANNELS failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }
    
#ifdef DEBUG
    fprintf(stderr, "Channels: %d\n", auheader.channels);
#endif

/*    sun_audio_info.play.channels = auheader.channels; */
    sun_audio_info.play.channels = 1;
    
    if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO CHANNELS failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }
    

  if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO BUFFER failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }
  sun_audio_info.play.buffer_size = 1024;

  if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO BUFFER failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }

  if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO OUTPUT failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }

  sun_audio_info.play.port = AUDIO_SPEAKER;

  if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO OUTPUT failed in sCoreAu, Errno");
#endif
	return ERR_DSP_SETFMT;
      }

#endif
    
#ifdef ESD
    if (auheader.channels > 1) {
	esdformat = (ESD_BITS8 | ESD_STEREO);
    }
    else {
	esdformat = (ESD_BITS8 | ESD_MONO);
    }
    work_int = (int)auheader.sample_rate;
    audiofd = esd_play_stream_fallback(esdformat,work_int,NULL,"wmsound");
#endif

#ifdef OSS_SOUND
    audiofd = open(audiodev, O_WRONLY, 0);
    if (audiofd == -1) {
	fclose(fstream);
	return NO_SOUND_DEV;
    }
    
    work_int = auheader.encoding;
    if (ioctl(audiofd, SNDCTL_DSP_SETFMT, &work_int) < 0) {
	fclose(fstream);
	close(audiofd);
	return ERR_DSP_SETFMT;
    }
    
    if (auheader.channels > 1) {
	work_int = 1;
    }
    else {
	work_int = 0;
    }
    
    if (ioctl(audiofd, SNDCTL_DSP_STEREO, &work_int) < 0) {
	fclose(fstream);
	close(audiofd);
	return ERR_DSP_STEREO;
    }
    
    work_ulong = auheader.sample_rate;
    if (ioctl(audiofd, SNDCTL_DSP_SPEED, &work_ulong) < 0) {
	fclose(fstream);
	close(audiofd);
	return ERR_DSP_SPEED;
    }
    
    if (ioctl(audiofd, SNDCTL_DSP_SYNC, NULL) < 0) {
	/* Do nothing for now */
    }
#endif
    
    if (datalen > 4096) {
	wrbytes = 4096;
    }
    else {
	wrbytes = datalen;
    }

    while(datalen > 0) {
	if (wrbytes > datalen) {
	    wrbytes = datalen;
	}
	inbuff = (unsigned char *)malloc(wrbytes);
	if (!inbuff) {
	    fclose(fstream);
	    close(audiofd);
	    return NO_MEMORY;
	}
	
	if ((index = fread(inbuff, wrbytes, 1, fstream)) < 1) {
	    free(inbuff);
	    fclose(fstream);
	    close(audiofd);
	    return ERR_READ_DATA;
	}
	
	if (write(audiofd, inbuff, wrbytes) != wrbytes) {
	    free(inbuff);
	    fclose(fstream);
	    close(audiofd);
	    return ERR_WRITE_DATA;
	}
	
	datalen -= (unsigned long) wrbytes;
	free(inbuff);
	inbuff = 0;
    }
    fclose(fstream);
    close(audiofd);
    return 0;
}


int sCoreRiff (char *sound_file, char *audiodev)
{
    FILE	*fstream;
    int		audiofd = 0,
		wrbytes,
		index,
		work_int;
    dword	datalen = 0,
		work_ulong;
    unsigned char *inbuff;
    riff_info	riffheader;
    wavheader	headerinfo;
#ifdef SVR4
    audio_info_t sun_audio_info;
#endif
#ifdef ESD
    esd_format_t esdformat;
#endif
    
    fstream = fopen(sound_file, "rb");
    if (!fstream) {
	return ERR_NOFSTREAM;
    }
    
    fread(&riffheader.riff, sizeof(riffheader.riff), 1, fstream);
    riffheader.riffsize = Read_LE_DWORD(fstream);
    riffheader.riff[3] = '\0';
    if (goto_chunk(fstream, format_chunk) != 0) {
	fclose(fstream);
	return ERR_FORMAT_TAG;
    }
    
    headerinfo.pcmheader = Read_LE_DWORD(fstream);
    headerinfo.pcmfmt	= Read_LE_WORD(fstream);
    headerinfo.channels	= Read_LE_WORD(fstream);
    headerinfo.sampleps	= Read_LE_DWORD(fstream);
    headerinfo.bytesps	= Read_LE_DWORD(fstream);
    headerinfo.bytespsamp = Read_LE_WORD(fstream);
    headerinfo.bitsperchan = Read_LE_WORD(fstream);
    
    if (headerinfo.bitsperchan != 16) {
	if (headerinfo.bitsperchan != 8) {
	    fclose(fstream);
	    return ERR_UNSUPPORTED;
	}
    }
    
    if (goto_chunk(fstream, data_chunk) != 0) {
	fclose(fstream);
	return ERR_DATA_TAG;
    }
    
    datalen = Read_LE_DWORD(fstream);
#ifdef SVR4    
    audiofd = open("/dev/audio", O_WRONLY);
    if (audiofd == -1) {
	fclose(fstream);
#ifdef DEBUG
      perror("open audiodev failed in sCoreRiff, Errno");
#endif
	return NO_SOUND_DEV;
    }

        /* ripped from mpg123m, audio.c */

        /* audio_set_format */
    if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO FORMAT failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }
    
#ifdef DEBUG
    fprintf(stderr, "Bits per channel: %d\n", headerinfo.bitsperchan);
#endif
    sun_audio_info.play.encoding = AUDIO_ENCODING_LINEAR;
/*    sun_audio_info.play.precision = headerinfo.bitsperchan; */
    sun_audio_info.play.precision = 16;
    
    if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO FORMAT failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }
 
        /* audio_set_channels */
    if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO CHANNELS failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }

#ifdef DEBUG
    fprintf(stderr, "Channels: %d\n", headerinfo.channels);
#endif

/*    sun_audio_info.play.channels = headerinfo.channels; */
    sun_audio_info.play.channels = 1;

    if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO CHANNELS failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }

        /* audio_set_rate */
    if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO RATE failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }

    sun_audio_info.play.sample_rate = headerinfo.sampleps;

    if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO RATE failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }

  if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO BUFFER failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }
  sun_audio_info.play.buffer_size = 1024;

  if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO BUFFER failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }

  if(ioctl(audiofd, AUDIO_GETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl GETINFO OUTPUT failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }

  sun_audio_info.play.port = AUDIO_SPEAKER;

  if(ioctl(audiofd, AUDIO_SETINFO, &sun_audio_info) == -1)
      {
	fclose(fstream);
	close(audiofd);
#ifdef DEBUG
        perror("ioctl SETINFO OUTPUT failed in sCoreRiff, Errno");
#endif
	return ERR_DSP_SETFMT;
      }
  
#endif

#ifdef ESD
    if (headerinfo.bitsperchan == 16) esdformat = ESD_BITS16;
    else esdformat = ESD_BITS8;

    if (headerinfo.channels > 1) esdformat |= ESD_STEREO;
    else esdformat |= ESD_MONO;

    work_int = (int)headerinfo.sampleps;
    audiofd = esd_play_stream_fallback(esdformat,work_int,NULL,"wmsound");
#endif
#ifdef OSS_SOUND
    audiofd = open(audiodev, O_WRONLY, 0);
    if (audiofd == -1) {
	fclose(fstream);
	return NO_SOUND_DEV;
    }
    
    work_int = headerinfo.bitsperchan;
    if (ioctl(audiofd, SNDCTL_DSP_SETFMT, &work_int) < 0) {
	fclose(fstream);
	close(audiofd);
	return ERR_DSP_SETFMT;
    }
    
    if (headerinfo.channels > 1) {
	work_int = 1;
    }
    else {
	work_int = 0;
    }
    if (ioctl(audiofd, SNDCTL_DSP_STEREO, &work_int) < 0) {
	fclose(fstream);
	close(audiofd);
	return ERR_DSP_SETFMT;
    }
    
    work_ulong = headerinfo.sampleps;
    if (ioctl(audiofd, SNDCTL_DSP_SPEED, &work_ulong) < 0) {
	fclose(fstream);
	close(audiofd);
	return ERR_DSP_SPEED;
    }
    
    if (ioctl(audiofd, SNDCTL_DSP_SYNC, NULL) < 0) {
	/* Do nothing for now */
    }
#endif
    
    if (datalen > 4096) {
	wrbytes = 4096;
    }
    else {
	wrbytes = datalen;
    }
    
    while (datalen > 0) {
	if (wrbytes > datalen) {
	    wrbytes = datalen;
	}
	
	inbuff = (unsigned char *)malloc(wrbytes);
	if (!inbuff) {
	    fclose(fstream);
	    close(audiofd);
	    return NO_MEMORY;
	}
	
	if ((index = fread(inbuff, wrbytes, 1, fstream)) < 1) {
	    free(inbuff);
	    fclose(fstream);
	    close(audiofd);
	    return ERR_READ_DATA;
	}
	
	if (write(audiofd, inbuff, wrbytes) != wrbytes) {
	    free(inbuff);
	    fclose(fstream);
	    close(audiofd);
	    return ERR_WRITE_DATA;
	}
	datalen -= (unsigned long) wrbytes;
	free(inbuff);
	inbuff = 0;
    }
    fclose(fstream);
    close(audiofd);
    return 0;
}


int	sCoreDumpWav (char *sound_file)
{
    FILE	*fstream;
    dword	datalen = 0;
    riff_info	riffheader;
    wavheader	headerinfo;
    
    fstream = fopen(sound_file, "rb");
    if (!fstream) {
	return ERR_NOFSTREAM;
    }

    fread(&riffheader.riff, sizeof(riffheader.riff), 1, fstream);
    riffheader.riffsize = Read_LE_DWORD(fstream);
    riffheader.riff[3] = '\0';
    if (goto_chunk(fstream, format_chunk) != 0) {
	fclose(fstream);
	return ERR_FORMAT_TAG;
    }
    
    headerinfo.pcmheader = Read_LE_DWORD(fstream);
    headerinfo.pcmfmt	= Read_LE_WORD(fstream);
    headerinfo.channels	= Read_LE_WORD(fstream);
    headerinfo.sampleps	= Read_LE_DWORD(fstream);
    headerinfo.bytesps	= Read_LE_DWORD(fstream);
    headerinfo.bytespsamp = Read_LE_WORD(fstream);
    headerinfo.bitsperchan = Read_LE_WORD(fstream);
    
    if (goto_chunk(fstream, data_chunk) != 0) {
	fclose(fstream);
	return ERR_DATA_TAG;
    }
    
    datalen = Read_LE_DWORD(fstream);

    fprintf(stderr, "----- Begin Wav file tag dump ------\n");
    fprintf(stderr, "RIFF Header info      : %s\n", riffheader.riff);
    fprintf(stderr, "RIFF Size with header : %d\n", riffheader.riffsize);
    fprintf(stderr, "PCM Header info       : %d\n", headerinfo.pcmheader);
    fprintf(stderr, "PCM Format info       : %d\n", headerinfo.pcmfmt);
    fprintf(stderr, "PCM Channels info     : %d\n", headerinfo.channels);
    fprintf(stderr, "PCM Samples per sec   : %d\n", headerinfo.sampleps);
    fprintf(stderr, "PCM Bytes per sec     : %d\n", headerinfo.bytesps);
    fprintf(stderr, "PCM Bytes per sample  : %d\n", headerinfo.bytespsamp);
    fprintf(stderr, "PCM Bits per channel  : %d\n", headerinfo.bitsperchan);
    fprintf(stderr, "Total data size       : %d\n", datalen);
    fprintf(stderr, "\n");
    fprintf(stderr, "----- End of Wav file tag dump -----\n");
    return 0;
}


int	sCoreDumpAu (char *sound_file)
{
    FILE	*fstream;
    au_info	auheader;
    
    fstream = fopen(sound_file, "rb");
    if (!fstream) {
	return ERR_NOFSTREAM;
    }

    auheader.magic	= Read_BE_DWORD(fstream);
    auheader.hdr_size	= Read_BE_DWORD(fstream);
    auheader.data_size	= Read_BE_DWORD(fstream);
    auheader.encoding	= Read_BE_DWORD(fstream);
    auheader.sample_rate = Read_BE_DWORD(fstream);
    auheader.channels	= Read_BE_DWORD(fstream);

    fprintf(stderr, "----- Begin AU file tag dump ------\n");
    fprintf(stderr, "AU magic              : %d\n", auheader.magic);
    fprintf(stderr, "AU header size        : %d\n", auheader.hdr_size);
    fprintf(stderr, "AU data size          : %d\n", auheader.data_size);
    fprintf(stderr, "AU encoding           : %d\n", auheader.encoding);
    fprintf(stderr, "AU sample rate        : %d\n", auheader.sample_rate);
    fprintf(stderr, "AU channels           : %d\n", auheader.channels);
    fprintf(stderr, "\n");
    fprintf(stderr, "----- End of AU file tag dump -----\n");
    return 0;
}

#ifndef _HPUX_SOURCE
int goto_chunk (FILE *instream, int chunk)
{
    int	indexCount = 0,
	found_tag = 1;
    unsigned char bit_op[5];
    
    switch (chunk) {
	case format_chunk:
	{
	    do
	    {
		if (indexCount > 1000) {
		    /* Get Out if we haven't found the WAVfmt tag after 1000 bytes */
		    break;
		}
		
		fread(&bit_op, sizeof(bit_op)-1, 1, instream);
		bit_op[4] = '\0';
		found_tag = strcmp(bit_op, "fmt ");
		indexCount++;
	    } while(found_tag != 0);
	    if (found_tag != 0) {
		return -1;
	    }
	    return found_tag;
	}
	
	case	data_chunk:
	{
	    do
	    {
		if (indexCount > 1000) {
		    /* Get Out is we haven't found the Data tag after 1000 bytes */
		    break;
		}
		
		fread(&bit_op, sizeof(bit_op)-1, 1, instream);
		bit_op[4] = '\0';
		found_tag = strcmp(bit_op, "data");
		if (found_tag != 0) {
		    bit_op[3] = '\0';
		    found_tag = strcmp(bit_op, "ata");
		    if (found_tag == 0) {
			/* Reset the file in order to get the datalen correct */
			fseek(instream, -1L, SEEK_CUR);
		    }
		    else {
			bit_op[2] = '\0';
			found_tag = strcmp(bit_op, "ta");
			if (found_tag == 0) {
			    /* Reset the file in order to get the datalen correct */
			    fseek(instream, -2L, SEEK_CUR);
			}
		    }
		}
		indexCount++;
	    } while (found_tag != 0);
	    
	    if (found_tag != 0) {
		return -1;
	    }
	    
	    return found_tag;
	}
    }
    return -1;
}
#endif

/**********************
* Begin Utils Section *
***********************/

dword Read_LE_DWORD (FILE *fstream)
{
    dword retVal;
    
    retVal  = fgetc(fstream);
    retVal |= fgetc(fstream)<<8;
    retVal |= fgetc(fstream)<<16;
    retVal |= fgetc(fstream)<<24;
    
    return retVal;
}


word Read_LE_WORD (FILE *fstream)
{
    word retVal;
    
    retVal  = fgetc(fstream);
    retVal |= fgetc(fstream)<<8;
    
    return retVal;
}


dword Read_BE_DWORD (FILE *fstream)
{
    dword retVal;
    
    retVal  = fgetc(fstream)<<24;
    retVal |= fgetc(fstream)<<16;
    retVal |= fgetc(fstream)<<8;
    retVal |= fgetc(fstream);
    
    return retVal;
}


word Read_BE_WORD (FILE *fstream)
{
    word retVal;
    
    retVal  = fgetc(fstream)<<8;
    retVal |= fgetc(fstream);
    
    return retVal;
}


float Read_BE_FLOAT (FILE *fstream)
{
    float retVal;
    
    fread(&retVal, sizeof(retVal),1, fstream);
    
    return retVal;
}


int Find_SndFmt (char *sound_file)
{
    FILE	*fstream;
    dword	chkhdr;
    
    fstream = fopen(sound_file, "rb");
    if (!fstream) {
	return ERR_NOFSTREAM;
    }
    
    chkhdr = Read_LE_DWORD(fstream);
    fclose(fstream);
    if (chkhdr == 1684960046) {
	return 1;
    }
    
    if (chkhdr == 1179011410) {
	return 2;
    }
    
    return -1;
}
