/* -*- linux-c -*- */

/*
    via.c - Part of a Linux module for reading sensor data.
    Copyright (c) 1998 Kysti Mlkki <kmalkki@cc.hut.fi>
    
    Modified from Simon Vogl's i2c-protocol bit-algorithm driver.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/config.h>

/* The next define disables the redefinition of kernel_version. As it is
   already defined and initialized in lm78.c, this would result in twice
   defined symbols. We must include <linux/version.h> by hand now. */
#define __NO_VERSION__

/* Hack to allow the use of CONFIG_MODVERSIONS. In the kernel itself, this
   is arranged from the main Makefile. */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif

/* #include <linux/kernel.h> */

#include <linux/module.h>
#include <linux/version.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <asm/semaphore.h>

/* #include <linux/malloc.h> */
#include <linux/errno.h>
#include <linux/sched.h>

#include "smbus.h"
#include "compat.h"
#include "via.h"

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,1,0))
#include <linux/bios32.h>
#endif


#define I2C_DEBUG  9

#define DEB(x) if (I2C_DEBUG>=1) x;
#define DEB2(x) if (I2C_DEBUG>=2) x;
#define DEBSTAT(x) if (I2C_DEBUG>=3) x;
#define DEBPROTO(x) if (I2C_DEBUG>=9) x;

u32 pm_io_base=0;

/* Physical lines on VIA VT82C586B */

#define setscl(b) \
	(outb(b ? inb(I2C_OUT)|I2C_SCL : inb(I2C_OUT)&~I2C_SCL, I2C_OUT))

#define setsda(b) \
	(outb(b ? inb(I2C_OUT)|I2C_SDA : inb(I2C_OUT)&~I2C_SDA, I2C_OUT))

#define getscl() (0 != (inb(I2C_IN) & I2C_SCL))
#define getsda() (0 != (inb(I2C_IN) & I2C_SDA))

/* SMBus line protocol */

inline void sdalo(void)
{
    setsda(0);
    udelay(BIT_DELAY);
}

inline void sdahi(void)
{
    setsda(1);
    udelay(BIT_DELAY);
}

inline void scllo(void)
{
    setscl(0);
    udelay(BIT_DELAY);
}

inline int sclhi(void)
{
	int start=jiffies;

	setscl(1);
	udelay(BIT_DELAY);

 	while (! getscl() ) {	
		setscl(1);
		if (start+TIMEOUT <= jiffies)
			return -ETIMEDOUT;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,1,0))
		if (current->need_resched) schedule();
#endif
	}
	DEBSTAT(printk("needed %ld jiffies\n", jiffies-start));
	return 0;
} 

void SMBus_start(void) 
{
	DEBPROTO(printk("S "));
	sdalo();
	scllo();
}

void SMBus_repstart(void) 
{
	DEBPROTO(printk(" Sr "));
	setsda(1);
	setscl(1);
	udelay(BIT_DELAY);	
	sdalo();
	scllo();
}

void SMBus_stop(void) 
{
	DEBPROTO(printk("P\n"));
	sdalo();
	sclhi(); 
	sdahi();
}

/* --- byte out, byte in --- */

int SMBus_outb(u8 c)
{
	int i;
	int sb;
	int ack;

	DEB2(printk(" SMBus_outb:%2.2X\n", c & 0xff));

	for ( i=7 ; i>=0 ; i-- ) {
		sb = (c >> i) & 0x1;
		setsda(sb);
		udelay(BIT_DELAY);
		DEBPROTO(printk("%d",sb));
		
		if ( sclhi() == -ETIMEDOUT ) {
			sdahi(); 
			return -ETIMEDOUT;
		};
		scllo();
	}
	sdahi();

	if (sclhi() == -ETIMEDOUT) return -ETIMEDOUT;
	
	/* read ack: SDA should be pulled down by slave */

	ack = getsda();	

	DEB2(printk(" SMBus_outb: getsda() =  0x%2.2x\n", ~ack ));
	DEBPROTO( printk("[%2.2x]",c & 0xff) );
	DEBPROTO(if (0==ack) printk(" A "); else printk(" NA ") );

	scllo();
	return (0==ack);
}

int SMBus_inb(u8 *c)
{
	int i;
	char indata;

	DEB2(printk("SMBus_inb.\n"));

	sdahi();
	indata=0;
	for (i=0;i<8;i++) {
		if ( sclhi() < 0 )
			return -ETIMEDOUT;
		indata *= 2;
		if (getsda()) indata |= 0x01;
		scllo();
	}
	*c = indata & 0xff;
	
	DEBPROTO(printk(" %2.2x", indata & 0xff));
	return 1;   /* ( 1 for acknowledged ) */
}

int SMBus_Via_Access(u8 addr, char read_write, u8 command, int size,
			 union SMBus_Data *data)
{
int ret=-1;
SMBus_start();	

/* More goes here? */



SMBus_stop();
return 0;
}

/* -- VIA Chipset and io-space handling -- */

static int via_io_check(void)
{
	if (check_region(I2C_DIR, IOSPACE) < 0 ) {
		return -ENODEV;
	} else {
		request_region(I2C_DIR, IOSPACE, IOTEXT);
		
		/* Set lines to output and make them high */
		outb(inb(I2C_DIR) | I2C_SCL | I2C_SDA, I2C_DIR);
		setsda(1);
		setscl(1);
	}
	return 0;
}

static u32 read_pm_iobase(void)
{
	struct pci_dev *s_bridge=NULL;
	u32 val;
	
	if (! pci_present())
		return(0);
		
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
	s_bridge = pci_find_device(
		PCI_VENDOR_ID_VIA, PCI_DEVICE_ID_VIA_82C586_3, NULL);

	if (! s_bridge) return 0;
	if ( PCIBIOS_SUCCESSFUL !=
		pci_read_config_dword(s_bridge, PM_IO_BASE_REGISTER, &val) ) 
		return 0;
	return (val & (0xff<<8));
#else
	return 0;  /* Sorry, at this point no compatibility with 2.0.x */
#endif
}

int SMBus_Via_Init(void)
{
	pm_io_base = read_pm_iobase();	
	if (! pm_io_base) return -EIO;
	if (via_io_check() != 0) return -EIO;
	printk("VIA SMBus Initialized\n");
	return 0;
}

void SMBus_Via_Cleanup(void)
{
	release_region(I2C_DIR, IOSPACE);
}
