#!/bin/bash

base_dir=/proc/sys/dev/lm_sensors


# Prints $1, adding a plus sign in front if needed
function print_value()
{
  if echo "$1" | grep -q "^-" ; then
    echo -n "$1"
  else
    echo -n +"$1"
  fi
}

# Print the IN voltages procedure
# $1 is the number to print
# $2 is alarm present (0 or 1)
function print_in() 
{
  local number="$1"
  local alarm="$2"
  set -- `cat "$base_dir"/config-in-labels`
  local count="$number"
  while [ "$count" -gt 0 ] ; do shift ; count=$[$count - 1] ; done
  echo -n "$1"
  set -- `cat "$base_dir"/in"$number"`
  echo -n -e ":\t`print_value $3`V\t"
  echo -n -e "(min = `print_value $1`V,\tmax = `print_value $2`V)"
  if [ "$alarm" -ne 0 ] ; then
    echo -n -e "\tALARM"
  fi
  echo
}

# Print the FAN voltages procedure
# $1 is the number to print
# $2 is alarm present (0 or 1)
function print_fan()
{
  local number="$1"
  local alarm="$2"
  set -- `cat "$base_dir"/config-fan-labels`
  local count="$number"
  while [ "$count" -gt 0 ] ; do shift ; count=$[$count - 1] ; done
  echo -n "$1"
  set -- `cat "$base_dir"/fan"$[$number+1]"`
  echo -n -e ":\t$2 rpm   \t(min=$1 rpm)"
  if [ "$alarm" -ne 0 ] ; then
    echo -n -e "\t\tALARM"
  fi
  if [ "$2" -eq 0 ] ; then 
    echo -n -e "\t(not connected?)" 
  fi
  echo
}

# Print the VID rating
# Very easy, as only one VID rating is supported right now
function print_vid()
{
  local value=`cat "$base_dir"/vid`
  echo -e -n "VID:\t $value"
  if [ $value = 3.50 ] ; then
    echo -e -n "\t(possibly not connected)"
  fi
  echo
}

# Print the LM75 temperature
# $1 is the number to print
function print_lm75()
{
  local number="$1"
  set -- `cat "$base_dir"/config-temp-labels`
  local count="$number"
  while [ "$count" -gt 0 ] ; do shift ; count=$[$count - 1] ; done
  echo -n "$1"
  set -- `cat "$base_dir"/temp-"$[$number+1]"`
  echo -e -n ":\t`print_value $3` C\t"
  echo -e "(Tos = `print_value $1` C,\tThyst = `print_value $2` C)"
}

# Print the mainboard temperature
# Very easy, as only one temperature is supported
function print_mbtemp()
{
  local alarm="$2"
  set -- `cat "$base_dir"/temp-mb`
  echo -n -e "Mainboard: $3 C\t(min = $1 C,\tmax = $2 C)"
  if [ "$alarm" -ne 0 ] ; then
    echo -n -e "\tALARM"
  fi
  echo
}
  
# $1 is the sensor class (in, fan, vid, temp, lm75)
# $2 is the availability bitvector as read from config_present
# $3 is the alarm bitvector
function print_present()
{
  local number=0
  local left="$2"
  local alarms="$3"
  while [ "$left" -ne 0 ]; do
    if [ $[$left % 2] -eq 1 ] ; then
      print_$1 $number $[$alarms % 2]
    fi
    number=$[$number + 1]
    alarms=$[$alarms / 2]
    left=$[$left / 2]
  done
}

# $1 is the alarms read
function print_other()
{
  echo -n "Other alarms:"
  if [ $[$1 & 0x0800] -ne 0 ] ; then echo -n " LM75" ; fi
  if [ $[$1 & 0x1000] -ne 0 ] ; then echo -n " Chassis_Intrusion" ; fi
  if [ $[$1 & 0x2000] -ne 0 ] ; then echo -n " FIFO_Overflow" ; fi
  if [ $[$1 & 0x4000] -ne 0 ] ; then echo -n " SMI_IN" ; fi
  if [ $[$1 & 0x8000] -ne 0 ] ; then echo -n " INT_IN" ; fi
  if [ $[$1 & 0xf800] -eq 0 ] ; then echo -n " (none)" ; fi
  echo
}

# Check out config-present
function print_all()
{
  set -- `cat "$base_dir"/config-present`
  local in_present="$1"
  local fan_present="$2"
  local vid_present="$3"
  local mbtemp_present="$4"
  local lm75_present="$5"
 
  local alarms=`cat "$base_dir"/alarm`
  local in_alarms=$[$alarms & 0x007f]
  local fan_alarms=$[($alarms & 0x0380) >> 7]
  local mbtemp_alarms=$[($alarms & 0x0400) >> 10]

  print_present in "$in_present" "$in_alarms"
  print_present fan "$fan_present" "$fan_alarms"
  print_present mbtemp "$mbtemp_present" "$mbtemp_alarms"
  print_other "$alarms"
  print_present vid "$vid_present" "$alarms" 0
  print_present lm75 "$lm75_present" "$alarms" 0
}

print_all
