/*
    lm78.c - A Linux module for reading sensor data.
    Copyright (c) 1997, 1998  Alexander Larsson <alla@lysator.liu.se>,
    Frodo Looijaard <frodol@dds.nl> and Philip Edelbrock <phil@netroedge.com>.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <linux/config.h>

/* Hack to allow the use of CONFIG_MODVERSIONS. In the kernel itself, this
   is arranged from the main Makefile. */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/module.h>
#include <linux/proc_fs.h>
#include "main.h"
#include "smbus.h"
#include "compat.h"

#define LM_FULL_CONFIG /* To include the full version of lmconfig.h */
#include "lmconfig.h"

#include "lmversion.h"

static int no_lm78 = 0;
MODULE_PARM(no_lm78,"i");
MODULE_PARM_DESC(no_lm78,"Set to disable all LM78(-clone) probing");

static int no_lm75 = 0;
MODULE_PARM(no_lm75,"i");
MODULE_PARM_DESC(no_lm75,"Set to disable all LM75(-clone) probing");


/* Kludge; if EXTERN==external, we can't put an initial value behind it;
   if EXTERN=static, it must be there, or we get an incompatible types
   error. 
*/
#ifdef ALLINONE
EXTERN 
#endif
       struct LM_Sensor_Data LM_Sensor_Data;

#ifdef ALLINONE
EXTERN
#endif
       struct LM_Sensor_Present LM_Sensor_Present =
              { 0x00, 0x00, 0x00, 0x00, 0x00 }; 
       /* Everything unavailable until proved otherwise */

extern int init_module(void);
extern void cleanup_module(void);

#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
static int read_sensors(char *buf, char **start, off_t offset,
                        int len, int *eof, void *private);
#else
static int read_sensors(char *, char **, off_t, int, int);
#endif

MODULE_AUTHOR("Alexander Larsson <alla@lysator.liu.se>, Frodo Looijaard <frodol@dds.nl>, Philip Edelbrock <phil@Ren.netroedge.com>");

MODULE_DESCRIPTION("Driver for reading sensor values from a sensor chip (lm78/lm79/lm80/etc) on the mainboard.");


#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
 static struct proc_dir_entry *proc_sensors = NULL;
#else
 static int inodeval = 0; /* Things depend on this initialization! */
 static struct proc_dir_entry dir =
 {
   0, 7, "sensors", 
   S_IFREG | S_IRUGO, 1, 0, 0,
   0, NULL,
   &read_sensors
 };
#endif

/*
 *	Initialize the module.
 */
int init_module(void)
{
  int err;

  printk("lm_sensors version %s (%s)\n",LM_VERSION,LM_DATE);
#ifndef SMBUS_MODULE
  if (SMBus_Init() != 0) 
    printk("SMBus Initialization failed.\n");
#endif


  if (no_lm78 == 0) {
   err=LM78_Init();
   if (err != 0) {
     printk("LM78 not found or busy.\n");
     no_lm78 = 1;
   }
  }

  if (no_lm75 == 0) {
    err=LM75_Init();
    if (err != 0) {
      printk("SMBus/LM75 not found or busy.\n");
      no_lm75 = 1;
    }
  }

/* If everything is disabled, then there isn't much point, is there? */
  if (no_lm75 && no_lm78) {
    SMBus_Cleanup();
    return -1;
  }

/* Force a full LM_Sensor_Data read next time, but wait a full second first. */
  LM_Sensor_Data.valid = 0;
  LM_Sensor_Data.last_read=jiffies;

/* Set the selected parts */
  LM_Sensor_Config.selected.in &= LM_Sensor_Present.in;
  LM_Sensor_Config.selected.fan &= LM_Sensor_Present.fan;
  LM_Sensor_Config.selected.vid &= LM_Sensor_Present.vid;
  LM_Sensor_Config.selected.temp_mb &= LM_Sensor_Present.temp_mb;
  LM_Sensor_Config.selected.lm75 &= LM_Sensor_Present.lm75;

  /* register /proc/sensors */
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
  
  proc_sensors = create_proc_entry("sensors", 0, 0);
  if (proc_sensors) {
    proc_sensors->read_proc = read_sensors;
  } else {
    printk("Couldn't create /proc/sensors\n");
    cleanup_module();
    return -ENOENT;
  }
  
#else
  
/*
 *	Try to register the file in the root directory of the proc
 *	filesystem.  Dynamic registration means we won't provide a fixed
 *	inode number, but rather the registration code will allocate one for
 *	us.  This is safer, since we never know when someone will add a new
 *	fixed inode number to the filesystem.  Possible returns are 0
 *	(success) and -ENOENT (failure), and since there's only one error we
 *	won't worry unduly about messages.  If successful, continue.
 */
  if (proc_register_dynamic(&proc_root, &dir)) {
    printk("Couldn't create /proc/sensors\n");
    cleanup_module();
  }
  
/*
 *      Save the inode value from the registration call, as we will need
 *      it to unregister.
 */
  inodeval = dir.low_ino;

#endif

  /* Register the sysctl interface */
  if (LM_Sysctl_Init())
    printk("Couldn't register /proc/sys/dev/lm_sensors!");
  
  return 0;
}

/*
 *	Remove the module.  If a nonzero inode value was returned,
 *	unregister the /proc directory entry; that's all we need to do.
 */

void cleanup_module(void)
{
	LM_Sysctl_Cleanup();
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
        if (proc_sensors)
		proc_unregister(&proc_root, proc_sensors->low_ino);
#else
        if (inodeval)
		proc_unregister(&proc_root, inodeval);
#endif

        if (! no_lm75)
                LM75_Cleanup();
	if (! no_lm78)
		LM78_Cleanup();

	if (SMBus_Initialized)
		SMBus_Cleanup();
      
}


/* This function is called whenever a 'read' is done on /proc/sensors.
   
   LINUX KERNEL 2.0.x
   ==================

   Checked for 2.0.34.
   This function is known as the get_info function.

   (INPUT)
     buf: A kernel-space buffer where we can put the information.
     len: The buffer length (we have a bit of slack here: if we write 
          'a little' beyond this, things will still work out)
   (OUTPUT)
     *buf: The complete contents of the /proc file, starting at the
           beginning.
     return value: the number of chars written to buf.

   This is all you have to do if all information fits in the buffer (it is 
   about 3K in length). You can safely ignore offset and start in this case.

   If your /proc entry gets larger, you have to do a bit more work.

   (INPUT)
     buf: See above
     len: see above
     offset: The offset from the beginning of the /proc file where the user
             wants to start reading
   (OUTPUT)
     *buf: A part of the /proc file. The requested information does not
           have to start at the beginning.
     start: this is the place within buf where the information starts (ie.
            you must use offset to compute this!).
     return value: The number of valid characters written to buf. Begin
                   counting at start, not at buf!

   You do not have to worry about not the whole information being in buf.
   Another call will be done (with a higher offset value) if more is
   needed.

   LINUX KERNEL 2.1.x
   ==================

   Checked for 2.1.104.
   This function is known as the read_proc function.

   (INPUT)
     buf: A kernel-space buffer where we can put the information.
     len: The buffer length (we have a bit of slack here: if we write 
          'a little' beyond this, things will still work out)
   (OUTPUT)
     *buf: The complete contents of the /proc file, starting at the
           beginning.
     eof: You can set this to true (anything but 0) to indicate we have
          reached the end of the file with this read. Not strictly
          necessary, but it is more efficient (else this function may be
          called again).
     return value: The number of valid characters written to buf. Begin
                   counting at start, not at buf! 

   If your /proc entry gets larger, you have to do a bit more work.

   (INPUT)
     buf: See above
     len: See above
     offset: The offset from the beginning of the /proc file where the user
             wants to start reading
   (OUTPUT)
     *buf: A part of the /proc file. The requested information does not
           have to start at the beginning.
     eof: See above
     start: this is the place within buf where the information starts (ie.
            you must use offset to compute this!).
     return value: The number of valid characters written to buf.

   You do not have to worry about not the whole information being in buf.
   Another call will be done (with a higher offset value) if more is
   needed. Ignore private.

   A third way to do this:

   (INPUT)
     buf: See above
     len: See above
     offset: The offset from the beginning of the /proc file where the user
             wants to start reading
   (OUTPUT)
     *buf: A part of the /proc file, with the requested information starting
           at the beginning.
     eof: See above
     start: (long)start is added to the offset pointer after reading. This
            allows you to play tricks with the file offset, independent of
            the number of read bytes.
     return value: the number of chars written to buf.
   
*/

#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
static int read_sensors(char *buf, char **start, off_t offset, int len, int *eof, void *private)
#else
static int read_sensors(char *buf, char **start, off_t offset, int len, int unused)
#endif
{

  len=0;

  LM_Update_Data();

  /* Grab the mutex */

  if (! no_lm78)
    len += LM78_Print_Values(buf+len);
  if (! no_lm75)
    len += LM75_Print_Values(buf+len);

  /* Release the mutex */
  
  return len;
}

/* This function will make sure all values in LM_Sensor_Data are up-to-date */
void LM_Update_Data(void)
{
  /* Static variables within function are only initialized once, at the
     start of the program! 
     No two processes may try to update all values at once. This would be
     a waste of resources. */
  static struct semaphore update_sem = MUTEX; 

  int tosleep;

  /* Note that check for the jiffies counter overflow. Note that we may
     probe once too many if the jiffy counter overflows (too bad).
     assume that when it overflows, it will be reset to 0. If not,
     we may probe once too many (too bad). */

  down(&update_sem);

  /* If a process sleeps here, no other process will get in either.
     But this is all right, because they would have to sleep too! */

  if (! LM_Sensor_Data.valid) { /* Sleep until valid */
    if (jiffies < LM_Sensor_Data.last_read) /* jiffies overflow! */
      tosleep=HZ+HZ/2;
    else
      tosleep=HZ+HZ/2 + LM_Sensor_Data.last_read - jiffies;
    if (tosleep > 0) {
      current->state = TASK_INTERRUPTIBLE;
      schedule_timeout(tosleep);
    }
  }

  if ((jiffies - LM_Sensor_Data.last_read > HZ+HZ/2 ) ||
      (jiffies < LM_Sensor_Data.last_read) || ! LM_Sensor_Data.valid) {
    if (! no_lm78)
      LM78_Update_Values();
    if (! no_lm75)
      LM75_Update_Values();
    LM_Sensor_Data.valid = 1;
    LM_Sensor_Data.last_read = jiffies;
    /* Any process which was locked out will probably see that the last
       update was less than 1.5 secs ago, and not update themselves. But
       they really had to wait until it was finished. */
  }
  
  up(&update_sem);

} 

