/*
    lm78.c - Part of a Linux module for reading sensor data.
    Copyright (c) 1997, 1998  Alexander Larsson <alla@lysator.liu.se>,
    Frodo Looijaard <frodol@dds.nl> and Philip Edelbrock <phil@netroedge.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/config.h>

/* The next define disables the redefinition of kernel_version. As it is
   already defined and initialized in lm78.c, this would result in twice
   defined symbols. We must include <linux/version.h> by hand now. */
#define __NO_VERSION__

/* Hack to allow the use of CONFIG_MODVERSIONS. In the kernel itself, this
   is arranged from the main Makefile. */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/module.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <asm/semaphore.h>
#include <linux/version.h>
#include "lm78.h"
#include "main.h"
#include "lmconfig.h"
#include "lm_sensors.h"
#include "compat.h"

/* These are functions used only internally. */
static u8 lm78_read_value(u8 address);
static void lm78_write_value(u8 address, u8 value);
static int fan_decode(u8 div);
static u8 fan_encode(int div);
static void read_vid_lines(void);

/* Kludge; if EXTERN==external, we can't put an initial value behind it;
   if EXTERN=static, it must be there, or we get an incompatible types
   error.
*/
#ifdef ALLINONE
EXTERN
#endif
       int LM_Type = LMTYPE_UNKNOWN;

#ifdef ALLINONE
EXTERN
#endif
       int LM78_SMBus = 0;

#ifdef ALLINONE
EXTERN
#endif
	struct semaphore LM78_Sem = MUTEX;

static const char *chip = ""; 
MODULE_PARM(chip,"1-10s");
MODULE_PARM_DESC(chip,"Force a certain chip to be detected (lm78, lm78-j, lm79, lm80, w83781d, probe)");

static int smbus = -1; /* 0=ISA, 1=SMBus, -1=Probe */
MODULE_PARM(smbus,"i");
MODULE_PARM_DESC(smbus,"Use ISA (0) or SMBus (1) access");

static int isa_addr = -1; /* Probe */
MODULE_PARM(isa_addr,"i");
MODULE_PARM_DESC(isa_addr,"ISA address to use");

static int smbus_addr = -1; /* Probe */
MODULE_PARM(smbus_addr,"i");
MODULE_PARM_DESC(smbus_addr,"SMBus address to use");

/* This function initializes the LM78. If it is found, and initializing
   succeeds, 0 is returned; if not, an error value is returned. */
int LM78_Init(void)
{
  u8 res;
  int i,found;

  if (!strcmp(chip,"lm78")) {
    printk("Chiptype LM78 forced (no detection done)\n");
    LM_Type=LMTYPE_78;
  } else if (!strcmp(chip,"lm78-j")) {
    printk("Chiptype LM78-J forced (no detection done)\n");
    LM_Type=LMTYPE_78_J;
  } else if (!strcmp(chip,"lm79")) {
    printk("Chiptype LM79 forced (no detection done)\n");
    LM_Type=LMTYPE_79;
  } else if (!strcmp(chip,"lm80")) {
    printk("Chiptype LM80 forced (no detection done)\n");
    LM_Type=LMTYPE_80;
  } else if (!strcmp(chip,"w83781d")) {
    printk("Chiptype W83781D forced (no detection done)\n");
    LM_Type=LMTYPE_WINB;
  } else if (strcmp(chip,"probe") && (strcmp(chip,"")))
    printk("Invalid type after chip= parameter (%s), will probe now\n",chip);

  if ((smbus < -1) || (smbus > 1)) {
    printk("Invalid number after smbus= parameter (%d), will probe now\n",
           smbus);
    smbus = -1;
  }

  if ((isa_addr < -1) || (isa_addr > 0xffff)) {
    printk("Invalid address after isa_addr= parameter (%d), default is used (%d)\n",isa_addr,LM78_BASE);
    isa_addr=LM78_BASE;
  }

  if ((smbus_addr < -1) || (smbus_addr > 0x7f)) {
    printk("Invalid address after smbus_addr= parameter (%d), will probe now\n",smbus_addr);
    smbus_addr=-1;
  }

/* Some things just don't go along, others are certain. */
  if (LM_Type == LMTYPE_80) {
    if (smbus == 0) {
      printk("LM80 over ISA? Impossible!\n");
      return -ENODEV;
    } else
      smbus = 1;
  }

  if (smbus <= 0) {
    found=0;
    if (isa_addr < 0)
      isa_addr = LM78_BASE;
    if (check_region(isa_addr, LM78_EXTENT)) {
      printk("Warning: ISA I/O region 0x%04x to 0x%04x already used.\n",
             isa_addr,isa_addr+LM78_EXTENT-1);
    }
    else {
      outb_p( LM78_DUMMY, LM78_ADDRESS_REG );
      res = inb_p(LM78_ADDRESS_REG);
      inb_p(LM78_DATA_REG); /* reset busy flag */

      if ((res & 0x7F) == LM78_DUMMY) {
        request_region(isa_addr, LM78_EXTENT, "lm78");
        smbus=0;
        found=1;
      }
    }
    if ((smbus == 0) && !found) {
      printk("LM78 not detected on ISA bus\n");
      return -ENODEV;
    }   
  }

  if ((smbus == -1) || (smbus == 1)) {
  
    if (! SMBus_Initialized) {
      printk("Can't probe SMBus for LM78(-clone) availability, as it is not initialized!\n");
      return -ENODEV;
    }

    found=0;
  /* Note: this looks for LM78's only within the LM80 addressing range,
  	    but LM78 can have any valid SMBus address (up to 0x7F) */
   if (smbus_addr==-1) {
     for (i=0x20; ((i <= 0x2f) && (!found)); i++)
     {
      if (!check_smbus(i) && (SMBus_Read_Byte_Data(i,1) != -1)) {
        request_smbus(i,"lm78");
        smbus=1;
        smbus_addr=i;
        found=1;
      }
     }
    } else {
      if (check_smbus(smbus_addr))
        printk("SMBus address 0x%0x already in use!\n",smbus_addr);
      else if (SMBus_Read_Byte_Data(smbus_addr,1) != -1) {
        request_smbus(smbus_addr,"lm78");
	found=1;
      }
    }

    if (!found) {
      if (smbus_addr == -1)
        printk("No LM78 compatible chip found on the SMBus\n");
      else
        printk("No LM78 compatible chip found on the SMBus, address 0x%02x\n",
               smbus_addr);
      return -ENODEV;
    }
  }

  /* OK. So now we have either grabbed an ISA range, or decided to use
     the SMBus and found an SMBus address. */

   LM78_SMBus = smbus;

  /* From now on, we will use lm78_write_value and lm78_read_value,
     as all other things are the same for SMBus and ISA bus access */
 
  /* Reset all except Watchdog values and last conversion values
     This sets fan-divs to 2, among others 
     Due to some magic, this will work for both LM80 and LM78 compatible
     chips. */
  lm78_write_value(LM78_CONFIG, 0x80 );

  /* Determine the LM version */

  /* Probe for an LM80. There is no 'nice' way to do this, but the
     following hack seems to work well: because the LM80 only uses
     a 6-bit register address, we can check whether a read/write
     to address returns the same data as one to (address + 0x40).
     We only need this to do if we are on the SMBus. We do some
     tricks with the configuration register (0x00/0x40) to check
     this. 0x00 is POST RAM on the LM78; I think Linux does not
     use this anyway. Perhaps we need a spinlock on this test (with
     IRQ disable) to make it really safe. */

  if (!LM_Type && LM78_SMBus && 
      (lm78_read_value(0x40) == (i=lm78_read_value(0x00)))) {
      /* The 0x40 == 0x00 is not really needed, but if it is false, 
         it is not an LM80 anyway */
    lm78_write_value(0x00,0x80); 
    if (lm78_read_value(0x00) != 0x80)
      /* Note that if this was the configuration register, it would now be
         changed to a power-on value */
      LM_Type=LMTYPE_80;
    else /* Restore POST RAM byte */
      lm78_write_value(0x00,i); /* Restore */
  }

  /* Check for W83781D. Note the careful probing: we do not write anything
     to a register until we are already almost sure we have found the
     Winbond. */
  if (! LM_Type)
    if (res = lm78_read_value(LM78_WINB_ID), (res == 0x5c) || (res == 0xa3) ) {
      res = lm78_read_value(LM78_WINB_BANK);
      lm78_write_value(LM78_WINB_BANK,res & 0x7f);
      if (lm78_read_value(LM78_WINB_ID) == 0xa3) {
        lm78_write_value(LM78_WINB_BANK,res | 0x80);
        if (lm78_read_value(LM78_WINB_ID) == 0x5c)
          LM_Type = LMTYPE_WINB;
      }
      lm78_write_value(LM78_WINB_BANK,res);
    }

  if (! LM_Type ) {
    res = lm78_read_value(LM78_RESET) >> 6;
    if (res == 3)
       LM_Type = LMTYPE_79;
    else if (res == 1)
       LM_Type = LMTYPE_78_J;
    else if (res == 0)
       LM_Type = LMTYPE_78;
   }

  /* Set defaults in LM_Sensor_Present */
  LM_Sensor_Present.in = 0x7f;
  LM_Sensor_Present.fan = 0x07;
  LM_Sensor_Present.vid = 0x01;
  LM_Sensor_Present.temp_mb = 0x01;

  /* LM80 initializations. */
  if (LM_Type == LMTYPE_80) {
    LM_Sensor_Present.fan &= 0x03;          /* Only two fans available */
    LM_Sensor_Present.vid = 0;              /* No VID-lines */
  }

  /* Winbond initializations. Note that, according to docs, the Winbond-
     specific registers are only set to default values when the Master
     Reset pin is accessed - which is impossible to do through software.
     Here, I try to touch only the necessary stuff, and assume the BIOS
     took care of the more arcane settings. This may have to change,
     depending on input from users */
  if (LM_Type == LMTYPE_WINB) {
     /* Disable the power-on BEEP */
     lm78_write_value(LM78_WINB_FANCONT,
                      lm78_read_value(LM78_WINB_FANCONT) | 0x80);
     /* Make sure we use BANK0 from now on. BANK1/2 are for LM75-compatible
        sensors access through ISA, which we do not want right now .
        Note that we really must clear all low 3 bits, which is not as the
        docs suggest! */
     lm78_write_value(LM78_WINB_BANK,
                      (lm78_read_value(LM78_WINB_BANK) & 0xf8));
     /* Disable the beeping once and for all */
     lm78_write_value(LM78_WINB_BEEP2,lm78_read_value(LM78_WINB_BEEP2) & 0x7f);
     /* Enable all temperature registers, reset SMBus locations */
     lm78_write_value(LM78_WINB_SMBADDR,0x01);
  }

  /* Note: We always set the limits here, even if the compile-time present
     bit is turned off. It can always be turned on later, and without
     sane limits, this would be not very nice. */

  LM78_Set(LM_PROBE_TEMP_MIN,0,LM78_TEMP_INIT_MIN);
  LM78_Set(LM_PROBE_TEMP_MAX,0,LM78_TEMP_INIT_MAX);

  /* Read the VID-values - may be needed to set some limits. */
  if (LM_Sensor_Present.vid)
    read_vid_lines();

  LM78_Set(LM_PROBE_IN_MIN,0,LM78_IN0_INIT_MIN);
  LM78_Set(LM_PROBE_IN_MAX,0,LM78_IN0_INIT_MAX);

  LM78_Set(LM_PROBE_IN_MIN,1,LM78_IN1_INIT_MIN);
  LM78_Set(LM_PROBE_IN_MAX,1,LM78_IN1_INIT_MAX);

  LM78_Set(LM_PROBE_IN_MIN,2,LM78_IN2_INIT_MIN);
  LM78_Set(LM_PROBE_IN_MAX,2,LM78_IN2_INIT_MAX);

  LM78_Set(LM_PROBE_IN_MIN,3,LM78_IN3_INIT_MIN);
  LM78_Set(LM_PROBE_IN_MAX,3,LM78_IN3_INIT_MAX);

  LM78_Set(LM_PROBE_IN_MIN,4,LM78_IN4_INIT_MIN);
  LM78_Set(LM_PROBE_IN_MAX,4,LM78_IN4_INIT_MAX);

  LM78_Set(LM_PROBE_IN_MIN,5,LM78_IN5_INIT_MIN);
  LM78_Set(LM_PROBE_IN_MAX,5,LM78_IN5_INIT_MAX);

  LM78_Set(LM_PROBE_IN_MIN,6,LM78_IN6_INIT_MIN);
  LM78_Set(LM_PROBE_IN_MAX,6,LM78_IN6_INIT_MAX);

  LM78_Set(LM_PROBE_FAN_DIV,1,LM78_FAN1_INIT_DIV);
  LM78_Set(LM_PROBE_FAN_MIN,1,LM78_FAN1_INIT_MIN);

  LM78_Set(LM_PROBE_FAN_DIV,2,LM78_FAN2_INIT_DIV);
  LM78_Set(LM_PROBE_FAN_MIN,2,LM78_FAN2_INIT_MIN);

  LM78_Set(LM_PROBE_FAN_DIV,3,LM78_FAN3_INIT_DIV);
  LM78_Set(LM_PROBE_FAN_MIN,3,LM78_FAN3_INIT_MIN);

  /* Start monitoring */
  if (LM_Type == LMTYPE_80)
    lm78_write_value(LM80_CONFIG, (lm78_read_value(LM80_CONFIG) & 0xf7) | 0x01);
  else
    lm78_write_value(LM78_CONFIG, (lm78_read_value(LM78_CONFIG) & 0xf7) | 0x01);

  if (LM_Type == LMTYPE_UNKNOWN) {
    printk("Unknown LM78 clone detected (treated as a plain LM78)\n");
    if (LM78_SMBus)
      printk("Accessed through the SMBus\n");
    else
      printk("Accessed through the ISA bus\n");
  }
  else {
    if (LM_Type == LMTYPE_78) 
      printk("LM78");
    else if (LM_Type == LMTYPE_78_J)
      printk("LM78-J");
    else if (LM_Type == LMTYPE_79)
      printk("LM79");
    else if (LM_Type == LMTYPE_WINB)
      printk("W83781D");
    else if (LM_Type == LMTYPE_80)
      printk("LM80");
    printk(" detected and installed (accessed through the ");
    if (LM78_SMBus) 
      printk("SMBus, address 0x%02x)\n",smbus_addr);
    else
      printk("ISA bus, I/O address 0x%04x)\n",isa_addr);
  }
  return 0;
}

/* This function cleans up after the module. Only call it after a succesful
   LM78_Init call! */
void LM78_Cleanup(void)
{
   if (LM78_SMBus && (smbus_addr != -1))
     release_smbus(smbus_addr);
   release_region(isa_addr, LM78_EXTENT);
}

/* This function tries to update the current sensor data and to put it
   in human-readable form in buf. It returns the number of characters
   written to buf. 
   TODO: Check length of buf. */
int LM78_Print_Values (char *buf)
{
  int len = 0;
  int v, ints;

  /* Redundant test for LM_Sensor_Data.valid - better safe then sorry! */
  if (LM_Sensor_Data.valid) {
#define PRINT_SIGN(val) (((val) < 0)?'-':'+')
#define PRINT_VALUES(label, value, max, min, intr) \
    sprintf(buf + len, "%s:\t%c%d.%02dV\t(min = %c%d.%02dV,\t" \
            "max = %c%d.%02dV)\t%s\n", label,PRINT_SIGN(value), \
            (int) (abs(value) / 100), (int) (abs(value) % 100), \
            PRINT_SIGN(min), (int) (abs(min) / 100),\
            (int) (abs(min) % 100), PRINT_SIGN(max), \
            (int) (abs(max) / 100), (int) (abs(max) % 100), intr?"ALARM":"")

    ints = LM78_Get(LM_PROBE_ALARM,0);

    if (LM_Sensor_Config.selected.in & 0x01)
      len += PRINT_VALUES(LM_Sensor_Config.in_label[0], 
                          LM78_Get(LM_PROBE_IN,0), 
                          LM78_Get(LM_PROBE_IN_MAX,0), 
                          LM78_Get(LM_PROBE_IN_MIN,0), 
                          ints & LMALARM_IN0);

    if (LM_Sensor_Config.selected.in & 0x02)
      len += PRINT_VALUES(LM_Sensor_Config.in_label[1], 
                          LM78_Get(LM_PROBE_IN,1), 
                          LM78_Get(LM_PROBE_IN_MAX,1), 
                          LM78_Get(LM_PROBE_IN_MIN,1), 
                          ints & LMALARM_IN1);

    if (LM_Sensor_Config.selected.in & 0x04)
      len += PRINT_VALUES(LM_Sensor_Config.in_label[2], 
                          LM78_Get(LM_PROBE_IN,2), 
                          LM78_Get(LM_PROBE_IN_MAX,2), 
                          LM78_Get(LM_PROBE_IN_MIN,2), 
                          ints & LMALARM_IN2);

    if (LM_Sensor_Config.selected.in & 0x08)
      len += PRINT_VALUES(LM_Sensor_Config.in_label[3], 
                          LM78_Get(LM_PROBE_IN,3), 
                          LM78_Get(LM_PROBE_IN_MAX,3), 
                          LM78_Get(LM_PROBE_IN_MIN,3), 
                          ints & LMALARM_IN3);

    if (LM_Sensor_Config.selected.in & 0x10)
      len += PRINT_VALUES(LM_Sensor_Config.in_label[4], 
                          LM78_Get(LM_PROBE_IN,4), 
                          LM78_Get(LM_PROBE_IN_MAX,4), 
                          LM78_Get(LM_PROBE_IN_MIN,4), 
                          ints & LMALARM_IN4);

    if (LM_Sensor_Config.selected.in & 0x20)
      len += PRINT_VALUES(LM_Sensor_Config.in_label[5], 
                          LM78_Get(LM_PROBE_IN,5), 
                          LM78_Get(LM_PROBE_IN_MAX,5), 
                          LM78_Get(LM_PROBE_IN_MIN,5), 
                          ints & LMALARM_IN5);

    if (LM_Sensor_Config.selected.in & 0x40)
      len += PRINT_VALUES(LM_Sensor_Config.in_label[6], 
                          LM78_Get(LM_PROBE_IN,6), 
                          LM78_Get(LM_PROBE_IN_MAX,6), 
                          LM78_Get(LM_PROBE_IN_MIN,6), 
                          ints & LMALARM_IN6);


    if (LM_Sensor_Config.selected.fan & 0x01)
      len += sprintf(buf+len,"%s:\t%d rpm   \t(min=%d rpm)   \t%s\t%s\n", 
                     LM_Sensor_Config.fan_label[0],
                     LM78_Get(LM_PROBE_FAN,1),LM78_Get(LM_PROBE_FAN_MIN,1),
                     (ints & LMALARM_FAN1) ?"ALARM":"",
                     LM78_Get(LM_PROBE_FAN,1)==0?"(not connected?)":"");

    if (LM_Sensor_Config.selected.fan & 0x02)
      len += sprintf(buf+len, "%s:\t%d rpm   \t(min=%d rpm)   \t%s\t%s\n", 
                     LM_Sensor_Config.fan_label[1],
                     LM78_Get(LM_PROBE_FAN,2),LM78_Get(LM_PROBE_FAN_MIN,2),
                     (ints & LMALARM_FAN2)?"ALARM":"",
                     LM78_Get(LM_PROBE_FAN,2)==0?"(not connected?)":"");

    if (LM_Sensor_Config.selected.fan & 0x04)
      len += sprintf(buf+len, "%s:\t%d rpm   \t(min=%d rpm)   \t%s\t%s\n", 
                     LM_Sensor_Config.fan_label[2],
                     LM78_Get(LM_PROBE_FAN,3),LM78_Get(LM_PROBE_FAN_MIN,3),
                     (ints & LMALARM_FAN3)?"ALARM":"",
                     LM78_Get(LM_PROBE_FAN,3)==0?"(not connected?)":"");



    if (LM_Sensor_Config.selected.temp_mb)
      len += sprintf(buf+len, "Mainboard: %d C\t(min = %d C,\t"
                     "max = %d C)\t%s\n", 
                     LM78_Get(LM_PROBE_TEMP,0), LM78_Get(LM_PROBE_TEMP_MIN,0),
                     LM78_Get(LM_PROBE_TEMP_MAX,0),
                     (ints & LMALARM_TEMP)?"ALARM":"");

    len += sprintf(buf+len, "Other alarms:");

    if(ints & LMALARM_LM75) len += sprintf(buf+len, " LM75");
    if(ints & LMALARM_CHAS) len += sprintf(buf+len, " Chassis_Intrusion");
    if(ints & LMALARM_FIFO) len += sprintf(buf+len, " FIFO_Overflow");
    if(ints & LMALARM_SMI) len += sprintf(buf+len, " SMI_IN");
    if(ints & LMALARM_INT) len += sprintf(buf+len, " INT_IN");
    if (! (ints & (LMALARM_LM75|LMALARM_CHAS|LMALARM_FIFO|LMALARM_SMI|
                   LMALARM_INT)))
      len += sprintf(buf+len," (none)");
    len += sprintf(buf+len, "\n");
        

    if (LM_Sensor_Config.selected.vid) {
      v = LM78_Get(LM_PROBE_VID,0);
      len += sprintf(buf+len,"VID:\t %d.%02d\t%s\n",abs(v)/100,abs(v)%100,
                     (v<0)?"(possibly not connected)":
                      ((v==0)?"(unknown internal value)":""));
    }
  }
  return len;
}


/* Read a value from the LM78. Theoretically, several drivers can use
   the LM78 at once by using this function. In practice, this is unlikely,
   but still it won't do any harm. 
   The semaphore is for the Winbond: when you read another of its banks,
   all registers are temporary hosed, so we can not read/write them. 
   We still need to check for LM78_BUSY, as some other program could access
   the chip. 
   Note that SMBus failed reads are not registered! */
u8 lm78_read_value(u8 address) 
{
  int res;

  if (LM78_SMBus)
    return LM78_SMBUS_READ(address);
  else {
    down(&LM78_Sem);
    while (LM78_BUSY) {
      current->state = TASK_INTERRUPTIBLE;
      schedule_timeout(10);
    }
    res = LM78_READ(address);
    up(&LM78_Sem);
    return res;
  }
}

/* See comments at lm78_read_value! */
void lm78_write_value(u8 address, u8 value)
{
  if (LM78_SMBus)
    LM78_SMBUS_WRITE(address,value);
  else {
    down(&LM78_Sem);
    while (LM78_BUSY) {
      current->state = TASK_INTERRUPTIBLE;
      schedule_timeout(10);
    }
    LM78_WRITE(address,value);
    up(&LM78_Sem);
  }
}
   
/* Read the chip for vid-lines and put the result in LM_Sensor_Data.vid.
   Call this only if LM_Sensor_Present.vid <> 0! */
void read_vid_lines(void) 
{
  LM_Sensor_Data.vid = lm78_read_value(LM78_VID) & 0x0f;
  if ((LM_Type == LMTYPE_79) || (LM_Type == LMTYPE_WINB))
    LM_Sensor_Data.vid |= ((lm78_read_value(LM78_RESET) & 0x01) << 4);
  else
    LM_Sensor_Data.vid |= 0x10;
} 

/* Reread the sensor information into struct LM_Sensor_Data. 
   Note that we always read everything, unless the hardware is not
   available! */
void LM78_Update_Values(void)
{
  /* TODO: grab Spin lock on LM_Sensor_Data */

if (LM_Type == LMTYPE_80) {

  if (LM_Sensor_Present.in & 0x01) {
    LM_Sensor_Data.in[0] = lm78_read_value(LM80_IN0);
    LM_Sensor_Data.in_max[0] = lm78_read_value(LM80_IN0_MAX);
    LM_Sensor_Data.in_min[0] = lm78_read_value(LM80_IN0_MIN);
  }

  if (LM_Sensor_Present.in & 0x02) {
    LM_Sensor_Data.in[1] = lm78_read_value(LM80_IN1);
    LM_Sensor_Data.in_max[1] = lm78_read_value(LM80_IN1_MAX);
    LM_Sensor_Data.in_min[1] = lm78_read_value(LM80_IN1_MIN);
  }

  if (LM_Sensor_Present.in & 0x04) {
    LM_Sensor_Data.in[2] = lm78_read_value(LM80_IN2);
    LM_Sensor_Data.in_max[2] = lm78_read_value(LM80_IN2_MAX);
    LM_Sensor_Data.in_min[2] = lm78_read_value(LM80_IN2_MIN);
  }

  if (LM_Sensor_Present.in & 0x08) {
    LM_Sensor_Data.in[3] = lm78_read_value(LM80_IN3);
    LM_Sensor_Data.in_max[3] = lm78_read_value(LM80_IN3_MAX);
    LM_Sensor_Data.in_min[3] = lm78_read_value(LM80_IN3_MIN);
  }

  if (LM_Sensor_Present.in & 0x10) {
    LM_Sensor_Data.in[4] = lm78_read_value(LM80_IN4);
    LM_Sensor_Data.in_max[4] = lm78_read_value(LM80_IN4_MAX);
    LM_Sensor_Data.in_min[4] = lm78_read_value(LM80_IN4_MIN);
  }

  if (LM_Sensor_Present.in & 0x20) {
    LM_Sensor_Data.in[5] = lm78_read_value(LM80_IN5);
    LM_Sensor_Data.in_max[5] = lm78_read_value(LM80_IN5_MAX);
    LM_Sensor_Data.in_min[5] = lm78_read_value(LM80_IN5_MIN);
  }

  if (LM_Sensor_Present.in & 0x40) {
    LM_Sensor_Data.in[6] = lm78_read_value(LM80_IN6);
    LM_Sensor_Data.in_max[6] = lm78_read_value(LM80_IN6_MAX);
    LM_Sensor_Data.in_min[6] = lm78_read_value(LM80_IN6_MIN);
  }

  LM_Sensor_Data.vid = 0x0f; /* Not available; this is safe for now */

  if (LM_Sensor_Present.temp_mb) {
    LM_Sensor_Data.temp_mb = lm78_read_value(LM80_TEMPERATURE);
    LM_Sensor_Data.temp_mb_max = lm78_read_value(LM80_TEMPERATURE_MAX);
    LM_Sensor_Data.temp_mb_min = lm78_read_value(LM80_TEMPERATURE_MIN);
  }

  if (LM_Sensor_Present.fan & 0x01) {
    LM_Sensor_Data.fan[0] = lm78_read_value(LM80_FAN1);
    LM_Sensor_Data.fan_min[0] = lm78_read_value(LM80_FAN1_MIN);
  }
  if (LM_Sensor_Present.fan & 0x02) {
    LM_Sensor_Data.fan[1] = lm78_read_value(LM80_FAN2);
    LM_Sensor_Data.fan_min[1] = lm78_read_value(LM80_FAN2_MIN);
  }

  LM_Sensor_Data.fan_div[0] = lm78_read_value(LM80_FANDIV);

  LM_Sensor_Data.fan[2] = 0; /* Not available; this is safe for now */

  LM_Sensor_Data.interrupt_status[0] = lm78_read_value(LM80_INTSTAT1);
  LM_Sensor_Data.interrupt_status[1] = lm78_read_value(LM80_INTSTAT2);

} else {

  if (LM_Sensor_Present.in & 0x01) {
    LM_Sensor_Data.in[0] = lm78_read_value(LM78_IN0);
    LM_Sensor_Data.in_max[0] = lm78_read_value(LM78_IN0_MAX);
    LM_Sensor_Data.in_min[0] = lm78_read_value(LM78_IN0_MIN);
  }

  if (LM_Sensor_Present.in & 0x02) {
    LM_Sensor_Data.in[1] = lm78_read_value(LM78_IN1);
    LM_Sensor_Data.in_max[1] = lm78_read_value(LM78_IN1_MAX);
    LM_Sensor_Data.in_min[1] = lm78_read_value(LM78_IN1_MIN);
  }

  if (LM_Sensor_Present.in & 0x04) {
    LM_Sensor_Data.in[2] = lm78_read_value(LM78_IN2);
    LM_Sensor_Data.in_max[2] = lm78_read_value(LM78_IN2_MAX);
    LM_Sensor_Data.in_min[2] = lm78_read_value(LM78_IN2_MIN);
  }

  if (LM_Sensor_Present.in & 0x08) {
    LM_Sensor_Data.in[3] = lm78_read_value(LM78_IN3);
    LM_Sensor_Data.in_max[3] = lm78_read_value(LM78_IN3_MAX);
    LM_Sensor_Data.in_min[3] = lm78_read_value(LM78_IN3_MIN);
  }

  if (LM_Sensor_Present.in & 0x10) {
    LM_Sensor_Data.in[4] = lm78_read_value(LM78_IN4);
    LM_Sensor_Data.in_max[4] = lm78_read_value(LM78_IN4_MAX);
    LM_Sensor_Data.in_min[4] = lm78_read_value(LM78_IN4_MIN);
  }

  if (LM_Sensor_Present.in & 0x20) {
    LM_Sensor_Data.in[5] = lm78_read_value(LM78_IN5);
    LM_Sensor_Data.in_max[5] = lm78_read_value(LM78_IN5_MAX);
    LM_Sensor_Data.in_min[5] = lm78_read_value(LM78_IN5_MIN);
  }

  if (LM_Sensor_Present.in & 0x40) {
    LM_Sensor_Data.in[6] = lm78_read_value(LM78_IN6);
    LM_Sensor_Data.in_max[6] = lm78_read_value(LM78_IN6_MAX);
    LM_Sensor_Data.in_min[6] = lm78_read_value(LM78_IN6_MIN);
  }

  if (LM_Sensor_Present.vid) 
    if (! LM_Sensor_Data.valid)
      read_vid_lines();

  if (LM_Sensor_Present.temp_mb) {
    LM_Sensor_Data.temp_mb = lm78_read_value(LM78_TEMPERATURE);
    LM_Sensor_Data.temp_mb_max = lm78_read_value(LM78_TEMPERATURE_MAX);
    LM_Sensor_Data.temp_mb_min = lm78_read_value(LM78_TEMPERATURE_MIN);
  }

  if (LM_Sensor_Present.fan & 0x01) {
    LM_Sensor_Data.fan[0] = lm78_read_value(LM78_FAN1);
    LM_Sensor_Data.fan_min[0] = lm78_read_value(LM78_FAN1_MIN);
  }
  if (LM_Sensor_Present.fan & 0x02) {
    LM_Sensor_Data.fan[1] = lm78_read_value(LM78_FAN2);
    LM_Sensor_Data.fan_min[1] = lm78_read_value(LM78_FAN2_MIN);
  }
  if (LM_Sensor_Present.fan & 0x04) {
    LM_Sensor_Data.fan[2] = lm78_read_value(LM78_FAN3);
    LM_Sensor_Data.fan_min[2] = lm78_read_value(LM78_FAN3_MIN);
  }

  LM_Sensor_Data.fan_div[0] = lm78_read_value(LM78_FANDIV);
  if (LM_Type == LMTYPE_WINB)
    LM_Sensor_Data.fan_div[1] = lm78_read_value(LM78_WINB_PIN);

  LM_Sensor_Data.interrupt_status[0] = lm78_read_value(LM78_INTSTAT1);
  LM_Sensor_Data.interrupt_status[1] = lm78_read_value(LM78_INTSTAT2);

}

  /* TODO: release Spin lock on LM_Sensor_Data */
}

/* This function will set a limit to a new value. It will scale it as
   needed. See lm78.h for the possible values of what and the scaling
   which is expected. */
void LM78_Set(int what, int nr, long value)
{
  u8 port;
  long write;

  switch(what) {
  case LM_PROBE_IN_MIN:
  case LM_PROBE_IN_MAX:
    switch(nr) {
    case 0:
      write=((value * 100000 / LM_Sensor_Config.in_conv[0]) + 8) / 16;
      port=(what==LM_PROBE_IN_MIN?
             (LM_Type==LMTYPE_80?LM80_IN0_MIN:LM78_IN0_MIN):
             (LM_Type==LMTYPE_80?LM80_IN0_MAX:LM78_IN0_MAX));
      break;
    case 1:
      write=((value * 100000 / LM_Sensor_Config.in_conv[1]) + 8) / 16;
      port=(what==LM_PROBE_IN_MIN?
             (LM_Type==LMTYPE_80?LM80_IN1_MIN:LM78_IN1_MIN):
             (LM_Type==LMTYPE_80?LM80_IN1_MAX:LM78_IN1_MAX));
      break;
    case 2:
      write=((value * 100000 / LM_Sensor_Config.in_conv[2]) + 8) / 16;
      port=(what==LM_PROBE_IN_MIN?
             (LM_Type==LMTYPE_80?LM80_IN2_MIN:LM78_IN2_MIN):
             (LM_Type==LMTYPE_80?LM80_IN2_MAX:LM78_IN2_MAX));
      break;
    case 3:
      write=((value * 100000 / LM_Sensor_Config.in_conv[3]) + 8) / 16;
      port=(what==LM_PROBE_IN_MIN?
             (LM_Type==LMTYPE_80?LM80_IN3_MIN:LM78_IN3_MIN):
             (LM_Type==LMTYPE_80?LM80_IN3_MAX:LM78_IN3_MAX));
      break;
    case 4:
      write=((value * 100000 / LM_Sensor_Config.in_conv[4]) + 8) / 16;
      port=(what==LM_PROBE_IN_MIN?
             (LM_Type==LMTYPE_80?LM80_IN4_MIN:LM78_IN4_MIN):
             (LM_Type==LMTYPE_80?LM80_IN4_MAX:LM78_IN4_MAX));
      break;
    case 5:
      write=((value * 100000 / LM_Sensor_Config.in_conv[5]) + 8) / 16;
      port=(what==LM_PROBE_IN_MIN?
             (LM_Type==LMTYPE_80?LM80_IN5_MIN:LM78_IN5_MIN):
             (LM_Type==LMTYPE_80?LM80_IN5_MAX:LM78_IN5_MAX));
      break;
    case 6:
      write=((value * 100000 / LM_Sensor_Config.in_conv[6]) + 8) / 16;
      port=(what==LM_PROBE_IN_MIN?
             (LM_Type==LMTYPE_80?LM80_IN6_MIN:LM78_IN6_MIN):
             (LM_Type==LMTYPE_80?LM80_IN6_MAX:LM78_IN6_MAX));
      break;
    default:
      return;
    }
    if ((write < 0) || (write > 255)) 
      return;
    lm78_write_value(port,write);
    if (what == LM_PROBE_IN_MIN)
      LM_Sensor_Data.in_min[nr] = write;
    else
      LM_Sensor_Data.in_max[nr] = write;
    break;
  case LM_PROBE_FAN_MIN:
    switch(nr) {
    case 1:
      port=LM_Type==LMTYPE_80?LM80_FAN1_MIN:LM78_FAN1_MIN;
      break;
    case 2:
      port=LM_Type==LMTYPE_80?LM80_FAN2_MIN:LM78_FAN2_MIN;
      break;
    case 3:
      if (LM_Type == LMTYPE_80)
        return; /* Not present on LM80 */
      else
        port=LM78_FAN3_MIN;
      break;
    default:
      return;
    }
    write=LM78_FAN_COUNT(value,2);
    if (write < 0 || write > 255) 
      return; /* value out of range */
    lm78_write_value(port,write);
    LM_Sensor_Data.fan_min[nr-1]=write;
    break;
  case LM_PROBE_FAN_DIV:
    /* This is somewhat tricky. We actually need to read a register here
       to modify it. Using LM_Sensor_Data.fan can't be done, as it need 
       not be valid when we come here. But we do need to put the results
       there. Hmm. */
 
    if (LM_Type == LMTYPE_80) {
      if (nr == 1) {
        write = (lm78_read_value(LM80_FANDIV) & 0xf3) | 
                (fan_encode(value) << 2);
        lm78_write_value(LM80_FANDIV,write);
        LM_Sensor_Data.fan_div[0] = write;
      } else if (nr == 2) {
        write = (lm78_read_value(LM80_FANDIV) & 0xcf) | 
                (fan_encode(value) << 4);
        lm78_write_value(LM80_FANDIV,write);
        LM_Sensor_Data.fan_div[0] = write;
      }
    } else {
      if (nr == 1) {
        write = (lm78_read_value(LM78_FANDIV) & 0xcf) |
                (fan_encode(value) << 4);
        lm78_write_value(LM78_FANDIV,write);
        LM_Sensor_Data.fan_div[0] = write;
      } else if (nr == 2) {
        write = (lm78_read_value(LM78_FANDIV) & 0x3f) |
                (fan_encode(value) << 6);
        lm78_write_value(LM78_FANDIV,write);
        LM_Sensor_Data.fan_div[0] = write;
      } else if ((nr == 3) && (LM_Type == LMTYPE_WINB)) {
        write = (lm78_read_value(LM78_WINB_PIN) & 0x3f) |
                (fan_encode(value) << 6);
        lm78_write_value(LM78_WINB_PIN,write);
        LM_Sensor_Data.fan_div[1] = write;
      }
    }
    break;
  case LM_PROBE_TEMP_MIN:
  case LM_PROBE_TEMP_MAX:
    port=(what==LM_PROBE_TEMP_MIN?
            (LM_Type==LMTYPE_80?LM80_TEMPERATURE_MIN:LM78_TEMPERATURE_MIN):
            (LM_Type==LMTYPE_80?LM80_TEMPERATURE_MAX:LM78_TEMPERATURE_MAX));
    if ((value >= 0) && (value <= 125))
      write=value;
    else if (value >= -55)
      write=value & 0xff;
    else /* value out of range */
      return;
    lm78_write_value(port,write);
    if (what == LM_PROBE_TEMP_MIN) 
      LM_Sensor_Data.temp_mb_min=write;
    else
      LM_Sensor_Data.temp_mb_max=write;
    break;
  default: /* what out of range */
      return;
  }
}      
       
/* We return 0 if what or nr are out-of-bounds. Note that you must call
   LM_Update_Data before you access this function! */
long LM78_Get(int what, int nr)
{
  int mult;
  int res;

  switch(what) {
  case LM_PROBE_IN:
  case LM_PROBE_IN_MIN:
  case LM_PROBE_IN_MAX:
    if ((nr < 0) || (nr > 6))
      return 0;
    if (what == LM_PROBE_IN)
      return LM_Sensor_Data.in[nr] * 16 * 
             LM_Sensor_Config.in_conv[nr] / 100000;
    else if (what == LM_PROBE_IN_MIN)
      return LM_Sensor_Data.in_min[nr] * 16 * 
             LM_Sensor_Config.in_conv[nr] / 100000;
    else
      return LM_Sensor_Data.in_max[nr] * 16 * 
             LM_Sensor_Config.in_conv[nr] / 100000;
  case LM_PROBE_FAN:
  case LM_PROBE_FAN_MIN:
    if ((nr >= 1) && (nr <= 3)) {
      if (what == LM_PROBE_FAN)
        return LM78_RPM(LM_Sensor_Data.fan[nr-1],2);
      else
        return LM78_RPM(LM_Sensor_Data.fan_min[nr-1],2);
    } else
      return 0;
  case LM_PROBE_FAN_DIV:
    if (LM_Type == LMTYPE_80) {
      if (nr == 1)
        return fan_decode((LM_Sensor_Data.fan_div[0] >> 2) & 0x03);
      else if (nr == 2) 
        return fan_decode((LM_Sensor_Data.fan_div[0] >> 4) & 0x03);
      else
        return 0;
    } else {
      if (nr == 3) 
        return (LM_Type==LMTYPE_WINB)?
                fan_decode((LM_Sensor_Data.fan_div[1] >> 6) & 0x03):2;
      else if (nr == 2)
        return fan_decode((LM_Sensor_Data.fan_div[0] >> 6) & 0x03);
      else if (nr == 1)
        return fan_decode((LM_Sensor_Data.fan_div[0] >> 4) & 0x03);
      else
        return 0;
    }
  case LM_PROBE_TEMP:
  case LM_PROBE_TEMP_MIN:
  case LM_PROBE_TEMP_MAX:
    if (what == LM_PROBE_TEMP)
      mult = LM_Sensor_Data.temp_mb;
    else if (what == LM_PROBE_TEMP_MIN)
      mult = LM_Sensor_Data.temp_mb_min;
    else 
      mult = LM_Sensor_Data.temp_mb_max;
    if (mult > 0x80)
      return mult-256;
    else
      return mult;
  case LM_PROBE_VID: /* Returns a negative value if the reading is suspect;
                        Returns 0 if no known reading found. */
    mult = LM_Sensor_Data.vid;
    if (mult <= 0x05)
      res = 205 - 5 * mult;
    else if (mult <= 0x0f)
      return 0;
    else if (mult <= 0x1e)
      res = 510 - mult * 10;
    else
      return 0;
    if ((mult & 0x1f) == 0x10) /* maybe unconnected */
      return -res;
    else
      return res;
  case LM_PROBE_ALARM:
    /* Pack the bits in a more logical way. This is the lm78 code, the
       lm80 code is quite somewhat different. 
       Format: bit 0..6 = IN0..6
               bit 7..9 = FAN1..3
               bit 10   = TEMP
               bit 11   = LM75
               bit 12   = Chassis intrusion
               bit 13   = FIFO
               bit 14   = SMI
               bit 15   = INT_IN
       See also lm78.h */
    if (LM_Type == LMTYPE_80) 
      return ((LM_Sensor_Data.interrupt_status[0] & 0x7f) |       /* IN0-6 */
             ((LM_Sensor_Data.interrupt_status[1] & 0x0c) << 5) | /* FAN1-2 */
             ((LM_Sensor_Data.interrupt_status[1] & 0x03) << 10)| /* TEMP,
                                                                     lm75 */
             ((LM_Sensor_Data.interrupt_status[1] & 0x10) << 8) | /* Chas */
             ((LM_Sensor_Data.interrupt_status[0] & 0x80) << 7)); /* INT_IN */
    else 
      return ((LM_Sensor_Data.interrupt_status[0] & 0x0f) |       /* IN0-3 */ 
             ((LM_Sensor_Data.interrupt_status[1] & 0x07) << 4) | /* IN4-6 */
             ((LM_Sensor_Data.interrupt_status[0] & 0xc0) << 1) | /* FAN1-2 */
             ((LM_Sensor_Data.interrupt_status[1] & 0x08) << 6) | /* FAN3 */
             ((LM_Sensor_Data.interrupt_status[0] & 0x30) << 6) | /* TEMP,
                                                                     LM75 */
             ((LM_Sensor_Data.interrupt_status[1] & 0x70) << 8)) & /* Chas, 
                                                                      FIFO,
                                                                      SMI */
             (LM_Type==LMTYPE_WINB?0x1fff:0x7fff);  /* WINB has no FIFO,SMI */
  default: /* what invalid */
    return 0;
  }
}
    
/* Input: 1, 2, 4 or 8. Output: this value encoded in LM78 format. */
u8 fan_encode(int div)
{
  if (div == 1)
    return 0x00;
  else if (div == 4)
    return 0x02;
  else if (div == 8)
    return 0x03;
  else /* div == 2 or strange input */
    return 0x01;
}

int fan_decode(u8 div)
{
  if (div == 0x00)
    return 1;
  else if (div == 0x01)
    return 2;
  else if (div == 0x02)
    return 4;
  else if (div == 0x03)
    return 8;
  else
    return 0;
}
