
/*
    compat.h - Part of a Linux module for reading sensor data.
    Copyright (c) 1997, 1998  Alexander Larsson <alla@lysator.liu.se>,
    Frodo Looijaard <frodol@dds.nl> and Philip Edelbrock <phil@netroedge.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* This module contains code to make 2.0 kernels more compatible with
   2.1 kernels. Not all incompatibilities can be resolved this way,
   but many #if statements can be removed through these defines. 

   Note that if you have an old 2.1 kernel you may have to redefine
   some of this stuff; it is often not clear when a feature or change
   was introduced. 
*/

#ifndef LM_COMPAT_H
#define LM_COMPAT_H


/* This useful macro is not defined in the 2.0 kernels */
#ifndef KERNEL_VERSION
#define KERNEL_VERSION(a,b,c) (((a) << 16) | ((b) << 8) | (c))
#endif


/* pci_present is not defined for 2.0 kernels; perhaps some early 2.1
   kernels do not have it either? */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,1,0))
#define pci_present pcibios_present
#endif


/* mdelay is not defined for 2.0 kernels; perhaps some early 2.1 kernels
   do no have it either? */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,1,0))
#define MAX_UDELAY_MS   5
#define mdelay(n) (\
        (__builtin_constant_p(n) && (n)<=MAX_UDELAY_MS) ? udelay((n)*1000) : \
        ({unsigned long msec=(n); while (msec--) udelay(1000);}))
#endif


/* copy_from/to_usr is called memcpy_from/to_fs in 2.0 kernels; perhaps in
   some early 2.1 kernels too? */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,1,0))
#define copy_from_user memcpy_fromfs
#define copy_to_user memcpy_tofs
#endif


/* get_user was redefined in 2.1 kernels to use two arguments, and returns
   an error code */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,1,0))
#define get_user_data(to,from) ((to) = get_user(from),0)
#else
#define get_user_data(to,from) get_user(to,from)
#endif

/* Add a scheduling fix for the new code in kernel 2.1.127 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,1,127))
#define schedule_timeout(x) ( current->timeout = jiffies + (x), schedule() )
#endif


/* Now follow some module settings. */
/* NB: You must include <linux/module.h> before this works - we can't do
   it ourselves, because some things must first be defined. */

#ifndef MODULE_AUTHOR
#define MODULE_AUTHOR(name)
#endif

#ifndef MODULE_DESCRIPTION
#define MODULE_DESCRIPTION(desc)
#endif

#ifndef MODULE_SUPPORTED_DEVICE
#define MODULE_SUPPORTED_DEVICE(dev)
#endif

#ifndef MODULE_PARM
#define MODULE_PARM(var,type)
#endif

#ifndef MODULE_PARM_DESC
#define MODULE_PARM_DESC(var,desc)
#endif

/* PCI interface stuff */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,1,0))
#define pci_present pcibios_present
#define pci_read_config_byte_united(device,bus,dev,com,res) \
                            pcibios_read_config_byte(bus,dev,com,res);
#define pci_read_config_word_united(device,bus,dev,com,res) \
                            pcibios_read_config_word(bus,dev,com,res);
#else
#define pci_read_config_byte_united(device,bus,dev,com,res) \
                            pci_read_config_byte(device,com,res);
#define pci_read_config_word_united(device,bus,dev,com,res) \
                            pci_read_config_word(device,com,res);
#endif

#endif /* LM_COMPAT_H */
