///////////////////////////////////////////////////////////////////////
//
// Temperature Monitor v1.1.0
// by Laurent Berger (berger@essi.fr) - March / May 1999
// 
// This software is licensed through the GNU General Public License.
//
///////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

/////////////////////////////////////////
// The XPM attributes
/////////////////////////////////////////

#include "wmsensors.xpm"

char wmSensorsMaskBits [128 * 64];

int  wmSensorsMaskWidth  = 128;
int  wmSensorsMaskHeight = 64;

/////////////////////////////////////////
// Constantes
/////////////////////////////////////////

#define WMSENSORS_VERSION "1.1.0"

#define MAX_MOUSE_REGION (4)

#define CHAR_WIDTH  6
#define CHAR_HEIGHT 7

/////////////////////////////////////////
// Structs
/////////////////////////////////////////

typedef struct 
{
  int nEnable;
  int nTop;
  int nBottom;
  int nLeft;
  int nRight;
} MOUSE_REGION;

typedef struct 
{
  Pixmap        xPixmap;
  Pixmap        xMask;
  XpmAttributes xAttributes;
} XpmIcon;

/////////////////////////////////////////
// Global variables
/////////////////////////////////////////

MOUSE_REGION MouseRegion [MAX_MOUSE_REGION];

int bCPU      = 1,
    bHasCPU   = 0,
    bCPUs     = 0,
    bShowMB   = 1,
    bNC       = 0,
    bCelsius  = 1,
    bVersion2 = 0;

int nCurrentState = 0,
    nSec          = 1,
    nCPU,
    nMaskCPU,
    nScreen,
    nDisplayDepth,
    nFD;

char szSensorsPath [256] = "/proc/sys/dev/lm_sensors/";

char *DisplayName = NULL,
     *szGeometry  = NULL;

Display   *sDisplay;
Window     Root,
           IconWindow,
           Win;
GC         NormalGC;
XpmIcon    wmGen;
Pixmap     Pixmask;
Pixel      BackPixel, 
           ForePixel;
XSizeHints xSizeHints;
XWMHints   wmHints;

///////////////////////////////////////////////////////////////////////
//
// GetXPM
//
///////////////////////////////////////////////////////////////////////

void GetXPM (XpmIcon *wmGen, char *PixmapBytes []) 
{
  XWindowAttributes xAttributes;

  // For the colormap
  XGetWindowAttributes (sDisplay, Root, &xAttributes);

  wmGen->xAttributes.valuemask |= (XpmReturnPixels | XpmReturnExtensions);

  if (XpmCreatePixmapFromData (sDisplay, Root, PixmapBytes, &(wmGen->xPixmap), &(wmGen->xMask), &(wmGen->xAttributes)) != XpmSuccess)
  {
    fprintf (stderr, "Not enough free colorcells.\n");
    exit (1);
  }
}

///////////////////////////////////////////////////////////////////////
//
// GetColor
//
///////////////////////////////////////////////////////////////////////

Pixel GetColor (char *szName) 
{
  XColor            xColor;
  XWindowAttributes xAttributes;

  XGetWindowAttributes (sDisplay, Root, &xAttributes);

  xColor.pixel = 0;

  if (!XParseColor (sDisplay, xAttributes.colormap, szName, &xColor)) 
    fprintf (stderr, "wmsensors: can't parse %s.\n", szName);
  else if (!XAllocColor (sDisplay, xAttributes.colormap, &xColor)) 
    fprintf (stderr, "wmsensors: can't allocate %s.\n", szName);
  
  return xColor.pixel;
}

///////////////////////////////////////////////////////////////////////
//
// FlushExpose
//
///////////////////////////////////////////////////////////////////////

int FlushExpose (Window w) 
{
  XEvent Dummy;
  int    i = 0;

  while (XCheckTypedWindowEvent (sDisplay, w, Expose, &Dummy))
    i++;

  return i;
}

///////////////////////////////////////////////////////////////////////
//
// openXwindow
//
///////////////////////////////////////////////////////////////////////

void openXwindow (int argc, char *argv[], char *PixmapBytes [], char *PixmaskBits, int PixmaskWidth, int PixmaskHeight) 
{
  unsigned int   BorderWidth  = 1;
  char          *wName        = argv [0];
  unsigned long	 gcm;
  int            Dummy        = 0,
                 wx, wy;

  XClassHint     ClassHint;
  XTextProperty  Name;
  XGCValues      xGCValues;
  
  if (!(sDisplay = XOpenDisplay (DisplayName))) 
  {
    fprintf (stderr, "%s: can't open display %s\n", wName, XDisplayName (DisplayName));
    exit (1);
  }

  nScreen       = DefaultScreen (sDisplay);
  Root          = RootWindow (sDisplay, nScreen);
  nDisplayDepth = DefaultDepth (sDisplay, nScreen);
  nFD           = XConnectionNumber (sDisplay);
  
  ///////////////////////////////////////
  // Convert XPM to XImage 
  GetXPM (&wmGen, PixmapBytes);
  
  ///////////////////////////////////////
  // Create a window to hold the stuff 
  xSizeHints.flags = USSize | USPosition;
  xSizeHints.x     = 0;
  xSizeHints.y     = 0;
  
  BackPixel = GetColor ("white");
  ForePixel = GetColor ("black");
  
  XWMGeometry (sDisplay, nScreen, szGeometry, NULL, BorderWidth, &xSizeHints, &xSizeHints.x, &xSizeHints.y, &xSizeHints.width,
	       &xSizeHints.height, &Dummy);
  
  xSizeHints.width  = 64;
  xSizeHints.height = 64;
  
  Win = XCreateSimpleWindow (sDisplay, Root, xSizeHints.x, xSizeHints.y, xSizeHints.width, xSizeHints.height, BorderWidth, ForePixel,
			     BackPixel);
  
  IconWindow = XCreateSimpleWindow (sDisplay, Win, xSizeHints.x, xSizeHints.y, xSizeHints.width, xSizeHints.height, BorderWidth, 
				 ForePixel, BackPixel);
  
  ///////////////////////////////////////
  // Activate hints 
  XSetWMNormalHints (sDisplay, Win, &xSizeHints);
  ClassHint.res_name  = wName;
  ClassHint.res_class = wName;
  XSetClassHint (sDisplay, Win, &ClassHint);
  
  XSelectInput (sDisplay, Win, ButtonPressMask | ExposureMask | ButtonReleaseMask | PointerMotionMask | StructureNotifyMask);
  XSelectInput (sDisplay, IconWindow, ButtonPressMask | ExposureMask | ButtonReleaseMask | PointerMotionMask | StructureNotifyMask);
  
  if (XStringListToTextProperty (&wName, 1, &Name) == 0) 
  {
    fprintf (stderr, "%s: can't allocate window name\n", wName);
    exit (1);
  }
  
  XSetWMName (sDisplay, Win, &Name);
  
  ///////////////////////////////////////
  // Create GC for drawing 
  gcm                          = GCForeground | GCBackground | GCGraphicsExposures;
  xGCValues.foreground         = ForePixel;
  xGCValues.background         = BackPixel;
  xGCValues.graphics_exposures = 0;
  NormalGC                     = XCreateGC (sDisplay, Root, gcm, &xGCValues);
  
  ///////////////////////////////////////
  // ONLYSHAPE ON 
  Pixmask = XCreateBitmapFromData (sDisplay, Win, PixmaskBits, PixmaskWidth, PixmaskHeight);
  
  XShapeCombineMask (sDisplay, Win, ShapeBounding, 0, 0, Pixmask, ShapeSet);
  XShapeCombineMask (sDisplay, IconWindow, ShapeBounding, 0, 0, Pixmask, ShapeSet);
  
  ///////////////////////////////////////
  // ONLYSHAPE OFF 
  wmHints.initial_state = WithdrawnState;
  wmHints.icon_window   = IconWindow;
  wmHints.icon_x        = xSizeHints.x;
  wmHints.icon_y        = xSizeHints.y;
  wmHints.window_group  = Win;
  wmHints.flags         = StateHint | IconWindowHint | IconPositionHint | WindowGroupHint;
  
  XSetWMHints (sDisplay, Win, &wmHints);
  
  XSetCommand (sDisplay, Win, argv, argc);
  XMapWindow (sDisplay, Win);
  
  if (szGeometry) 
  {
    if (sscanf (szGeometry, "+%d+%d", &wx, &wy) != 2) 
    {
      fprintf (stderr, "Bad geometry string.\n");
      exit (1);
    }

    XMoveWindow (sDisplay, Win, wx, wy);
  }
}

///////////////////////////////////////////////////////////////////////
//
// createXBMfromXPM
//
///////////////////////////////////////////////////////////////////////

void createXBMfromXPM (char *xbm, char **xpm, int sx, int sy) 
{
  int           nIndex, 
                nJndex,
                nWidth, 
                nHeight, 
                nNumCol,
                bCount;
  char          cZero;
  unsigned char	bWrite;
  
  sscanf (*xpm, "%d %d %d", &nWidth, &nHeight, &nNumCol);
  
  cZero = xpm [1][0];

  for (nIndex = nNumCol + 1; nIndex < nNumCol + sy + 1; nIndex++) 
  {
    bCount = 0;
    bWrite = 0;
    
    for (nJndex = 0; nJndex < sx; nJndex++) 
    {
      bWrite >>= 1;

      if (xpm [nIndex][nJndex] != cZero) 
	bWrite += 128;

      bCount++;

      if (bCount == 8) 
      {
	*xbm = bWrite;
	
	xbm++;

	bCount = 0;
	bWrite = 0;
      }
    }
  }
}

///////////////////////////////////////////////////////////////////////
//
// RedrawWindow
//
///////////////////////////////////////////////////////////////////////

void RedrawWindow (int x, int y) 
{
  FlushExpose (IconWindow);
  XCopyArea (sDisplay, wmGen.xPixmap, IconWindow, NormalGC, x,y, wmGen.xAttributes.width, wmGen.xAttributes.height, 0, 0);

  FlushExpose (Win);
  XCopyArea (sDisplay, wmGen.xPixmap, Win, NormalGC, x,y, wmGen.xAttributes.width, wmGen.xAttributes.height, 0, 0);
}

///////////////////////////////////////////////////////////////////////
//
// copyXPMArea
//
///////////////////////////////////////////////////////////////////////

void copyXPMArea (int x, int y, int sx, int sy, int dx, int dy) 
{
  XCopyArea (sDisplay, wmGen.xPixmap, wmGen.xPixmap, NormalGC, x, y, sx, sy, dx, dy);
}

///////////////////////////////////////////////////////////////////////
//
// setMaskXY
//
///////////////////////////////////////////////////////////////////////

void setMaskXY (int x, int y) 
{
  XShapeCombineMask (sDisplay, Win, ShapeBounding, x, y, Pixmask, ShapeSet);
  XShapeCombineMask (sDisplay, IconWindow, ShapeBounding, x, y, Pixmask, ShapeSet);
}

///////////////////////////////////////////////////////////////////////
//
// AddMouseRegion
//
///////////////////////////////////////////////////////////////////////

void AddMouseRegion (int nIndex, int nLeft, int nTop, int nRight, int nBottom) 
{
  if (nIndex < MAX_MOUSE_REGION) 
  {
    MouseRegion [nIndex].nEnable = 1;
    MouseRegion [nIndex].nTop    = nTop;
    MouseRegion [nIndex].nLeft   = nLeft;
    MouseRegion [nIndex].nBottom = nBottom;
    MouseRegion [nIndex].nRight  = nRight;
  }
}

///////////////////////////////////////////////////////////////////////
//
// CheckMouseRegion
//
///////////////////////////////////////////////////////////////////////

int CheckMouseRegion (int x, int y) 
{
  int nIndex,
      nFound = 0;
  
  for (nIndex = 0; nIndex < MAX_MOUSE_REGION && !nFound; nIndex++) 
    if (MouseRegion [nIndex].nEnable && x <= MouseRegion [nIndex].nRight && x >= MouseRegion [nIndex].nLeft && 
	y <= MouseRegion [nIndex].nBottom && y >= MouseRegion [nIndex].nTop)
      nFound = 1;

  if (!nFound) 
    return -1;

  return (nIndex - 1);
}

///////////////////////////////////////////////////////////////////////
//
// BlitString
//
///////////////////////////////////////////////////////////////////////

void BlitString (char *name, int x, int y)
{
  int nIndex, c;

  for (nIndex = 0; name [nIndex]; nIndex++)
  {
    c = toupper (name [nIndex]); 

    if (c >= 'A' && c <= 'Z')
    {
      copyXPMArea ((c - 'A') * 6, 74, 6, 8, x, y);

      x += 6;
    }
    else if ((c >= '0' && c <= '9') || (c == 32))
    {
      copyXPMArea ((c - '0') * 6, 64, 6, 8, x, y);

      x += 6;
    }
    else 
    {
      switch (c)
      {
	case '/':
	  copyXPMArea (68, 64, 6, 8, x, y);
	  break;

	case '+':
	  copyXPMArea (126, 64, 6, 8, x, y);
	  break;

	case '-':
	  copyXPMArea (132, 64, 6, 8, x, y);
	  break;

	case '.':
	  copyXPMArea (120, 64, 6, 8, x, y);
	  break;
      }

      x += 6;
    }
  }
}

///////////////////////////////////////////////////////////////////////
//
// DrawCPU
//
///////////////////////////////////////////////////////////////////////

void DrawCPU (void)
{
  FILE *fp;
  int   nTemp, nDummy1, nDummy2, nDummy3, nDummy4, nDummy5;
  char  szText [8];

  if (bCPU)
  {
    char szFile [256];

    sprintf (szFile, "%stemp-%d", szSensorsPath, nCPU);

    fp = fopen (szFile, "r");

    if (!fp)
    {
      bCPU = 0;

      if (bShowMB)
      {
	copyXPMArea (0, 84, (4 * CHAR_WIDTH), 8, 2, 2);
	BlitString ("mb  ", 2, 2);

	copyXPMArea (0, 84, (4 * CHAR_WIDTH), 8, 66, 2);
	BlitString ("mb  ", 66, 2);
      }
	
      sprintf (szFile, "%stemp", szSensorsPath);

      fp = fopen (szFile, "r");
	  
      if (!fp)
      {
        sprintf (szFile, "%stemp-mb", szSensorsPath); 
	fp = fopen (szFile, "r");
      }
    }
  }
  else
  {
    char szFile [256];
    
    sprintf (szFile, "%stemp", szSensorsPath);

    fp = fopen (szFile, "r");

    if (!fp)
    {
      sprintf (szFile, "%stemp-mb", szSensorsPath); 
      fp = fopen (szFile, "r");
    }
  }

  if (fp)
  {
    if (bVersion2)
      fscanf (fp, "%d.%d %d.%d %d.%d", &nDummy1, &nDummy2, &nDummy3, &nDummy4, &nTemp, &nDummy5);
    else
      fscanf (fp, "%d %d %d", &nDummy1, &nDummy2, &nTemp);

    fclose (fp);
      
    if (!bCelsius)
    {
      nTemp = 32 + (int)((9.0 / 5.0) * (float)(nTemp));

      if (nTemp > 99)
	sprintf (szText, "%dF", nTemp);
      else
	sprintf (szText, "%d F", nTemp);
    }
    else
      if (nTemp > 99)
	sprintf (szText, "%dC", nTemp);
      else
	sprintf (szText, "%d C", nTemp);
  }
  else
    strcpy (szText, "NA");

  copyXPMArea (0, 84, (4 * CHAR_WIDTH), 8, 30, 2);
  BlitString (szText, 54 - (strlen (szText) * CHAR_WIDTH), 2);

  copyXPMArea (0, 84, (4 * CHAR_WIDTH), 8, 94, 2);
  BlitString (szText, 118 - (strlen (szText) * CHAR_WIDTH), 2);
}

///////////////////////////////////////////////////////////////////////
//
// DrawFANs
//
///////////////////////////////////////////////////////////////////////

void DrawFANs (int bNC)
{
  FILE *fp;
  char  szText [256];
  int   nDummy, nFAN;

  sprintf (szText, "%sfan1", szSensorsPath);

  fp = fopen (szText, "r");
  
  if (fp)
  {
    fscanf (fp, "%d %d", &nDummy, &nFAN);
    fclose (fp);
    
    sprintf (szText, "%d", nFAN);
  
    if (!nFAN && bNC)
      strcpy (szText, "NC");
  }
  else
    strcpy (szText, "NA");
  
  copyXPMArea (0, 84, (4 * CHAR_WIDTH), 8, 30, 20);
  BlitString (szText, 54 - (strlen (szText) * CHAR_WIDTH), 20);

  sprintf (szText, "%sfan2", szSensorsPath);

  fp = fopen (szText, "r");
  
  if (fp)
  {
    fscanf (fp, "%d %d", &nDummy, &nFAN);
    fclose (fp);

    sprintf (szText, "%d", nFAN);
    
    if (!nFAN && bNC)
      strcpy (szText, "NC");
  }
  else
    strcpy (szText, "NA");
    
  copyXPMArea (0, 84, (4 * CHAR_WIDTH), 8, 30, 33);
  BlitString (szText, 54 - (strlen (szText) * CHAR_WIDTH), 33);
  
  sprintf (szText, "%sfan3", szSensorsPath);

  fp = fopen (szText, "r");
  
  if (fp)
  {
    fscanf (fp, "%d %d", &nDummy, &nFAN);
    fclose (fp);
  
    sprintf (szText, "%d", nFAN);
    
    if (!nFAN && bNC)
      strcpy (szText, "NC");
  }
  else
    strcpy (szText, "NA");
  
  copyXPMArea (0, 84, (4 * CHAR_WIDTH), 8, 30, 46);
  BlitString (szText, 54 - (strlen (szText) * CHAR_WIDTH), 46);
}

///////////////////////////////////////////////////////////////////////
//
// DrawVoltage
//
///////////////////////////////////////////////////////////////////////

void DrawVoltage (char cIn, char cType, int bFirst)
{
  FILE *fp;
  char  szText [256];
  int nDummy1, nDummy2, nDummy3, nDummy4, nIntVal, nRealVal, nOffset, nIndex;

  if (bFirst)
    nOffset = 0;
  else
    nOffset = 13;

  copyXPMArea (88, 84, 33, 8, 86, 33 + nOffset);
  copyXPMArea (72, 84, 14, 8, 64, 33 + nOffset);
  
  sprintf (szText, "%sin ", szSensorsPath);
  nIndex = strlen (szText) - 1;

  if (cIn != ' ')
  {
    if (cIn != '-')
      szText [nIndex] = cIn;
    else
      strcpy (&(szText [nIndex - 2]), "vid");

    fp = fopen (szText, "r");

    if (fp)
    {
      if (cIn != '-')
	fscanf (fp, "%d.%d %d.%d %d.%d", &nDummy1, &nDummy2, &nDummy3, &nDummy4, &nIntVal, &nRealVal);
      else
	fscanf (fp, "%d.%d", &nIntVal, &nRealVal);

      fclose (fp);

      if (bVersion2)
      {
        float fVal = (float)nIntVal + ((float)nRealVal / 100.0);
        int nVal, nNeg = 1;

        switch (cIn)
        {
          case '3' :
            fVal = fVal * 1.68;
            break;

          case '4' :
            fVal = fVal * 3.8;
            break;

          case '5' :
            fVal = fVal * 3.477;
            nNeg = -1;
            break;

          case '6' :
            fVal = fVal * 1.505;
            nNeg = -1;
            break;
        }

        fVal *= 100.0;

        nVal = (int)fVal;

        if ((fVal - (float)nVal) > 0.5)
        {
          fVal += 0.5;

          nVal = (int)fVal;
        }

        if (nVal < 100)
        {
          nIntVal = 0;
          nRealVal = nVal;
        }
        else
        {
          nIntVal = nVal / 100;
          nRealVal = nVal - (nIntVal * 100);
        }  

        nIntVal *= nNeg;
      }

      sprintf (szText, "%02d", nRealVal);
    
      BlitString (szText, 106, 33 + nOffset);
      
      sprintf (szText, "%+d", nIntVal);
      
      BlitString (szText, 104 - (strlen (szText) * CHAR_WIDTH), 33 + nOffset);
    }
    else
    {
      strcpy (szText, " ");
    
      BlitString (szText, 106, 33 + nOffset);

      strcpy (szText, "NA ");
    
      BlitString (szText, 104 - (strlen (szText) * CHAR_WIDTH), 33 + nOffset);
    }

    szText [0] = cType;
    szText [1] = '\0';

    BlitString (szText, 68, 33 + nOffset);
  }
}

///////////////////////////////////////////////////////////////////////
//
// DrawValues
//
///////////////////////////////////////////////////////////////////////

void DrawValues ()
{
  DrawCPU ();

  switch (nCurrentState)
  {
    case 0:
      DrawFANs (bNC);
      break;
      
    case 1:
      DrawVoltage ('3', '+', 1);
      DrawVoltage ('6', '-', 0);
      break;
      
    case 2:
      DrawVoltage ('4', '+', 1);
      DrawVoltage ('5', '-', 0);
      break;
      
    case 3:
      DrawVoltage ('0', '1', 1);
      DrawVoltage ('1', '2', 0);
      break;
      
    case 4:
      DrawVoltage ('2', ' ', 1);
      DrawVoltage (' ', ' ', 0);
      break;
      
    case 5:
      DrawVoltage ('-', ' ', 1);
      DrawVoltage (' ', ' ', 0);
      break;
  }
  
  RedrawWindow (nCurrentState ? 64 : 0, 0);
}

///////////////////////////////////////////////////////////////////////
//
// wmSensors
//
///////////////////////////////////////////////////////////////////////

void wmSensors (int argc, char **argv)
{
  XEvent Event;
  int    nReleaseStatus,
         nButtonStatus = -1,
         nCount,
         nMask = 1;

  int    xpmX = 0, 
         xpmY = 0;

  {
    FILE *fp;

    fp = fopen (szSensorsPath, "r");

    if (!fp)
      fprintf (stderr, "\nAlexander Larsson's kernel module not found. Please load one...\n");
    else
      fclose (fp);
  }

  for (nCount = 1; nCount < 8; nCount++)
  {
    FILE *fp;
    char  szFile [256];

    sprintf (szFile, "%stemp-%d", szSensorsPath, nCount);

    fp = fopen (szFile, "r");

    if (fp)
    {
      fclose (fp);

      if (bHasCPU)
	bCPUs = 1;
      else
      {
	nCPU = nCount;
	nMaskCPU = nMask;
      }

      bHasCPU += nMask;
    }

    nMask <<= 1;
  }

  nCount = 0;

  createXBMfromXPM (wmSensorsMaskBits, wmsensors_xpm, wmSensorsMaskWidth, wmSensorsMaskHeight);
  
  openXwindow (argc, argv, wmsensors_xpm, wmSensorsMaskBits, wmSensorsMaskWidth, wmSensorsMaskHeight);

  ///////////////////////////////////////
  // set up labels
  ///////////////////////////////////////

 {
   char szText [8];

   if (bCPUs)
     sprintf (szText, "cpu%d", nCPU);
   else
     strcpy (szText, "cpu ");
   
   BlitString (szText, 2, 2);
   BlitString (szText, 66, 2);
 }

  BlitString ("fan1", 2, 20);
  BlitString ("fan2", 2, 33);
  BlitString ("fan3", 2, 46);

  ///////////////////////////////////////
  // adds clickable mouse regions
  ///////////////////////////////////////

  AddMouseRegion (0, 30, 3, 54, 13);
  AddMouseRegion (1, 2, 3, 26, 13);
  AddMouseRegion (2, 2, 19, 54, 56);
  
  ///////////////////////////////////////
  // prepare the GC
  ///////////////////////////////////////

  copyXPMArea (39, 84, (4 * CHAR_WIDTH), 8, 30, 2);
  copyXPMArea (39, 84, (4 * CHAR_WIDTH), 8, 30, 20);
  copyXPMArea (39, 84, (4 * CHAR_WIDTH), 8, 30, 33);
  copyXPMArea (39, 84, (4 * CHAR_WIDTH), 8, 30, 46);
  
  ///////////////////////////////////////
  // default values
  ///////////////////////////////////////

  BlitString ("0 C", 36, 2);
  BlitString ("0 C", 100, 2);

  BlitString ("0", 48, 20);
  BlitString ("0", 48, 33);
  BlitString ("0", 48, 46);

  ///////////////////////////////////////
  // display the values
  ///////////////////////////////////////
  
  if (nCurrentState) 
  {
    xpmX = 64;
    setMaskXY (-64, 0);
  }
  else 
  {
    xpmX = 0;
    setMaskXY (0, 0);
  }

  copyXPMArea (27, 84, (7 * CHAR_WIDTH), 8, 66, 20);

  switch (nCurrentState)
  {
    case 1:
      BlitString ("+/- 5V", 66, 20);
      break;
      
    case 2:
      BlitString ("+/- 12V", 66, 20);
      break;
      
    case 3:
      BlitString ("VCORE", 66, 20);
      break;
      
    case 4:
      BlitString ("+3.3V", 66, 20);
      break;
      
    case 5:
      BlitString ("VID", 66, 20);
      break;
  }

  RedrawWindow (xpmX, xpmY);

  while (1)
  {
    if (!nCount)
      DrawValues ();

    while (XPending (sDisplay))
    {
      XNextEvent (sDisplay, &Event);

      switch (Event.type)
      {
        case Expose:
	  RedrawWindow (xpmX, xpmY);
	  break;

        case DestroyNotify:
	  XCloseDisplay (sDisplay);
	  exit (0);
	  break;
		
        case ButtonPress:
	  nButtonStatus = CheckMouseRegion (Event.xbutton.x, Event.xbutton.y);
	  break;
      
        case ButtonRelease:
	  nReleaseStatus = CheckMouseRegion (Event.xbutton.x, Event.xbutton.y);

	  if (nButtonStatus == nReleaseStatus && nButtonStatus >= 0)
          {
	    switch (nButtonStatus)
            {
	      case 0:
		bCelsius = !bCelsius;
		break;
		
	      case 1:
		if (bHasCPU)
		{
		  char szText [8];

		  if (!bCPU)
		    nMaskCPU = 1;
		  else
		    nMaskCPU <<= 1;

		  while (nMaskCPU && !(bHasCPU & nMaskCPU))
		    nMaskCPU <<= 1;

		  if (!nMaskCPU)
		    bCPU = 0;
		  else
		  {
		    int nTemp = nMaskCPU;

		    nCPU = 0;

		    while (nTemp)
		    {
		      nTemp >>= 1;
		      nCPU++;
		    }

		    bCPU = 1;

		    if (bCPUs)
		      sprintf (szText, "cpu%d", nCPU);
		    else
		      strcpy (szText, "cpu ");
		  }

                  copyXPMArea (0, 84, (4 * CHAR_WIDTH), 8, 2, 2);
		  BlitString (bCPU ? szText : "mb  ", 2, 2);

		  copyXPMArea (0, 84, (4 * CHAR_WIDTH), 8, 66, 2);
		  BlitString (bCPU ? szText : "mb  ", 66, 2);
		}
		break;
		
	      case 2:
		if (++nCurrentState == 6)
		  nCurrentState = 0;
		
		if (nCurrentState) 
		{
		  xpmX = 64;
		  setMaskXY (-64, 0);
		}
		else 
		{
		  xpmX = 0;
		  setMaskXY (0, 0);
		}

		copyXPMArea (27, 84, (7 * CHAR_WIDTH), 8, 66, 20);

		switch (nCurrentState)
		{
  		  case 1:
 		    BlitString ("+/- 5V", 66, 20);
		    break;
		    
  		  case 2:
		    BlitString ("+/- 12V", 66, 20);
		    break;
		    
  		  case 3:
		    BlitString ("VCORE", 66, 20);
		    break;
		    
  		  case 4:
		    BlitString ("+3.3V", 66, 20);
		    break;
		    
  		  case 5:
		    BlitString ("VID", 66, 20);
		    break;
		}
		break;

	    }

	    nButtonStatus = -1;

	    DrawValues ();
	    RedrawWindow (xpmX, xpmY);
	  }
      }
    }
    
    usleep (100000L);
    
    if (++nCount == (10 * nSec))
      nCount = 0;
  }
}

///////////////////////////////////////////////////////////////////////
//
// Usage
//
///////////////////////////////////////////////////////////////////////

void Usage (void)
{
  fprintf (stderr, "\nwmSensors : Temperature and Voltages Monitor\n");
  fprintf (stderr, "Copyright (c) Berger Laurent <freewares.linux@bigfoot.com> - March/May 1999\n\n");
  fprintf (stderr, "Usage :\n");
  fprintf (stderr, "  -display <DisplayName>    set the display to DisplayName\n");
  fprintf (stderr, "  -geometry +XPOS+YPOS      initial window position\n");
  fprintf (stderr, "  -c --cpu                  display the CPU temperature. Default\n");
  fprintf (stderr, "  -m --mainboard            display the mainboard temperature\n");
  fprintf (stderr, "  -l --labelcpu             set the label for the temperature to CPU\n");
  fprintf (stderr, "  -u --update <Delay>       delay between two updates (in seconds).\n");
  fprintf (stderr, "                            Default = 1 sec\n");
  fprintf (stderr, "  -f --fahrenheit           display temperature in Fahrenheit instead of Celsius\n");
  fprintf (stderr, "  -s --state <Screen>       default screen to display :\n");
  fprintf (stderr, "                              1 : FANs\n");
  fprintf (stderr, "                              2 : +/- 5V\n");
  fprintf (stderr, "                              3 : +/- 12V\n");
  fprintf (stderr, "                              4 : Vcore\n");
  fprintf (stderr, "                              5 : +3.3V\n");
  fprintf (stderr, "                              6 : VID\n");
  fprintf (stderr, "  -p --path <Path>          for use with sensors module version 2.0 or above.\n");
  fprintf (stderr, "                            use Path as base directory for sensors files\n");
  fprintf (stderr, "  -h --help                 this help screen\n");
  fprintf (stderr, "  -v --version              print the version number\n");
  fprintf (stderr, "\n");
}

///////////////////////////////////////////////////////////////////////
//
// PrintVersion
//
///////////////////////////////////////////////////////////////////////

void PrintVersion (void)
{
  fprintf (stderr, "wmSensors version %s\n", WMSENSORS_VERSION);
}

///////////////////////////////////////////////////////////////////////
//
// main
//
///////////////////////////////////////////////////////////////////////

int main (int argc, char *argv []) 
{
  int nIndex,
      bDouble;

  ///////////////////////////////////////
  // Parse Command Line
  ///////////////////////////////////////

  for (nIndex = 1; nIndex < argc; nIndex++) 
  {
    char *arg = argv [nIndex];
    int   nRes;

    bDouble = 0;

    if (*arg=='-')
    {
      arg++;

      while (*arg) 
      {
	switch (*arg)
	{
	  case '-':
	    if (!bDouble)
	      bDouble = 1;
	    else
	    {
	      Usage ();
	      exit (1);
	    }
	    break;

	  case 'c':
	    if ((bDouble == 1) && strcmp (arg, "cpu"))
	    {
	      Usage ();
	      exit (1);
	    }

	    bCPU = 1;

	    if (bDouble == 1)
	      arg [1] = '\0';

	    break;

	  case 'd':
	    if (strcmp (arg, "display") || bDouble)
	    {
	      Usage ();
	      exit (1);
	    }
	    else if (argc == ++nIndex)
	    { 
	      Usage ();
	      exit (1);
	    }
	    else
	      DisplayName = argv [nIndex];
	    
	    arg [1] = '\0';
	     
	    break;

	  case 'g':
	    if (strcmp (arg, "geometry") || bDouble)
	    {
	      Usage();
	      exit(1);
	    }
	    else if (argc == ++nIndex)
	    {
	      Usage ();
	      exit (1);
	    }
	    else
	      szGeometry = argv [nIndex];

	    arg [1] = '\0';
	     
	    break;

	  case 'l':
	    if ((bDouble == 1) && strcmp (arg, "labelcpu"))
	    {
	      Usage ();
	      exit (1);
	    }

	    bShowMB = 0;

	    if (bDouble == 1)
	      arg [1] = '\0';

	    break;

	  case 'f':
	    if ((bDouble == 1) && strcmp (arg, "fahrenheit"))
	    {
	      Usage ();
	      exit (1);
	    }

	    bCelsius = 0;

	    if (bDouble == 1)
	      arg [1] = '\0';

	    break;

	  case 'm':
	    if ((bDouble == 1) && strcmp (arg, "mainboard"))
	    {
	      Usage ();
	      exit (1);
	    }

	    bCPU = 0;

	    if (bDouble == 1)
	      arg [1] = '\0';

	    break;

          case 'p':
	    if ((bDouble == 1) && strcmp (arg, "path"))
	    {
	      Usage ();
	      exit (1);
	    }
	    else if (argc == ++nIndex)
	    { 
	      Usage ();
	      exit (1);
	    }
	    else
	      strcpy (szSensorsPath, argv [nIndex]);

            if (szSensorsPath [strlen (szSensorsPath) - 1] != '/')
              strcat (szSensorsPath, "/");
	    
	    arg [1] = '\0';
            bVersion2 = 1;

            break;

	  case 's':
	    if (((bDouble == 1) && strcmp (arg, "state")) || (bDouble == -1) || (!bDouble && arg [1] != '\0') || (argc == ++nIndex))
	    {
	      Usage ();
	      exit (1);
	    }

	    nRes = sscanf (argv [nIndex], "%i", &nCurrentState);

	    if (!nRes || (nCurrentState < 1) || (nCurrentState > 6))
	    {
	      Usage ();
	      exit (1);
	    }

	    nCurrentState--;
	    
	    arg [1] = '\0';

	    break;

	  case 'u':
	    if (((bDouble == 1) && strcmp (arg, "update")) || (bDouble == -1) || (!bDouble && arg [1] != '\0') || (argc == ++nIndex))
	    {
	      Usage ();
	      exit (1);
	    }

	    nRes = sscanf (argv [nIndex], "%i", &nSec);

	    if (!nRes || (nSec < 1))
	    {
	      Usage ();
	      exit (1);
	    }

	    arg [1] = '\0';

	    break;

	  case 'v':
	    if ((bDouble == 1) && strcmp (arg, "version"))
	    {
	      Usage ();
	      exit (1);
	    }

	    PrintVersion ();
	    exit (0);

	  default:
	    Usage ();
	    exit(0);
        }

	if (*arg != '-')
	  bDouble = -1;

	arg++;
      }
    }
  }

  wmSensors (argc, argv);

  return 0;
}

