// wmmount - Yet another lm7x monitor for WindowMaker
// 10/11/98  Release 0.1
// Copyright (C) 1998  Sam Hawker <shawkie@geocities.com>
// This software comes with ABSOLUTELY NO WARRANTY
// This software is free software, and you are welcome to redistribute it
// under certain conditions
// See the README file for a more complete notice.


// Defines, includes and global variables
// --------------------------------------

// User defines - standard
#define WINDOWMAKER false
#define USESHAPE    false
#define AFTERSTEP   false
#define NORMSIZE    64
#define ASTEPSIZE   56
#define NAME        "wmlm7x"
#define CLASS       "WMLM7x"

// User defines - custom
#define BACKCOLOR   "#282828"
#define FORECOLOR   "#c7c3c7"
#define VOLTAGEVIEW false
#define UINTERVAL   60           // 20ths of a second
#define TEMPMIN     30
#define TEMPMAX     60

// Includes - standard
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

// Includes - custom
#include <ctype.h>

// X-Windows includes - standard
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

// Pixmaps - standard
Pixmap pm_main;
Pixmap pm_tile;
Pixmap pm_disp;
Pixmap pm_mask;

// Pixmaps - custom
Pixmap pm_letters;
Pixmap pm_arrows;

// Xpm images - standard
#include "XPM/wmlm7x.xpm"
#include "XPM/tile.xpm"

// Xpm images - custom
#include "XPM/letters.xpm"
#include "XPM/arrows.xpm"

// Variables for command-line arguments - standard
bool wmaker=WINDOWMAKER;
bool ushape=USESHAPE;
bool astep=AFTERSTEP;
char display[256]="";
char position[256]="";
int winsize;

// Variables for rc file - custom
char backcolor[256]=BACKCOLOR;
char forecolor[256]=FORECOLOR;

// X-Windows basics - standard
Atom _XA_GNUSTEP_WM_FUNC;
Atom deleteWin;
Display *d_display;
Window w_icon;
Window w_main;
Window w_root;
Window w_activewin;

// X-Windows basics - custom
GC gc_gc;
unsigned long color[3];


// Misc custom global variables
// ----------------------------

// Current state information
bool voltagev=VOLTAGEVIEW;
int voltage=0;
float voltval=0;

// For buttons
int btnstate=0;
#define BTNNEXT  1
#define BTNPREV  2

// For repeating next and prev buttons
#define RPTINTERVAL   5
int rpttimer=0;

int ucount=0;
int mbtemp[52];
int cputemp[52];
char vlabels[10][7];


// Procedures and functions
// ------------------------

// Procedures and functions - standard
void initXWin(int argc, char **argv);
void freeXWin();
void createWin(Window *win, int x, int y);
unsigned long getColor(char *colorname);
unsigned long mixColor(char *colorname1, int prop1, char *colorname2, int prop2);

// Procedures and functions - custom
void scanArgs(int argc, char **argv);
void checkStats(bool forced=false);
void checkVoltage(bool forced=false);
void pressEvent(XButtonEvent *xev);
void releaseEvent(XButtonEvent *xev);
void repaint();
void update();
void drawBtns(int btns);
void drawText(int x, int y, char *text);


// Implementation
// --------------

int main(int argc, char **argv)
{
   scanArgs(argc, argv);
   initXWin(argc, argv);

   XGCValues gcv;
   unsigned long gcm;
   gcm=GCGraphicsExposures;
   gcv.graphics_exposures=false;
   gc_gc=XCreateGC(d_display, w_root, gcm, &gcv);

   color[0]=getColor(backcolor);
   color[1]=mixColor(forecolor, 60, backcolor, 40);
   color[2]=getColor(forecolor);

   XpmAttributes xpmattr;
   XpmColorSymbol xpmcsym[3]={{"back_color", NULL, color[0]},
                              {"mid_color", NULL, color[1]},
                              {"fore_color", NULL, color[2]}};
   xpmattr.numsymbols=3;
   xpmattr.colorsymbols=xpmcsym;
   xpmattr.exactColors=false;
   xpmattr.closeness=40000;
   xpmattr.valuemask=XpmColorSymbols | XpmExactColors | XpmCloseness;
   XpmCreatePixmapFromData(d_display, w_root, wmlm7x_xpm, &pm_main, &pm_mask, &xpmattr);
   XpmCreatePixmapFromData(d_display, w_root, tile_xpm, &pm_tile, NULL, &xpmattr);
   XpmCreatePixmapFromData(d_display, w_root, letters_xpm, &pm_letters, NULL, &xpmattr);
   XpmCreatePixmapFromData(d_display, w_root, arrows_xpm, &pm_arrows, NULL, &xpmattr);
   pm_disp=XCreatePixmap(d_display, w_root, 64, 64, DefaultDepth(d_display, DefaultScreen(d_display)));

   if(wmaker || ushape || astep)
      XShapeCombineMask(d_display, w_activewin, ShapeBounding, winsize/2-32, winsize/2-32, pm_mask, ShapeSet);
   else
      XCopyArea(d_display, pm_tile, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);

   XSetClipMask(d_display, gc_gc, pm_mask);
   XCopyArea(d_display, pm_main, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);
   XSetClipMask(d_display, gc_gc, None);

   FILE *f;
   if((f=fopen("/proc/sys/dev/lm_sensors/config-in-labels","r"))==NULL)
      fprintf(stderr, "%s : Cannot load label config file. Kernel module probably not installed.\n", NAME);
   else{
      fscanf(f, "%s %s %s %s %s %s %s", vlabels[0], vlabels[1], vlabels[2], vlabels[3], vlabels[4],
         vlabels[5], vlabels[6], vlabels[7]);
      fclose(f);
      for(int i=0;i<52;i++){
         mbtemp[i]=0;
         cputemp[i]=0;
      }

      checkStats(true);

      XEvent xev;
      XSelectInput(d_display, w_activewin, ExposureMask | ButtonPressMask | ButtonReleaseMask);
      XMapWindow(d_display, w_main);

      bool done=false;
      while(!done){
         while(XPending(d_display)){
            XNextEvent(d_display, &xev);
            switch(xev.type){
             case Expose:
                repaint();
             break;
             case ButtonPress:
                pressEvent(&xev.xbutton);
             break;
             case ButtonRelease:
                releaseEvent(&xev.xbutton);
             break;
             case ClientMessage:
                if(xev.xclient.data.l[0]==deleteWin)
                   done=true;
             break;
            }
         }

         if(btnstate & (BTNPREV | BTNNEXT)){
            rpttimer++;
            if(rpttimer>=RPTINTERVAL){
               if(btnstate & BTNNEXT)
                  voltage++;
               else
                  voltage--;
               if(voltage<0)
                  voltage=7-1;
               if(voltage>=7)
                  voltage=0;
               checkVoltage(true);
               rpttimer=0;
            }
         }
         ucount++;
         if(ucount>=UINTERVAL)
            checkStats(false);
         XFlush(d_display);
         usleep(50000);
      }
   }
   XFreeGC(d_display, gc_gc);
   XFreePixmap(d_display, pm_main);
   XFreePixmap(d_display, pm_tile);
   XFreePixmap(d_display, pm_letters);
   XFreePixmap(d_display, pm_arrows);
   XFreePixmap(d_display, pm_disp);
   XFreePixmap(d_display, pm_mask);
   freeXWin();
   return 0;
}

void initXWin(int argc, char **argv){
   winsize=astep ? ASTEPSIZE : NORMSIZE;

   if((d_display=XOpenDisplay(display))==NULL){
      fprintf(stderr,"%s : Unable to open X display '%s'.\n", NAME, XDisplayName(display));
      exit(1);
   }
   _XA_GNUSTEP_WM_FUNC=XInternAtom(d_display, "_GNUSTEP_WM_FUNCTION", false);
   deleteWin=XInternAtom(d_display, "WM_DELETE_WINDOW", false);

   w_root=DefaultRootWindow(d_display);

   XWMHints wmhints;
   XSizeHints shints;
   shints.x=0;
   shints.y=0;
   shints.flags=0;
   bool pos=(XWMGeometry(d_display, DefaultScreen(d_display), position, NULL, 0, &shints, &shints.x, &shints.y,
      &shints.width, &shints.height, &shints.win_gravity) & (XValue | YValue));
   shints.min_width=winsize;
   shints.min_height=winsize;
   shints.max_width=winsize;
   shints.max_height=winsize;
   shints.base_width=winsize;
   shints.base_height=winsize;
   shints.flags=PMinSize | PMaxSize | PBaseSize;

   createWin(&w_main, shints.x, shints.y);

   if(wmaker || astep || pos)
      shints.flags |= USPosition;
   if(wmaker){
      wmhints.initial_state=WithdrawnState;
      wmhints.flags=WindowGroupHint | StateHint | IconWindowHint;
      createWin(&w_icon, shints.x, shints.y);
      w_activewin=w_icon;
      wmhints.icon_window=w_icon;
   }
   else{
      wmhints.initial_state=NormalState;
      wmhints.flags=WindowGroupHint | StateHint;
      w_activewin=w_main;
   }
   wmhints.window_group=w_main;
   XSetWMHints(d_display, w_main, &wmhints);
   XSetWMNormalHints(d_display, w_main, &shints);
   XSetCommand(d_display, w_main, argv, argc);
   XStoreName(d_display, w_main, NAME);
   XSetIconName(d_display, w_main, NAME);
   XSetWMProtocols(d_display, w_activewin, &deleteWin, 1);
}

void freeXWin(){
   XDestroyWindow(d_display, w_main);
   if(wmaker)
      XDestroyWindow(d_display, w_icon);
   XCloseDisplay(d_display);
}

void createWin(Window *win, int x, int y){
   XClassHint classHint;
   *win=XCreateSimpleWindow(d_display, w_root, x, y, winsize, winsize, 0, 0, 0);
   classHint.res_name=NAME;
   classHint.res_class=CLASS;
   XSetClassHint(d_display, *win, &classHint);
}

unsigned long getColor(char *colorname){
   XColor color;
   XWindowAttributes winattr;
   XGetWindowAttributes(d_display, w_root, &winattr);
   color.pixel=0;
   XParseColor(d_display, winattr.colormap, colorname, &color);
   color.flags=DoRed | DoGreen | DoBlue;
   XAllocColor(d_display, winattr.colormap, &color);
   return color.pixel;
}

unsigned long mixColor(char *colorname1, int prop1, char *colorname2, int prop2){
   XColor color, color1, color2;
   XWindowAttributes winattr;
   XGetWindowAttributes(d_display, w_root, &winattr);
   XParseColor(d_display, winattr.colormap, colorname1, &color1);
   XParseColor(d_display, winattr.colormap, colorname2, &color2);
   color.pixel=0;
   color.red=(color1.red*prop1+color2.red*prop2)/(prop1+prop2);
   color.green=(color1.green*prop1+color2.green*prop2)/(prop1+prop2);
   color.blue=(color1.blue*prop1+color2.blue*prop2)/(prop1+prop2);
   color.flags=DoRed | DoGreen | DoBlue;
   XAllocColor(d_display, winattr.colormap, &color);
   return color.pixel;
}

void scanArgs(int argc, char **argv){
   for(int i=1;i<argc;i++){
      if(strcmp(argv[i], "-h")==0 || strcmp(argv[i], "-help")==0 || strcmp(argv[i], "--help")==0){
         fprintf(stderr, "wmmount - Yet another lm7x monitor for Window Maker\n10/11/98  Release 0.1\n");
         fprintf(stderr, "Copyright (C) 1998  Sam Hawker <shawkie@geocities.com>\n");
         fprintf(stderr, "This software comes with ABSOLUTELY NO WARRANTY\n");
         fprintf(stderr, "This software is free software, and you are welcome to redistribute it\n");
         fprintf(stderr, "under certain conditions\n");
         fprintf(stderr, "See the README file for a more complete notice.\n\n");
         fprintf(stderr, "usage:\n\n   %s [options]\n\noptions:\n\n",argv[0]);
         fprintf(stderr, "   -h | -help | --help    display this help screen\n");
         fprintf(stderr, "   -w                     use WithdrawnState    (for WindowMaker)\n");
         fprintf(stderr, "   -s                     shaped window\n");
         fprintf(stderr, "   -a                     use smaller window    (for AfterStep Wharf)\n");
         fprintf(stderr, "   -v                     start in voltage view\n");
         fprintf(stderr, "   -f fore_color          use the specified color for foregrounds\n");
         fprintf(stderr, "   -b back_color          use the specified color for backgrounds\n");
         fprintf(stderr, "   -position position     set window position   (see X manual pages)\n");
         fprintf(stderr, "   -display display       select target display (see X manual pages)\n");
         exit(0);
      }
      if(strcmp(argv[i], "-w")==0)
         wmaker=!wmaker;
      if(strcmp(argv[i], "-s")==0)
         ushape=!ushape;
      if(strcmp(argv[i], "-a")==0)
         astep=!astep;
      if(strcmp(argv[i], "-v")==0)
         voltagev=!voltagev;
      if(strcmp(argv[i], "-f")==0){
         if(i<argc-1){
            i++;
            sprintf(forecolor, "%s", argv[i]);
         }
         continue;
      }
      if(strcmp(argv[i], "-b")==0){
         if(i<argc-1){
            i++;
            sprintf(backcolor, "%s", argv[i]);
         }
         continue;
      }
      if(strcmp(argv[i], "-position")==0){
         if(i<argc-1){
            i++;
            sprintf(position, "%s", argv[i]);
         }
         continue;
      }
      if(strcmp(argv[i],"-display")==0){
         if(i<argc-1){
            i++;
            sprintf(display, "%s", argv[i]);
         }
         continue;
      }
   }
}

void checkStats(bool forced=false) {
   ucount=0;
   memmove(&mbtemp[0], &mbtemp[1], sizeof(float)*51);
   memmove(&cputemp[0], &cputemp[1], sizeof(float)*51);
   float value1, value2, value3;
   FILE *f;
   if((f=fopen("/proc/sys/dev/lm_sensors/temp-mb", "r"))==NULL)
      fprintf(stderr, "%s : Unable to read motherboard temperature.\n", NAME);
   else{
      fscanf(f, "%f %f %f", &value1, &value2, &value3);
      mbtemp[51]=(int)value3;
      fclose(f);
   }
   if((f=fopen("/proc/sys/dev/lm_sensors/temp-1", "r"))==NULL)
      fprintf(stderr, "%s : Unable to read cpu temperature.\n", NAME);
   else{
      fscanf(f, "%f %f %f", &value1, &value2, &value3);
      cputemp[51]=(int)value3;
      fclose(f);
   }
   char tval[10];
   sprintf(tval, "%3i", mbtemp[51]);
   drawText(31, 17, tval);
   sprintf(tval, "%3i", cputemp[51]);
   drawText(31, 6, tval);
   if(!voltagev && !forced){
      XCopyArea(d_display, pm_disp, pm_disp, gc_gc, 7, 28, 51, 29, 6, 28);
      int value1=((int)((float)(mbtemp[51]-30)*(29.0/(float)(TEMPMAX-TEMPMIN))) >? 0) <? 29;
      int value2=((int)((float)(cputemp[51]-30)*(29.0/(float)(TEMPMAX-TEMPMIN))) >? 0) <? 29;
      XSetForeground(d_display, gc_gc, color[2]);
      XDrawLine(d_display, pm_disp, gc_gc, 57, 56, 57, 56-value1);
      XSetForeground(d_display, gc_gc, color[1]);
      XDrawLine(d_display, pm_disp, gc_gc, 57, 56-value1-1, 57, 56-value2);
      XSetForeground(d_display, gc_gc, color[0]);
      XDrawLine(d_display, pm_disp, gc_gc, 57, 56-value2-1, 57, 56-29);
   }
   checkVoltage(forced);
}

void checkVoltage(bool forced=false) {
   FILE *f;
   char fpath[256];
   sprintf(fpath, "/proc/sys/dev/lm_sensors/in%i", voltage);
   if((f=fopen(fpath, "r"))==NULL)
      fprintf(stderr, "%s : Unable to read voltage.\n", NAME);
   else{
      float value1, value2, value3;
      fscanf(f, "%f %f %f", &value1, &value2, &value3);
      voltval=value3;
      fclose(f);
   }
   if(voltagev && !forced){
      drawText(10, 33, vlabels[voltage]);
      char vval[10];
      sprintf(vval, "%+3i", (int)voltval);
      drawText(16, 43, vval);
      sprintf(vval, "%02i", abs((int)((voltval-(int)voltval)*100)));
      drawText(36, 43, vval);
   }
   if(forced)
      update();
   repaint();
}

void pressEvent(XButtonEvent *xev){
   int x=xev->x-(winsize/2-32);
   int y=xev->y-(winsize/2-32);
   if(voltagev){
      if(x>=47 && y>=26 && x<=60 && y<=30+(x-47)){
         voltage++;
         if(voltage>=7)
            voltage=0;
         btnstate|=BTNNEXT;
         rpttimer=0;
         checkVoltage(true);
         return;
      }
      if(x>=4 && y>=49+(x-4) && x<=17 && y<=59){
         voltage--;
         if(voltage<0)
            voltage=7-1;
         btnstate|=BTNPREV;
         rpttimer=0;
         checkVoltage(true);
         return;
      }
   }
   voltagev=!voltagev;
   update();
}

void releaseEvent(XButtonEvent *xev){
   btnstate&=~(BTNPREV | BTNNEXT);
   drawBtns(BTNPREV | BTNNEXT);
   checkVoltage(false);
   repaint();
}

void repaint(){
   XCopyArea(d_display, pm_disp, w_activewin, gc_gc, 0, 0, 64, 64, winsize/2-32, winsize/2-32);
   XEvent xev;
   while(XCheckTypedEvent(d_display, Expose, &xev));
}

void update(){
   XCopyArea(d_display, pm_main, pm_disp, gc_gc, 4, 26, 56, 33, 4, 26);
   drawBtns(BTNPREV | BTNNEXT);
   if(!voltagev){
      for(int i=0;i<52;i++){
         int value1=((int)((float)(mbtemp[i]-30)*(29.0/(float)(TEMPMAX-TEMPMIN))) >? 0) <? 29;
         int value2=((int)((float)(cputemp[i]-30)*(29.0/(float)(TEMPMAX-TEMPMIN))) >? 0) <? 29;
         XSetForeground(d_display, gc_gc, color[2]);
         XDrawLine(d_display, pm_disp, gc_gc, 6+i, 56, 6+i, 56-value1);
         XSetForeground(d_display, gc_gc, color[1]);
         XDrawLine(d_display, pm_disp, gc_gc, 6+i, 56-value1-1, 6+i, 56-value2);
         XSetForeground(d_display, gc_gc, color[0]);
         XDrawLine(d_display, pm_disp, gc_gc, 6+i, 56-value2-1, 6+i, 56-29);
      }
   }
   else{
      drawText(10, 33, vlabels[voltage]);
      char vval[10];
      sprintf(vval, "%+3i", (int)voltval);
      drawText(16, 43, vval);
      sprintf(vval, "%02i", abs((int)((voltval-(int)voltval)*100)));
      drawText(36, 43, vval);
   }
}

void drawBtns(int btns){
   if(!voltagev)
      return;
   if(btns & BTNPREV){
      if(btnstate & BTNPREV)
         XCopyArea(d_display, pm_arrows, pm_disp, gc_gc, 0, 0, 15, 15, 4, 44);
      else
         XCopyArea(d_display, pm_main, pm_disp, gc_gc, 4, 44, 15, 15, 4, 44);
   }
   if(btns & BTNNEXT){
      if(btnstate & BTNNEXT)
         XCopyArea(d_display, pm_arrows, pm_disp, gc_gc, 15, 0, 15, 15, 45, 26);
      else
         XCopyArea(d_display, pm_main, pm_disp, gc_gc, 45, 26, 15, 15, 45, 26);
   }
}

void drawText(int x, int y, char *text){
   char letters[]="+-.0123456789 ABCDEFGHIJKLMNOPQRSTUVWXYZ";
   for(int i=0;i<strlen(text);i++){
      char *letter=strchr(letters, toupper(text[i]));
      int lpos=(letter==NULL) ? 0 : (letter-letters);
      XCopyArea(d_display, pm_letters, pm_disp, gc_gc, lpos*6, 0, 5, 7, x+6*i, y);
   }
}
