/* wmhealth.c (C source file)
 *
 * wmhealth - A system health monitor designed for the Window Maker dock
 *
 * 18/06/99  Release 1.0 beta1
 * Copyright (C) 1999  Sam Hawker <shawkie@geocities.com>
 * This software comes with ABSOLUTELY NO WARRANTY
 * This software is free software, and you are welcome to redistribute it
 * under certain conditions
 * See the README file for details.
 */


/* User defines
 */

/* Intervals
 *   INTERVAL  - Minimum interval unit in usecs
 *   INTERVAL1 - Update interval in multiples of INTERVAL
 *   INTERVAL2 - Button repeat interval in multiples of INTERVAL
 */
#define INTERVAL  250000
#define INTERVAL1 10
#define INTERVAL2 1

/* View mode
 *   0 (temperatures history)
 *   1 (voltage readout)
 */
#define VIEWMODE 0

/* Number of voltage readouts
 */
#define VOLTAGES 7

/* Minimum and maximum temperatures for history display
 */
#define TEMPMIN 30
#define TEMPMAX 60


/* End of user defines
 * AVOID MAKING MODIFICATIONS BEYOND THIS POINT
 */


/* Includes
 */

#include "dockapp.h"

#include <ctype.h>

#include "xpm/main.xpm"
#include "xpm/arrw.xpm"
#include "xpm/lttr.xpm"


/* Variables
 */

Pixmap pm_main;
Pixmap pm_main_mask;
Pixmap pm_arrw;
Pixmap pm_lttr;

#define BUTTON1 (int)(1<<1)
#define BUTTON2 (int)(1<<2)
#define PREV    BUTTON1
#define NEXT    BUTTON2

int state=0;

int select1=VIEWMODE;
int select2=0;

int counter1=0;
int counter2=0;

float voltage=0;
char *vlabels[VOLTAGES];

int mbtemp[52];
int cputemp[52];


/* Functions and procedures
 */

int main(int argc, char **argv);

void init(int argc, char * const *argv);
void done();

void mainloop();

void buttonpress(XButtonEvent *xev);
void buttonrelease(XButtonEvent *xev);

int lm7x_init();
void lm7x_done();

void lm7x_readtemps();
void lm7x_readvoltage();

void xpm_init();
void xpm_done();

void checktemps(bool force);
void checkvoltage(bool force);

void drawall();
void drawbtns(int btnmask);
void drawtemps();
void drawvoltage();
void drawtext(int x, int y, char *text);
void drawgraph();
void drawgraphbar(int bar);


/* Implementation
 */

int main(int argc, char **argv) {
  /* Initialize
   */
  init(argc, (char * const *)argv);

  /* Initialize dockapp
   */
  dockapp_init(argc, (char * const *)argv);

  /* Initialize lm7x
   */
  if(!lm7x_init()) {

    /* Open dockapp
     */
    if(!dockapp_open()) {

      /* Initialize xpms
       */
      xpm_init();

      /* Check temperatures and voltage
       */
      checktemps(true);
      checkvoltage(true);

      /* Draw everything
       */
      drawall();
      
      /* Execute main loop
       */
      mainloop();
      
      /* Free xpms
       */
      xpm_done();

      /* Close dockapp
       */
      dockapp_close();
      
    }

    /* Free lm7x
     */
    lm7x_done();

  }

  /* Free dockapp
   */
  dockapp_done();

  /* Free
   */
  done();

  /* Finish
   */
  return 0;
}

void init(int argc, char * const *argv) {
  /* Process command-line arguments
   */
  int option_index;
  int option_c;
  struct option long_options[]={
    {"help",        0, NULL, 'h' },
    {"viewmode",    1, NULL, 'v' },
    {NULL,          0, NULL, '\0'}
  };
  while(true) {
    option_index=0;
    option_c=getopt_long_only(argc, (char * const *)argv, "-hv:", long_options, &option_index);
    if(option_c==-1)
      break;
    if(option_c==1) {
      fprintf(stderr, "%s: unwanted argument -- '%s'\n", argv[0], optarg);
    }
    if(option_c>1) {
      switch(option_c) {
      case 'h':
	printf("wmhealth - A system health monitor designed for the Window Maker dock\n");
        printf("\n");
	printf("18/06/99  Release 1.0 beta1\n");
        printf("Copyright (C) 1999  Sam Hawker <shawkie@geocities.com>\n");
        printf("This software comes with ABSOLUTELY NO WARRANTY\n");
        printf("This software is free software, and you are welcome to redistribute it\n");
        printf("under certain conditions\n");
	printf("See the README file for details.\n");
	printf("\n");
        printf("usage:\n");
        printf("\n");
	printf("   %s [options] [-- dockapp_options]\n", argv[0]);
        printf("\n");
        printf("options:\n");
        printf("\n");
	printf("   -h, --help           Show this help message\n");
        printf("   -v, --viewmode mode  Select view mode (0 - 1)\n");
        printf("\n");
        printf("dockapp options:\n");
        printf("\n");
        printf("   -h, --help           Show dockapp help message\n");
	exit(0);
      case 'v':
	sscanf(optarg, "%i", &select1);
	break;
      }
    }
  }
}

void done() {
  /* Free command-line option arguments
   */
}

void mainloop() {
  /* Do the main loop
   */
  XEvent xev;

  fd_set fds;
  struct timeval tv;

  XSelectInput(dockapp_d, dockapp_w_active, ButtonPressMask | ButtonReleaseMask | ExposureMask);
  XMapWindow(dockapp_d, dockapp_w_main);

  while(true) {
    while(XPending(dockapp_d)) {
      XNextEvent(dockapp_d, &xev);
      switch(xev.type) {
      case Expose:
	dockapp_expose();
	break;
      case ButtonPress:
        buttonpress(&xev.xbutton);
	break;
      case ButtonRelease:
	buttonrelease(&xev.xbutton);
	break;
      case ClientMessage:
	if(xev.xclient.data.l[0]==dockapp_a_delwin)
          return;
      }
    }
    FD_ZERO(&fds);
    FD_SET(ConnectionNumber(dockapp_d), &fds);
    tv.tv_sec=(long)INTERVAL/(long)1000000;
    tv.tv_usec=(long)INTERVAL%(long)1000000;
    if(!select(ConnectionNumber(dockapp_d)+1, &fds, NULL, NULL, &tv)) {
      counter1++;
      if(counter1>=INTERVAL1) {
	counter1=0;
	/* Update interval
	 */
	checktemps(true);
	checkvoltage(false);
      }
      counter2++;
      if(counter2>=INTERVAL2) {
	counter2=0;
	/* Button repeat interval
	 */
	if(state & (PREV | NEXT)) {
	  if(state & PREV)
	    select2--;
	  else
	    select2++;
	  if(select2<0)
	    select2=VOLTAGES-1;
	  if(select2>VOLTAGES-1)
	    select2=0;
	  checkvoltage(true);
	} 
      }
    }
  }
}

void buttonpress(XButtonEvent *xev) {
  /* Process button press events
   */
  int x=xev->x-(dockapp_size/2-32);
  int y=xev->y-(dockapp_size/2-32);

  if(select1==1) {
    if(x>=4 && y>=41+(x-4) && x<=17 && y<=59) {
      select2--;
      if(select2<0)
        select2=VOLTAGES-1;
      counter2=0;
      state|=PREV;
      checkvoltage(true);
      return;
    }
    if(x>=47 && y>=26 && x<=60 && y<=30+(x-47)) {
      select2++;
      if(select2>VOLTAGES-1)
        select2=0;
      counter2=0;
      state|=NEXT;
      checkvoltage(true);
      return;
    }
  }
  select1++;
  if(select1>2-1)
    select1=0;
  drawall();
  dockapp_expose();
}

void buttonrelease(XButtonEvent *xev) {
  /* Process button release events
   */
  state&=~(PREV | NEXT);
  drawall();
  dockapp_expose();
}

void checktemps(bool force) {
  /* Read temperatures and update display
   */
  lm7x_readtemps();
  if(force)
    drawall();
  else {
    drawtemps();
    if(select1==0) {
      /* Scroll display
       */
      XCopyArea(dockapp_d, dockapp_pm_main, dockapp_pm_main, dockapp_gc_clip, 7, 28, 51, 29, 6, 28);
      /* Draw new bar
       */
      drawgraphbar(51);
    }
  }
  dockapp_expose();
}

void checkvoltage(bool force) {
  /* Read voltage and update display
   */
  lm7x_readvoltage();
  if(force)
    drawall();
  else {
    if(select1==1)
      drawvoltage();
  }
  dockapp_expose();
}

int lm7x_init() {
  int i;

  /* Load lm7x voltage labels
   */
  FILE *f;
  char buf[256];
  if((f=fopen("/proc/sys/dev/lm_sensors/config-in-labels", "r"))==NULL) {
    fprintf(stderr, "%s: Cannot load label config file. Kernel module probably not installed\n", dockapp_argv[0]);
    return -1;
  }
  else {
    for(i=0;i<VOLTAGES;i++) {
      fscanf(f, "%s ", buf);
      vlabels[i]=strdup(buf);
    }
    fclose(f);
  }

  /* Set temperature graph values
   */
  for(i=0;i<52;i++) {
    mbtemp[i]=0;
    cputemp[i]=0;
  }

  return 0;
}

void lm7x_done() {
  /* Free lm7x voltage labels
   */
  int i;
  for(i=0;i<VOLTAGES;i++)
    free(vlabels[i]);
}

void lm7x_readtemps() {
  /* Read temperatures from lm7x
   */
  float value1, value2, value3;
  FILE *f;
  memmove(&mbtemp[0], &mbtemp[1], sizeof(float)*51);
  memmove(&cputemp[0], &cputemp[1], sizeof(float)*51);
  if((f=fopen("/proc/sys/dev/lm_sensors/temp-mb", "r"))==NULL)
    fprintf(stderr, "%s: Unable to read motherboard temperature\n", dockapp_argv[0]);
  else {
    fscanf(f, "%f %f %f", &value1, &value2, &value3);
    mbtemp[51]=(int)value3;
    fclose(f);
  }
  if((f=fopen("/proc/sys/dev/lm_sensors/temp-1", "r"))==NULL)
    fprintf(stderr, "%s: Unable to read cpu temperature\n", dockapp_argv[0]);
  else {
    fscanf(f, "%f %f %f", &value1, &value2, &value3);
    cputemp[51]=(int)value3;
    fclose(f);
  }
}

void lm7x_readvoltage() {
  /* Read voltage from lm7x
   */
  char fpath[256];
  float value1, value2, value3;
  FILE *f;
  sprintf(fpath, "/proc/sys/dev/lm_sensors/in%i", select2);
  if((f=fopen(fpath, "r"))==NULL)
    fprintf(stderr, "%s: Unable to read voltage\n", dockapp_argv[0]);
  else {
    fscanf(f, "%f %f %f", &value1, &value2, &value3);
    voltage=value3;
    fclose(f);
  }
}

void xpm_init() {
  /* Create pixmaps from xpm data
   */
  XpmAttributes xpmattr;
  XpmColorSymbol xpmcsym[4]={{"dockapp_color0", NULL, dockapp_color0},
			     {"dockapp_color1", NULL, dockapp_color1},
			     {"dockapp_color2", NULL, dockapp_color2},
			     {"dockapp_color3", NULL, dockapp_color3}};

  xpmattr.numsymbols=4;
  xpmattr.colorsymbols=xpmcsym;
  xpmattr.exactColors=false;
  xpmattr.closeness=40000;
  xpmattr.valuemask=XpmColorSymbols | XpmExactColors | XpmCloseness | XpmSize;

  XpmCreatePixmapFromData(dockapp_d, dockapp_w_root, main_xpm, &pm_main, &pm_main_mask, &xpmattr);
  XpmCreatePixmapFromData(dockapp_d, dockapp_w_root, arrw_xpm, &pm_arrw, NULL, &xpmattr);
  XpmCreatePixmapFromData(dockapp_d, dockapp_w_root, lttr_xpm, &pm_lttr, NULL, &xpmattr);

  XCopyArea(dockapp_d, pm_main_mask, dockapp_pm_mask, dockapp_gc_mask, 0, 0, 64, 64, 0, 0);
  dockapp_installmask();
}

void xpm_done() {
  /* Free xpms
   */

  /* Done automatically by X, so I won't bother
   */
}

void drawall() {
  /* Draw everything
   */
  XCopyArea(dockapp_d, pm_main, dockapp_pm_main, dockapp_gc_clip, 0, 0, 64, 64, 0, 0);
  drawtemps();
  if(select1==0) {
    drawgraph();
  }
  if(select1==1) {
    drawbtns(PREV | NEXT);
    drawtext(10, 33, vlabels[select2]);
    drawvoltage();
  }
}

void drawbtns(int btnmask) {
  /* Draw buttons
   */
  if(btnmask & BUTTON1) {
    if(state & BUTTON1)
      XCopyArea(dockapp_d, pm_arrw, dockapp_pm_main, dockapp_gc_main, 0, 0, 15, 15, 4, 44);
    else
      XCopyArea(dockapp_d, pm_main, dockapp_pm_main, dockapp_gc_main, 4, 44, 15, 15, 4, 44);
  }
  if(btnmask & BUTTON2) {
    if(state & BUTTON2)
      XCopyArea(dockapp_d, pm_arrw, dockapp_pm_main, dockapp_gc_main, 15, 0, 15, 15, 45, 26);
    else
      XCopyArea(dockapp_d, pm_main, dockapp_pm_main, dockapp_gc_main, 45, 26, 15, 15, 45, 26);
  }
}

void drawtemps() {
  /* Draw temperature value texts
   */
  char text[8];
  sprintf(text, "%3i", mbtemp[51]);
  drawtext(31, 17, text);
  sprintf(text, "%3i", cputemp[51]);
  drawtext(31, 6, text);
}

void drawvoltage() {
  /* Draw voltage value text
   */
  char text[8];
  sprintf(text, "%+3i", (int)voltage);
  drawtext(16, 43, text);
  sprintf(text, "%02i", abs((int)(voltage*100)%100));
  drawtext(36, 43, text);
}

void drawtext(int x, int y, char *text) {
  /* Draw a piece of text
   */
  int i;
  char letters[]="+-.0123456789 ABCDEFGHIJKLMNOPQRSTUVWXYZ";
  char *letter;
  int lpos;
  for(i=0;i<strlen(text);i++) {
    letter=strchr(letters, toupper(text[i]));
    if(letter==NULL)
      lpos=0;
    else
      lpos=letter-letters;
    XCopyArea(dockapp_d, pm_lttr, dockapp_pm_main, dockapp_gc_main, lpos*6, 0, 5, 7, x+6*i, y);
  }
}

void drawgraph() {
  /* Draw entire temperature graph
   */
  int i;
  for(i=0;i<52;i++)
    drawgraphbar(i);
}

void drawgraphbar(int bar) {
  /* Draw particular temperature graph bar
   */
  int value1=(int)((float)(mbtemp[bar]-30)*(29.0/(float)(TEMPMAX-TEMPMIN)));
  int value2=(int)((float)(cputemp[bar]-30)*(29.0/(float)(TEMPMAX-TEMPMIN)));
  if(value1<0)
    value1=0;
  if(value1>29)
    value1=29;
  if(value2<0)
    value2=0;
  if(value2>29)
    value2=29;
  XSetForeground(dockapp_d, dockapp_gc_main, dockapp_color0);
  XDrawLine(dockapp_d, dockapp_pm_main, dockapp_gc_main, 6+bar, 56, 6+bar, 56-value1);
  XSetForeground(dockapp_d, dockapp_gc_main, dockapp_color2);
  XDrawLine(dockapp_d, dockapp_pm_main, dockapp_gc_main, 6+bar, 56-value1-1, 6+bar, 56-value2);
  XSetForeground(dockapp_d, dockapp_gc_main, dockapp_color1);
  XDrawLine(dockapp_d, dockapp_pm_main, dockapp_gc_main, 6+bar, 56-value2-1, 6+bar, 56-29); 
}
