
/*
    smbus.h - Part of a Linux module for reading sensor data.
    Copyright (c) 1998  Alexander Larsson <alla@lysator.liu.se>,
    Frodo Looijaard <frodol@dds.nl> and Philip Edelbrock <phil@netroedge.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef __LINUX_SMBUS_H
#define __LINUX_SMBUS_H

#include <asm/types.h>
#include <linux/unistd.h>

/*
	Syscall Stuff
*/
#define __NR_smbus_action   235
#define SMBUS_READ	1
#define SMBUS_WRITE	0

/* SMBus transaction types (size parameter in the above functions) */
#define SMBUS_QUICK      0x00
#define SMBUS_BYTE       0x04
#define SMBUS_BYTE_DATA  0x08 
#define SMBUS_WORD_DATA  0x0C
#define SMBUS_BLOCK_DATA 0x14

/* Note that we use __u8, not u8; this is to keep the namespace clean */
union SMBus_Data { 
        __u8 byte;
        __u16 word;
        __u8 block[32];
};

#ifndef __KERNEL__

/* Use the following line in your sources if you need this syscall:

  _syscall5(int, smbus_action, __u8, addr, char, read_write, __u8, command, 
            int, size,union SMBus_Data *,data);

  Note that due to the 'extern inline' hack, you need to use at least -O1 if 
  you want to use the easier syscall access functions defined below (read
  the GCC manual if you want to know why).

*/

extern int smbus_action(__u8 addr, char read_write, __u8 command, int size,
                        union SMBus_Data * data);

extern inline int SMBus_Write_Quick(__u8 addr, __u8 value)
{
  return smbus_action(addr,value,0,SMBUS_QUICK,NULL);
}

extern inline int SMBus_Read_Byte(__u8 addr)
{
  union SMBus_Data data;
  if (smbus_action(addr,SMBUS_READ,0,SMBUS_BYTE,&data))
    return -1;
  else
    return data.byte;
}

extern inline int SMBus_Write_Byte(__u8 addr, __u8 value)
{
  return smbus_action(addr,SMBUS_WRITE,value,SMBUS_BYTE,NULL);
}

extern inline int SMBus_Read_Byte_Data(__u8 addr, __u8 command)
{
  union SMBus_Data data;
  if (smbus_action(addr,SMBUS_READ,command,SMBUS_BYTE_DATA,&data))
    return -1;
  else
    return data.byte;
}

extern inline int SMBus_Write_Byte_Data(__u8 addr, __u8 command, __u8 value)
{
  union SMBus_Data data;
  data.byte = value;
  return smbus_action(addr,SMBUS_WRITE,command,SMBUS_BYTE_DATA,&data);
}

extern inline int SMBus_Read_Word_Data(__u8 addr, __u8 command)
{
  union SMBus_Data data;
  if (smbus_action(addr,SMBUS_READ,command,SMBUS_WORD_DATA,&data))
    return -1;
  else
    return data.word;
}

extern inline int SMBus_Write_Word_Data(__u8 addr, __u8 command, __u16 value)
{
  union SMBus_Data data;
  data.word = value;
  return smbus_action(addr,SMBUS_WRITE,command,SMBUS_WORD_DATA,&data);
}

/* Returns the number of read bytes */
extern inline int SMBus_Read_Block_Data(__u8 addr, __u8 command, __u8 *values)
{
  union SMBus_Data data;
  int i;
  if (smbus_action(addr,SMBUS_READ,command,SMBUS_BLOCK_DATA,&data))
    return -1;
  else {
    for (i = 1; i <= data.block[0]; i++)
      values[i-1] = data.block[i];
    return data.block[0];
  }
}

extern inline int SMbus_Write_Block_Data(__u8 addr, __u8 command, __u8 length,
                                         __u8 *values)
{
  union SMBus_Data data;
  int i;
  if (length < 0)
    length = 0;
  if (length > 32)
    length = 32;
  for (i = 1; i <= length; i++)
    data.block[i] = values[i-1];
  data.block[0] = length;
  return smbus_action(addr,SMBUS_WRITE,command,SMBUS_BLOCK_DATA,&data);
}
  
#else

/*
	Prototypes 
*/
extern int SMBus_Init(void);
extern void SMBus_Cleanup(void);
extern int SMBus_Access(u8 addr, char read_write, u8 command, int size,
                        union SMBus_Data *data);


extern int SMBus_Initialized;


extern int check_smbus(u8 address);
extern void request_smbus(u8 address, const char *name);
extern void release_smbus(u8 address);

extern inline int SMBus_Write_Quick(u8 addr, u8 value)
{
  return SMBus_Access(addr,value,0,SMBUS_QUICK,NULL);
}

extern inline int SMBus_Read_Byte(u8 addr)
{
  union SMBus_Data data;
  if (SMBus_Access(addr,SMBUS_READ,0,SMBUS_BYTE,&data))
    return -1;
  else
    return data.byte;
}

extern inline int SMBus_Write_Byte(u8 addr, u8 value)
{
  return SMBus_Access(addr,SMBUS_WRITE,value,SMBUS_BYTE,NULL);
}

extern inline int SMBus_Read_Byte_Data(u8 addr, u8 command)
{
  union SMBus_Data data;
  if (SMBus_Access(addr,SMBUS_READ,command,SMBUS_BYTE_DATA,&data))
    return -1;
  else
    return data.byte;
}

extern inline int SMBus_Write_Byte_Data(u8 addr, u8 command, u8 value)
{
  union SMBus_Data data;
  data.byte = value;
  return SMBus_Access(addr,SMBUS_WRITE,command,SMBUS_BYTE_DATA,&data);
}

extern inline int SMBus_Read_Word_Data(u8 addr, u8 command)
{
  union SMBus_Data data;
  if (SMBus_Access(addr,SMBUS_READ,command,SMBUS_WORD_DATA,&data))
    return -1;
  else
    return data.word;
}

extern inline int SMBus_Write_Word_Data(u8 addr, u8 command, u16 value)
{
  union SMBus_Data data;
  data.word = value;
  return SMBus_Access(addr,SMBUS_WRITE,command,SMBUS_WORD_DATA,&data);
}

/* Returns the number of read bytes */
extern inline int SMBus_Read_Block_Data(u8 addr, u8 command, u8 *values)
{
  union SMBus_Data data;
  int i;
  if (SMBus_Access(addr,SMBUS_READ,command,SMBUS_BLOCK_DATA,&data))
    return -1;
  else {
    for (i = 1; i <= data.block[0]; i++)
      values[i-1] = data.block[i];
    return data.block[0];
  }
}

extern inline int SMbus_Write_Block_Data(u8 addr, u8 command, u8 length,
                                         u8 *values)
{
  union SMBus_Data data;
  int i;
  if (length > 32)
    length = 32;
  for (i = 1; i <= length; i++)
    data.block[i] = values[i-1];
  data.block[0] = length;
  return SMBus_Access(addr,SMBUS_WRITE,command,SMBUS_BLOCK_DATA,&data);
}
  
#endif /* __KERNEL__ */

#endif
