/* -*- linux-c -*- */


/*
    SMBus.c - Part of a Linux module for reading sensor data.
    Copyright (c) 1998  Alexander Larsson <alla@lysator.liu.se>,
    Frodo Looijaard <frodol@dds.nl> and Philip Edelbrock <phil@netroedge.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*
 *  SMBus driver for LINUX
 *
 * Copyright (c) 1998 Philip Edelbrock, Alexander Larsson
 * <alla@lysator.liu.se>, and Frodo Looijaard <frodol@dds.nl>
 * 
 * part of a module which provides a 
 * /proc/sensors file which shows
 * some information about your system.
 * Voltages, temperature and fan speeds
 * are currently reported.
 *
 *
 * please report any bugs/patches to me:
 *  Philip Edelbrock <phil@netroedge.com>, and 
 *  Frodo Looijaard <frodol@dds.nl>
 *
 *
 * Reference:
 *   National Semiconductor
 *   LM75 Digital Temperature Sensor and Thermal Watchdog with
 *   Two-Wire Interface, PDF, April 1997.
 *   http://www.national.com/ds/LM/LM75.pdf
 *
 *   Intel
 *   82371AB PCI-TO-ISA / IDE XCELERATOR (PIIX4), PDF, April 1997.
 *   ftp://ftp.intel.com
 *   
 *   System Management Bus Specification, Rev. 1.0, 1996.
 *   Contributors:
 *   Benchmarq Microelectronics Inc ., Duracell Inc.,
 *   Energizer Power Systems, Intel Corporation, Linear Technology Corporation,
 *   Maxim Integrated Products, Mitsubishi Electric Corporation,
 *   National Semiconductor Corporation, Toshiba Battery Co.,
 *   Varta Batterie AG, All rights reserved.
 *   http://www.sbs-forum.org/
 */
#include <linux/config.h>

#ifndef SMBUS_MODULE
/* The next define disables the redefinition of kernel_version. As it is
   already defined and initialized in lm78.c, this would result in twice
   defined symbols. We must include <linux/version.h> by hand now. */
#define __NO_VERSION__
#include <linux/version.h>
#endif

/* Hack to allow the use of CONFIG_MODVERSIONS. In the kernel itself, this
   is arranged from the main Makefile. */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <asm/semaphore.h>


#include "smbus.h"
#include "lmsmbus.h"
#include "compat.h"
#include "lmversion.h"

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,1,0))
#include <linux/bios32.h>
#else
#include <asm/uaccess.h>
#endif

/*
	Global variables
*/
static int syscalls_initialized = 0;

/* This is straight from kernel/resource.c 
   The first entry in smbuslist is fake; it is just there to keep the
   code smaller.
   We use smbuslist to do our searching etc.; smbustable is only there
   to make dynamic allocation unnecessary. 
*/

static smbus_entry_t smbuslist = { 0, "", NULL };
static smbus_entry_t smbustable[SMBUSTABLE_SIZE];

static unsigned short PIIX4_smba = 0;
static struct semaphore smbus_sem = MUTEX;

int SMBus_Initialized=0;


/*
	External function which are not in smbus.h
*/

#ifdef SMBUS_MODULE
extern int init_module(void);
extern void cleanup_module(void);
#endif

/*
	Local functions
*/

static void do_pause( unsigned int amount );
static smbus_entry_t *find_gap(smbus_entry_t *root, u8 address);
asmlinkage static int sys_smbus_action(u8 addr, char read_write, u8 command,
                                       int size,union SMBus_Data *data);
static int SMBus_Transaction(void);

#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
 static struct proc_dir_entry *proc_smbus = NULL;
 static int get_smbus_list (char *buf, char **start, off_t offset,
                            int len, int *eof, void *private);
#else
 static int get_smbus_list(char *, char **, off_t, int, int);
 static int inodeval = 0; /* Things depend on this initialization! */
 static struct proc_dir_entry dir =
 {
   0, 5, "smbus",
   S_IFREG | S_IRUGO, 1, 0, 0,
   0, NULL,
   &get_smbus_list
 };
#endif

#ifdef SMBUS_MODULE

static int forced=0;

MODULE_AUTHOR("Alexander Larsson <alla@lysator.liu.se>, Frodo Looijaard <frodol@dds.nl>, Philip Edelbrock <phil@Ren.netroedge.com>");

MODULE_DESCRIPTION("Driver for SMBus (System Management Bus) access");

MODULE_PARM(forced,"i");
MODULE_PARM_DESC(forced,"If set, it forces the SMBus module to load, even if no SMBus is detected");

int init_module(void)
{
  int res;

  printk("smbus version %s (%s)\n",LM_VERSION,LM_DATE);
  res=SMBus_Init();
  if (res)
    printk("SMBus not detected\n");
  else
    printk("SMBus detected and initialized\n");
  if (! forced)
    return res;
  else
    return 0;
}

void cleanup_module(void)
{
  SMBus_Cleanup();
}
#endif

int SMBus_Init(void)
{
  int error_return=0;
  unsigned char temp;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,1,0))
  struct pci_dev *PIIX4_dev;
#else
  unsigned char PIIX4_bus, PIIX4_devfn;
#endif

  if (pci_present() == 0) {
    printk("SMBus: Error: No PCI-bus found!\n");
    error_return=-ENODEV;
  } else {

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,1,0))
    PIIX4_dev = pci_find_device(PCI_VENDOR_ID_INTEL, 
                                 PCI_DEVICE_ID_INTEL_82371AB_0, NULL);
    if(PIIX4_dev == NULL) {
#else
    if(pcibios_find_device(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82371AB_0, 
                           0, &PIIX4_bus, &PIIX4_devfn)) {
#endif
      printk("SMBus: Error: Can't detect PIIX4!\n");
      error_return=-ENODEV;
    } else {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,1,0))
      /* For some reason, pci_read_config fails here sometimes! */
      pcibios_read_config_word(PIIX4_dev->bus->number, PIIX4_dev->devfn | 3, 
                               SMBBA, &PIIX4_smba);
#else
      pci_read_config_word_united(PIIX4_dev, PIIX4_bus ,PIIX4_devfn | 3,
                                  SMBBA,&PIIX4_smba);
#endif
      PIIX4_smba &= 0xfff0;
      if (check_region(PIIX4_smba, 8)) {
        printk("PIIX4_smb region 0x%x already in use!\n",
               PIIX4_smba);
        error_return=-ENODEV;
      } else {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,1,0))
        pcibios_read_config_byte(PIIX4_dev->bus->number, 
		PIIX4_dev->devfn | 3, SMBHSTCFG, &temp);
#else
        pci_read_config_byte_united(PIIX4_dev, PIIX4_bus, PIIX4_devfn | 3, 
                                    SMBHSTCFG, &temp);
#endif
#ifdef FORCE_PIIX4_ENABLE
/* This should never need to be done, but has been noted that
   many Dell machines have the SMBus interface on the PIIX4
   disabled!? NOTE: This assumes I/O space and other allocations WERE
   done by the Bios!  Don't complain if your hardware does weird 
   things after enabling this. :') Check for Bios updates before
   resorting to this.  */
	if ((temp & 1) == 0) {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,1,0))
          pcibios_write_config_byte(PIIX4_dev->bus->number, PIIX4_dev->devfn | 3, 
                               SMBHSTCFG, temp | 1);
#else
          pcibios_write_config_byte(PIIX4_bus, PIIX4_devfn | 3, SMBHSTCFG, temp | 1);
#endif
	  printk("SMBus: WARNING: PIIX4 SMBus interface has been FORCEFULLY ENABLED!!\n");
	  /* Update configuration value */
          pci_read_config_byte_united(PIIX4_dev, PIIX4_bus, PIIX4_devfn | 3, SMBHSTCFG, &temp);
	  /* Note: We test the bit again in the next 'if' just to be sure... */
	}
#endif
	if ((temp & 1) == 0) {
          printk("SMBUS: Error: Host SMBus controller not enabled!\n");     
          SMBus_Initialized=0;
	  error_return=-ENODEV;
	} else {
        	/* Everything is happy, let's grab the memory and set things up. */
        	request_region(PIIX4_smba, 8, "SMBus");       
        	SMBus_Initialized=1;
	}
#ifdef DEBUG
        if ((temp & 0x0E) == 8)
          printk("SMBUS: PIIX4 using Interrupt 9 for SMBus.\n");
        else if ((temp & 0x0E) == 0)
          printk("SMBUS: PIIX4 using Interrupt SMI# for SMBus.\n");
        else 
          printk( "SMBUS: PIIX4: Illegal Interrupt configuration (or code out of date)!\n");
        pci_read_config_byte_united(PIIX4_dev, PIIX4_bus, PIIX4_devfn | 3, 
                                        SMBREV, &temp);
        printk("SMBUS: SMBREV = 0x%X\n",temp);
#endif
      }
    }
  }
  if (error_return)
    printk("SMBus setup failed.\n");
#ifdef DEBUG
  else
    printk("PIIX4_smba = 0x%X\n",PIIX4_smba);
#endif

/* Setup Syscall stuff */
  if (! error_return) {
	if (sys_call_table[__NR_smbus_action]) {
	 printk("System call %d already in use!\n",__NR_smbus_action);
/*	 SMBus_Cleanup();  -- We don't really need to completely bail out
	 error_return = -EBUSY; */
	} else {
	  sys_call_table[__NR_smbus_action] = sys_smbus_action;
	  syscalls_initialized = 1;
	 }
  }


  if (! error_return) {
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))

    proc_smbus = create_proc_entry("smbus", 0, 0);
    if (proc_smbus) {
      proc_smbus->read_proc = get_smbus_list;
    } else {
      printk("Couldn't create /proc/smbus\n");
      SMBus_Cleanup();
      error_return = -ENOENT;
    }
#else
    if (proc_register_dynamic(&proc_root, &dir)) {
      printk("Couldn't create /proc/smbus\n");
      SMBus_Cleanup();
      error_return = -ENOENT;
    } else
      inodeval = dir.low_ino;
#endif
  }
  return error_return;
}

/*
 *      Clean-up the SMBus code.
 */
void SMBus_Cleanup(void){
	if (SMBus_Initialized) {
        	release_region(PIIX4_smba, 8); 
		SMBus_Initialized=0;
	}
	if (syscalls_initialized) {
	 sys_call_table[__NR_smbus_action] = NULL;
	 syscalls_initialized = 0;
	}

#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
        if (proc_smbus)
                proc_unregister(&proc_root, proc_smbus->low_ino);
#else
        if (inodeval)
                proc_unregister(&proc_root, inodeval);
#endif

}

asmlinkage int sys_smbus_action(u8 addr, char read_write, u8 command, 
                                int size, union SMBus_Data *data)
{
 
  int res,datasize;
  union SMBus_Data temp;

#ifdef DEBUG
  printk("Syscall made on SMBus module.  Parameters:\n");
  printk("  addr:0x%X read/write:%d command:0x%X size:0x%X data:%p\n", 
         addr,read_write,command,size,data);
#endif

  if (!suser()) { /* Do we need a securelevel or capabilities check? */
#ifdef DEBUG
    printk("Only root may do this.\n");
#endif
    return -EPERM;
  }

  if ((size != SMBUS_BYTE) && (size != SMBUS_QUICK) &&
     (size != SMBUS_BYTE_DATA) && (size != SMBUS_WORD_DATA) &&
     (size != SMBUS_BLOCK_DATA)) { 
#ifdef DEBUG
    printk("SMBus: sys_smbus_action: size out of range (%x).\n",size);
#endif
    return -EINVAL;
  }

  if (addr > 0x7f) {
#ifdef DEBUG
     printk("SMBus: sys_smbus_action: addr out of range (%x).\n",addr);
#endif
   return -EINVAL;
  }

  /* Note that SMBUS_READ and SMBUS_WRITE are 0 and 1, so the check is
     valid if size==SMBUS_QUICK too. */
  if ((read_write != SMBUS_READ) && (read_write != SMBUS_WRITE)) {
#ifdef DEBUG
    printk("SMBus: sys_smbus_action: read_write out of range. (%x)\n",
           read_write);
#endif
    return -EINVAL;
  }

  /* Note that command values are always valid! */

  if ((size == SMBUS_QUICK) || 
      ((size == SMBUS_BYTE) && (read_write == SMBUS_WRITE)))
    /* These are special: we do not use data */
    return SMBus_Access(addr,read_write,command,size,NULL);
  else {
    if (data == NULL) {
#ifdef DEBUG
      printk("SMBus: sys_smbus_action: data is NULL pointer.\n");
#endif
      return -EINVAL;
    }
    
    /* This seems unlogical but it is not: if the user wants to read a
       value, we must write that value to user memory! */
    res = (read_write == SMBUS_WRITE)?VERIFY_READ:VERIFY_WRITE;

    if ((size == SMBUS_BYTE_DATA) || (size == SMBUS_BYTE))
      datasize = sizeof(data->byte);
    else if (size == SMBUS_WORD_DATA)
      datasize = sizeof(data->word);
    else /* size == SMBUS_BLOCK_DATA */ 
      datasize = sizeof(data->block);
  
    if (verify_area(res,data,datasize)) {
#ifdef DEBUG
      printk("SMBus: sys_smbus_action: invalid pointer data (%p).\n",data);
#endif
      return -EINVAL;
    }

    if (read_write == SMBUS_WRITE) {
      copy_from_user(&temp,data,datasize);
      return SMBus_Access(addr,read_write,command,size,&temp);
    } else {
      res = SMBus_Access(addr,read_write,command,size,&temp);
      if (!res)
        copy_to_user(data,&temp,datasize);
      return res;
    }
  }
}


/*
 *      Software pause .
 */
void do_pause( unsigned int amount )
{
      current->state = TASK_INTERRUPTIBLE;
      current->timeout = jiffies + amount;
      schedule();
}

int SMBus_Transaction(void) 
{
  int temp;
  int result=0;
  int timeout=0;

  /* Make sure the SMBus host is ready to start transmitting */
  if ((temp = inb_p(SMBHSTSTS)) != 0x00) {
#ifdef DEBUG
    printk("SMBus: SMBus_Read: SMBus busy (%02x). Resetting... ",temp);
#endif
    outb_p(temp, SMBHSTSTS);
    if ((temp = inb_p(SMBHSTSTS)) != 0x00) {
#ifdef DEBUG
      printk("Failed! (%02x)\n",temp);
#endif
      return -1;
    } else {
#ifdef DEBUG
      printk("Successfull!\n");
#endif
    }
  }

  /* start the transaction by setting bit 6 */
  outb_p(inb(SMBHSTCNT) | 0x040, SMBHSTCNT); 

  /* Wait for a fraction of a second! (See PIIX4 docs errata) */
  do_pause(1);

  /* Poll Host_busy bit */
  temp=inb_p(SMBHSTSTS) & 0x01;
  while (temp & (timeout++ < MAX_TIMEOUT)) {
    /* Wait for a while and try again*/
    do_pause(1);
    temp = inb_p(SMBHSTSTS) & 0x01;
  }

  /* If the SMBus is still busy, we give up */
  if (timeout >= MAX_TIMEOUT) {
#ifdef DEBUG
    printk("SMBus: SMBus_Read: SMBus Timeout!\n"); 
    result = -1;
#endif
  }

  temp = inb_p(SMBHSTSTS);

  if (temp  & 0x10) {
    result = -1;
#ifdef DEBUG
    printk("SMBus error: Failed bus transaction\n");
#endif
  }

  if (temp & 0x08) {
    result = -1;
    printk("SMBus error: Bus collision! SMBus may be locked until next hard reset. (sorry!)\n");
    /* Clock stops and slave is stuck in mid-transmission */
  }

  if (temp & 0x04) {
    result = -1;
#ifdef DEBUG
    printk("SMBus error: no response!\n");
#endif
  }

  if (inb_p(SMBHSTSTS) != 0x00)
    outb_p( inb(SMBHSTSTS), SMBHSTSTS);

  if ((temp = inb_p(SMBHSTSTS)) != 0x00) {
#ifdef DEBUG
    printk("SMBus error: Failed reset at end of transaction (%02x)\n",temp);
#endif
  }
  return result;
}

int SMBus_Access(u8 addr, char read_write, u8 command, int size, 
                 union SMBus_Data *data)
{
  int i,len;

  down(&smbus_sem);

  outb_p((size & 0x1C) + (ENABLE_INT9 & 1), SMBHSTCNT);

  switch(size) {
    case SMBUS_QUICK:
      outb_p(((addr & 0x7f) << 1) | (read_write & 0x01), SMBHSTADD);
      break;
    case SMBUS_BYTE:
      outb_p(((addr & 0x7f) << 1) | (read_write & 0x01), SMBHSTADD);
      if (read_write == SMBUS_WRITE)
        outb_p(command, SMBHSTCMD);
      break;
    case SMBUS_BYTE_DATA:
      outb_p(((addr & 0x7f) << 1) | (read_write & 0x01), SMBHSTADD);
      outb_p(command, SMBHSTCMD);
      if (read_write == SMBUS_WRITE)
        outb_p(data->byte,SMBHSTDAT0);
      break;
    case SMBUS_WORD_DATA:
      outb_p(((addr & 0x7f) << 1) | (read_write & 0x01), SMBHSTADD);
      outb_p(command, SMBHSTCMD);
      if (read_write == SMBUS_WRITE) {
        outb_p(data->word & 0xff,SMBHSTDAT0);
        outb_p((data->word & 0xff00) >> 8,SMBHSTDAT1);
      }
      break;
    case SMBUS_BLOCK_DATA:
      outb_p(((addr & 0x7f) << 1) | (read_write & 0x01), SMBHSTADD);
      outb_p(command, SMBHSTCMD);
      if (read_write == SMBUS_WRITE) {
        len = data->block[0];
        if (len < 0) 
          len = 0;
        if (len > 32)
          len = 32;
        outb_p(len,SMBHSTDAT0);
        i = inb_p(SMBHSTCNT); /* Reset SMBBLKDAT */
        for (i = 1; i <= len; i ++)
          outb_p(data->block[i],SMBBLKDAT);
        break;
      }
  }

  if (SMBus_Transaction()) { /* Error in transaction */ 
    up(&smbus_sem);
    return -1; 
  }

  if ((read_write == SMBUS_WRITE) || (size == SMBUS_QUICK)) {
    up(&smbus_sem);
    return 0;
  }

  switch(size) {
    case SMBUS_BYTE: /* Where is the result put? I assume here it is in 
                        SMBHSTDAT0 but it might just as well be in the
                        SMBHSTCMD. No clue in the docs */
      
      data->byte = inb_p(SMBHSTDAT0);
      break;
    case SMBUS_BYTE_DATA:
      data->byte = inb_p(SMBHSTDAT0);
      break;
    case SMBUS_WORD_DATA:
      data->word = inb_p(SMBHSTDAT0) + (inb_p(SMBHSTDAT1) << 8);
      break;
    case SMBUS_BLOCK_DATA:
      data->block[0] = inb_p(SMBHSTDAT0);
      i = inb_p(SMBHSTCNT); /* Reset SMBBLKDAT */
      for (i = 1; i <= data->block[0]; i++)
        data->block[i] = inb_p(SMBBLKDAT);
      break;
  }
  up(&smbus_sem);
  return 0;
}


/*
 * This generates the report for /proc/smbus
 */
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
int get_smbus_list(char *buf, char **start, off_t offset, int len, int *eof, 
                   void *private)
#else
int get_smbus_list(char *buf, char **start, off_t offset, int len, int unused)
#endif
{
        smbus_entry_t *p;

        len = 0;

        for (p = smbuslist.next; (p) && (len < 4000); p = p->next)
                len += sprintf(buf+len, "%02x : %s\n", p->address, p->name);
        if (p)
                len += sprintf(buf+len, "4K limit reached!\n");
        return len;
}

/*
 * The workhorse function: find where to put a new entry
 */
smbus_entry_t *find_gap(smbus_entry_t *root, u8 address)
{
        smbus_entry_t *p;
        unsigned long flags;
	
        save_flags(flags);
        cli();
        for (p = root; ; p = p->next) {
                if ((p != root) && (p->address >= address)) {
                        p = NULL;
                        break;
                }
                if ((p->next == NULL) || (p->next->address > address))
                        break;
        }
        restore_flags(flags);
        return p;
}

void request_smbus(u8 address, const char *name)
{
        smbus_entry_t *p;
        int i;

	if (! name)
		return;

        for (i = 0; i < SMBUSTABLE_SIZE; i++)
                if (smbustable[i].name == NULL)
                        break;
        if (i == SMBUSTABLE_SIZE)
                printk("warning: smbusport table is full\n");
        else {
                p = find_gap(&smbuslist, address);
                if (p == NULL)
                        return;
                smbustable[i].name = name;
                smbustable[i].address = address;
                smbustable[i].next = p->next;
                p->next = &smbustable[i];
                return;
        }
}

/*
 * Call this when the device driver is unloaded
 */
void release_smbus(u8 address)
{
        smbus_entry_t *p, *q;

        for (p = &smbuslist; ; p = q) {
                q = p->next;
                if (q == NULL)
                        break;
                if (q->address == address) {
                        q->name = NULL;
                        p->next = q->next;
                        return;
                }
        }
}

/*
 * Call this to check the ioport region before probing
 */
int check_smbus(u8 address)
{
        return (find_gap(&smbuslist, address) == NULL) ? -EBUSY : 0;
}

