/*
    main.h - Header file for the Linux sensor programs
    Copyright (c) 1997, 1998  Alexander Larsson <alla@lysator.liu.se>,
    Frodo Looijaard <frodol@dds.nl> and Philip Edelbrock <phil@netroedge.com>.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* This file contains all external definition used throughout the module
   sources, as well as several compatibility macro's for older kernels. */

#ifndef LM78_MAIN_H
#define LM78_MAIN_H

#include <linux/types.h>

/* In lm75.c */
EXTERN int LM75_Init (void);
EXTERN void LM75_Cleanup (void);
EXTERN int LM75_Print_Values (char *buf);
EXTERN void LM75_Update_Values (void);
EXTERN void LM75_Set(int what, int nr, long value);
EXTERN long LM75_Get(int what, int nr);


/* In lm78.c */
EXTERN int LM78_Init(void);
EXTERN void LM78_Cleanup(void);
EXTERN int LM78_Print_Values (char *buf);
EXTERN void LM78_Update_Values(void);
EXTERN void LM78_Set(int what, int nr, long value);
EXTERN long LM78_Get(int what, int nr);

/* In sysctl.c */
EXTERN int LM_Sysctl_Init(void);
EXTERN void LM_Sysctl_Cleanup(void);

/* In main.c */
EXTERN void LM_Update_Data(void);

struct lm75_data {
  u16 temp;        /* Last gathered temperature. Format: lm75 encoding */

  u16 temphyst;    /* Max. temperature. Format: lm75 encoding */
  u16 tempos;      /* Min. temperature. Format: lm75 encoding */

#ifdef DEBUG
  u8 config;      /* Last gathered LM75 config. Format: lm75 encoding */
#endif

  /* We may need to add a correction factor to this! */

};

#define MAX_LM75 8
#define LM75_NOT_PRESENT 0xffff

/* The maximum length a label can get (including the #0 at the end!) */
#define LABEL_LENGTH 7

struct LM_Sensor_Present {
  u32 in;          /* For each voltage, whether is must be read and printed.
                      Format: bitvector. */
  u32 fan;         /* For each fan, whether it must be read and printed.
                      Format: bitvector. */
  u8 vid;          /* Vid lines present. Format: boolean */
  u8 temp_mb;      /* Motherboard temperature present. Format: boolean */
  u32 lm75;        /* For each LM75, a bit which tells us the LM75
                      is available. Format: bitvector. */
};

struct LM_Sensor_Config {
  char in_label[7][LABEL_LENGTH]; 
                   /* Labels for the voltages. Format: strings */
  int in_conv[7];  /* Conversion values for voltages. Format: 10000 times
                      the normalization value */

  char fan_label[3][LABEL_LENGTH];
                   /* Labels for the fans. Format: strings */

  char lm75_label[MAX_LM75][LABEL_LENGTH];
                   /* Labels for the LM75 sensors. Format: strings */
  
  struct LM_Sensor_Present selected;
                   /* The user-selected hardware which is present in 
                      /proc/sensors */
};


struct LM_Sensor_Data {


  int valid;       /* Data in this structure is valid. Format: boolean */

  /* Read values */

  unsigned long last_read;

  u8 in[7];        /* Last gathered voltages. Format: lm78 encoding */
  u8 in_max[7];    /* Max limits for voltages. Format: lm78 encoding */
  u8 in_min[7];    /* Min limits for voltages. Format: lm78 encoding */

  u8 fan[3];       /* Last gathered fan speeds. Format: lm78 encoding */
  u8 fan_min[3];   /* Min limits for fan speeds. Format: lm78 encoding */
  u8 fan_div[2];   /* The division factors for fan speeds. 
                      Format: lm78 encoding */
  
  u8 vid;          /* Last gathered vid lines core voltage. 
                      Format: lm78 encoding */

  u8 temp_mb;      /* Last gathered motherboard temperature. 
                      Format: lm78 encoding */
  u8 temp_mb_max;  /* Max limit for motherboard temperature.  
                      Format: lm78 encoding */
  u8 temp_mb_min;  /* Min limit for motherboard temperature. 
                      Format: lm78 encoding */

  u8 interrupt_status[2];
                   /* Interrupt status lines. Format: lm78 encoding. */
  
  struct lm75_data lm75 [MAX_LM75];

};

EXTERN struct LM_Sensor_Data LM_Sensor_Data;
EXTERN struct LM_Sensor_Config LM_Sensor_Config;
EXTERN struct LM_Sensor_Present LM_Sensor_Present;

#endif /* LM78_MAIN_H */

