/*
    sysctl.c - Part of a Linux module for reading sensor data.
    Copyright (c) 1997, 1998  Alexander Larsson <alla@lysator.liu.se>,
    Frodo Looijaard <frodol@dds.nl> and Philip Edelbrock <phil@netroedge.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/config.h>

/* The next define disables the redefinition of kernel_version. As it is
   already defined and initialized in lm78.c, this would result in twice
   defined symbols. We must include <linux/version.h> by hand now. */
#define __NO_VERSION__

/* Hack to allow the use of CONFIG_MODVERSIONS. In the kernel itself, this
   is arranged from the main Makefile. */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/version.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/proc_fs.h>
#include <linux/ctype.h>
#include <linux/types.h>   /* Needed for sysctl.h */
#include <linux/linkage.h> /* Needed for sysctl.h */
#include <linux/sysctl.h>
#include "lm_sensors.h"
#include "main.h"
#include "lm78.h"
#include "lm75.h"
#include "compat.h"

#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,1,0))
#include <asm/uaccess.h>
#endif

static struct ctl_table entry(int id,const char *filename);
static struct ctl_table empty_entry(void);
static struct ctl_table entry_str(int id,const char *filename);
static int lm_proc (ctl_table *ctl, int write, struct file * filp,
                    void *buffer, size_t *lenp);
static int lm_proc_str (ctl_table *ctl, int write, struct file * filp,
                        void *buffer, size_t *lenp);
static int lm_sysctl (ctl_table *table, int *name, int nlen,
                      void *oldval, size_t *oldlenp,
                      void *newval, size_t newlen,
                      void **context);
static int lm_sysctl_str (ctl_table *table, int *name, int nlen, void *oldval, 
                          size_t *oldlenp, void *newval, size_t newlen,
                          void **context);
static void write_in(int nr, int nrels, long *results);
static void read_in(int nr, long *results);
static void write_fan(int nr, int nrels, long *results);
static void read_fan(int nr, long *results);
static void write_LM75(int nr, int nrels, long *results);
static void read_LM75(int nr, long *results);
static void write_temp(int nrels, long *results);
static void read_temp(long *results);
static void write_in_conv(int nrels, long *results);
static void read_in_conv(long *results);
static void write_present(int nrels, long *results);
static void read_present(long *results);
static void write_fan_div(int nrels, long *results);
static void read_fan_div(long *results);
static void read_fan_labels(char (*results)[LABEL_LENGTH]);
static void write_fan_labels(int nrels,char (*results)[LABEL_LENGTH]);
static void read_in_labels(char (*results)[LABEL_LENGTH]);
static void write_in_labels(int nrels,char (*results)[LABEL_LENGTH]);
static void read_lm75_labels(char (*results)[LABEL_LENGTH]);
static void write_lm75_labels(int nrels,char (*results)[LABEL_LENGTH]);
static void read_vid(long *results);
static void write_vid(int nrels, long *results);
static void read_alarm(long *results);
static void write_alarm(int nrels, long *results);
static void parse_elements(int *nrels, void *buffer, int bufsize, 
                           long *results, int magnitude);
static void write_elements(int nrels,void *buffer,int *bufsize,
                           long *results,int mag);
static void parse_strings(int *nrels, void *buffer, int bufsize, 
                          char (*results)[LABEL_LENGTH]);
static void write_strings(int nrels,void *buffer,int *bufsize,
                          char (*results)[LABEL_LENGTH]);

static ctl_table lm_lm_table[30] = { 
  { LMT_PRESENT, "config-present", NULL, 0, 0644, NULL, &lm_proc, &lm_sysctl },
  { LMT_ALARM, "alarm", NULL, 0, 0644, NULL, &lm_proc, &lm_sysctl }
};

static ctl_table lm_dev_table[] = {
  { DEV_LMT, "lm_sensors", NULL, 0, 0555, lm_lm_table },
  { 0 }
};

static ctl_table lm_root_table[] = {
  { CTL_DEV, "dev", NULL, 0, 0555, lm_dev_table },
  { 0 }
};

static struct ctl_table_header *lm_table_header;

/* Note that the triple zeros at the end are needed to stop gcc-2.7.2.3
   from generating a memset, which will be unidentified at insertion-time. */
struct ctl_table entry(int id,const char *filename)
{
  struct ctl_table res = { id, filename, NULL, 0, 0644, NULL, &lm_proc, 
                           &lm_sysctl,0,0,0 };
  return res;
}

struct ctl_table entry_str(int id,const char *filename)
{
  struct ctl_table res = { id, filename, NULL, 0, 0644, NULL, &lm_proc_str, 
                           &lm_sysctl_str,0,0,0 };
  return res;
}

struct ctl_table empty_entry(void)
{
  struct ctl_table res = { 0 };
  return res;
}


/* Call this function only after all hardware detection has finished, and
   LM_Sensor_Present contains valid data! */
int LM_Sysctl_Init(void) 
{
  int i=2;

  if (LM_Sensor_Present.in) {
    lm_lm_table[i++] = entry(LMT_IN_CONV,"config-in-conv");
    lm_lm_table[i++] = entry_str(LMT_IN_LABELS,"config-in-labels");
  }
  if (LM_Sensor_Present.fan) {
    lm_lm_table[i++] = entry(LMT_FAN_DIV, "config-fan-div");
    lm_lm_table[i++] = entry_str(LMT_FAN_LABELS, "config-fan-labels");
  }
  if (LM_Sensor_Present.lm75) 
    lm_lm_table[i++] = entry_str(LMT_TEMP_LABELS, "config-temp-labels");

  if (LM_Sensor_Present.in & 0x01)
    lm_lm_table[i++] = entry(LMT_IN0,"in0");
  if (LM_Sensor_Present.in & 0x02)
    lm_lm_table[i++] = entry(LMT_IN1,"in1");
  if (LM_Sensor_Present.in & 0x04)
    lm_lm_table[i++] = entry(LMT_IN2,"in2");
  if (LM_Sensor_Present.in & 0x08)
    lm_lm_table[i++] = entry(LMT_IN3,"in3");
  if (LM_Sensor_Present.in & 0x10)
    lm_lm_table[i++] = entry(LMT_IN4,"in4");
  if (LM_Sensor_Present.in & 0x20)
    lm_lm_table[i++] = entry(LMT_IN5,"in5");
  if (LM_Sensor_Present.in & 0x40)
    lm_lm_table[i++] = entry(LMT_IN6,"in6");

  if (LM_Sensor_Present.fan & 0x01)
    lm_lm_table[i++] = entry(LMT_FAN1,"fan1");
  if (LM_Sensor_Present.fan & 0x02)
    lm_lm_table[i++] = entry(LMT_FAN2,"fan2");
  if (LM_Sensor_Present.fan & 0x04)
    lm_lm_table[i++] = entry(LMT_FAN3,"fan3");

  if (LM_Sensor_Present.lm75 & 0x01)
    lm_lm_table[i++] = entry(LMT_TEMP_1,"temp-1");
  if (LM_Sensor_Present.lm75 & 0x02)
    lm_lm_table[i++] = entry(LMT_TEMP_2,"temp-2");
  if (LM_Sensor_Present.lm75 & 0x04)
    lm_lm_table[i++] = entry(LMT_TEMP_3,"temp-3");
  if (LM_Sensor_Present.lm75 & 0x08)
    lm_lm_table[i++] = entry(LMT_TEMP_4,"temp-4");
  if (LM_Sensor_Present.lm75 & 0x10)
    lm_lm_table[i++] = entry(LMT_TEMP_5,"temp-5");
  if (LM_Sensor_Present.lm75 & 0x20)
    lm_lm_table[i++] = entry(LMT_TEMP_6,"temp-6");
  if (LM_Sensor_Present.lm75 & 0x40)
    lm_lm_table[i++] = entry(LMT_TEMP_7,"temp-7");
  if (LM_Sensor_Present.lm75 & 0x80)
    lm_lm_table[i++] = entry(LMT_TEMP_8,"temp-8");

  if (LM_Sensor_Present.vid)
    lm_lm_table[i++] = entry(LMT_VID,"vid");

  if (LM_Sensor_Present.temp_mb)
    lm_lm_table[i++] = entry(LMT_TEMP_MB,"temp-mb");

  /* Just to make double sure */
    lm_lm_table[i] = empty_entry();

  lm_table_header = register_sysctl_table(lm_root_table,0);
  return lm_table_header?0:-ENOMEM;
}

void LM_Sysctl_Cleanup(void)
{
  if (lm_table_header)
    unregister_sysctl_table(lm_table_header);
}

int lm_proc (ctl_table *ctl, int write, struct file * filp,
             void *buffer, size_t *lenp)
{
  int nrels,mag;
  long results[7];

  /* If buffer is size 0, or we try to read when not at the start, we 
     return nothing. Note that I think writing when not at the start
     does not work either, but anyway, this is straight from the kernel
     sources. */
  if (!*lenp || (filp->f_pos && !write)) {
    *lenp = 0;
    return 0;
  }

  /* How many numbers are found within these files, and how to scale them? */
  switch (ctl->ctl_name) {
    case LMT_IN0: case LMT_IN1: case LMT_IN2: case LMT_IN3: case LMT_IN4:
    case LMT_IN5: case LMT_IN6:
      nrels=3;
      mag=2;
      break;
    case LMT_TEMP_MB:
      nrels=3;
      mag=0;
      break;
    case LMT_FAN1: case LMT_FAN2: case LMT_FAN3:
      nrels=2;
      mag=0;
      break;
    case LMT_TEMP_1: case LMT_TEMP_2: case LMT_TEMP_3: case LMT_TEMP_8:
    case LMT_TEMP_4: case LMT_TEMP_5: case LMT_TEMP_6: case LMT_TEMP_7:
      nrels=3;
      mag=1;
      break;
    case LMT_IN_CONV:
      nrels=7;
      mag=4;
      break;
    case LMT_FAN_DIV:
      nrels=3;
      mag=0;
      break;
    case LMT_PRESENT:
      nrels=5;
      mag=0;
      break;
    case LMT_VID:
      nrels=1;
      mag=2;
      break;
    case LMT_ALARM:
      nrels=1;
      mag=0;
      break;
    default: /* Should never be called */
      return -EINVAL;
  }
 
  /* OK, try writing stuff. */
  if (write) {
    parse_elements(&nrels,buffer,*lenp,results,mag);
    if (nrels == 0)
      return 0;
    switch (ctl->ctl_name) {
      case LMT_IN0: write_in(0,nrels,results); break;
      case LMT_IN1: write_in(1,nrels,results); break;
      case LMT_IN2: write_in(2,nrels,results); break;
      case LMT_IN3: write_in(3,nrels,results); break;
      case LMT_IN4: write_in(4,nrels,results); break;
      case LMT_IN5: write_in(5,nrels,results); break;
      case LMT_IN6: write_in(6,nrels,results); break;
      case LMT_FAN1: write_fan(1,nrels,results); break;
      case LMT_FAN2: write_fan(2,nrels,results); break;
      case LMT_FAN3: write_fan(3,nrels,results); break;
      case LMT_TEMP_1: write_LM75(0,nrels,results);break;
      case LMT_TEMP_2: write_LM75(1,nrels,results);break;
      case LMT_TEMP_3: write_LM75(2,nrels,results);break;
      case LMT_TEMP_4: write_LM75(3,nrels,results);break;
      case LMT_TEMP_5: write_LM75(4,nrels,results);break;
      case LMT_TEMP_6: write_LM75(5,nrels,results);break;
      case LMT_TEMP_7: write_LM75(6,nrels,results);break;
      case LMT_TEMP_8: write_LM75(7,nrels,results);break;
      case LMT_TEMP_MB: write_temp(nrels,results); break;
      case LMT_IN_CONV: write_in_conv(nrels,results);break;
      case LMT_FAN_DIV: write_fan_div(nrels,results);break;
      case LMT_PRESENT: write_present(nrels,results);break;
      case LMT_VID: write_vid(nrels,results);break;
      case LMT_ALARM: write_alarm(nrels,results);break;
      default: /* Should never be called */ *lenp=0; return -EINVAL; break;
    }
    filp->f_pos += *lenp;
    return 0;
  } else { /* read */
    /* Update all values in LM_Sensor_Data */
    if ((ctl->ctl_name != LMT_IN_CONV) && (ctl->ctl_name != LMT_PRESENT))
      LM_Update_Data();

    /* Read the values to print into results */
    switch (ctl->ctl_name) {
      case LMT_IN0: read_in(0,results);break;
      case LMT_IN1: read_in(1,results);break;
      case LMT_IN2: read_in(2,results);break;
      case LMT_IN3: read_in(3,results);break;
      case LMT_IN4: read_in(4,results);break;
      case LMT_IN5: read_in(5,results);break;
      case LMT_IN6: read_in(6,results);break;
      case LMT_FAN1: read_fan(1,results);break;
      case LMT_FAN2: read_fan(2,results);break;
      case LMT_FAN3: read_fan(3,results);break;
      case LMT_TEMP_1: read_LM75(0,results);break;
      case LMT_TEMP_2: read_LM75(1,results);break;
      case LMT_TEMP_3: read_LM75(2,results);break;
      case LMT_TEMP_4: read_LM75(3,results);break;
      case LMT_TEMP_5: read_LM75(4,results);break;
      case LMT_TEMP_6: read_LM75(5,results);break;
      case LMT_TEMP_7: read_LM75(6,results);break;
      case LMT_TEMP_8: read_LM75(7,results);break;
      case LMT_TEMP_MB: read_temp(results);break;
      case LMT_IN_CONV: read_in_conv(results);break;
      case LMT_FAN_DIV: read_fan_div(results);break;
      case LMT_PRESENT: read_present(results);break;
      case LMT_VID: read_vid(results);break;
      case LMT_ALARM: read_alarm(results);break;
      default: /* Should never be called */ return -EINVAL;
    }
    /* OK, print it now */
    write_elements(nrels,buffer,lenp,results,mag);
    filp->f_pos += *lenp;
    return 0;
  }
}

int lm_proc_str (ctl_table *ctl, int write, struct file * filp,
                 void *buffer, size_t *lenp)
{
  char results[8][LABEL_LENGTH];
  int nrels;

  /* If buffer is size 0, or we try to read when not at the start, we
     return nothing. Note that I think writing when not at the start
     does not work either, but anyway, this is straight from the kernel
     sources. */
  if (!*lenp || (filp->f_pos && !write)) {
    *lenp = 0;
    return 0;
  }

  switch (ctl->ctl_name) {
    case LMT_IN_LABELS:
      nrels = 7; break;
    case LMT_FAN_LABELS:
      nrels = 3; break;
    case LMT_TEMP_LABELS:
      nrels = 8; break;
    default: /* Should never happen! */ {
      *lenp=0;
      return -EINVAL;
    }
  }
  
  if (write) {
    parse_strings(&nrels,buffer,*lenp,results);
    if (nrels == 0)
      return 0;
    switch (ctl->ctl_name) {
      case LMT_IN_LABELS: write_in_labels(nrels,results); break;
      case LMT_FAN_LABELS: write_fan_labels(nrels,results); break;
      case LMT_TEMP_LABELS: write_lm75_labels(nrels,results); break;
      default:  /* Should never be called */ return -EINVAL; break;
    }
    filp->f_pos += *lenp;
    return -EINVAL;
  } else { /* read */
    switch (ctl->ctl_name) {
      case LMT_IN_LABELS: read_in_labels(results); break;
      case LMT_FAN_LABELS: read_fan_labels(results); break;
      case LMT_TEMP_LABELS: read_lm75_labels(results); break;
      default: /* Should never be called */ return -EINVAL;
    }
    /* OK, print it now */
    write_strings(nrels,buffer,lenp,results);
    filp->f_pos += *lenp;
    return 0;
  }
}


int lm_sysctl (ctl_table *table, int *name, int nlen, void *oldval, 
               size_t *oldlenp, void *newval, size_t newlen,
               void **context)
{
  long results[7];
  int nrels,oldlen;
 
  /* How many numbers are found within these files, and how to scale them? */
  switch (table->ctl_name) {
    case LMT_IN0: case LMT_IN1: case LMT_IN2: case LMT_IN3: case LMT_IN4:
    case LMT_FAN_DIV:
    case LMT_IN5: case LMT_IN6: case LMT_TEMP_MB:
    case LMT_TEMP_8: case LMT_TEMP_1: case LMT_TEMP_2: case LMT_TEMP_3:
    case LMT_TEMP_4: case LMT_TEMP_5: case LMT_TEMP_6: case LMT_TEMP_7:
      nrels=3;
      break;
    case LMT_FAN1: case LMT_FAN2: case LMT_FAN3:
      nrels=2;
      break;
    case LMT_IN_CONV:
      nrels=7;
      break;
    case LMT_PRESENT:
      nrels=5;
      break;
    case LMT_VID: case LMT_ALARM:
      nrels=1;
      break;
    default: /* Should never be called */
      return -EINVAL;
  }

  /* Check if we need to output the old values */
  if (oldval && oldlenp && ! get_user_data(oldlen,oldlenp) && oldlen) {

    /* Update all values in LM_Sensor_Data */
    if ((table->ctl_name != LMT_IN_CONV) && (table->ctl_name != LMT_PRESENT))
      LM_Update_Data();

    switch (table->ctl_name) {
      case LMT_IN0: read_in(0,results);break;
      case LMT_IN1: read_in(1,results);break;
      case LMT_IN2: read_in(2,results);break;
      case LMT_IN3: read_in(3,results);break;
      case LMT_IN4: read_in(4,results);break;
      case LMT_IN5: read_in(5,results);break;
      case LMT_IN6: read_in(6,results);break;
      case LMT_FAN1: read_fan(1,results);break;
      case LMT_FAN2: read_fan(2,results);break;
      case LMT_FAN3: read_fan(3,results);break;
      case LMT_TEMP_1: read_LM75(0,results);break;
      case LMT_TEMP_2: read_LM75(1,results);break;
      case LMT_TEMP_3: read_LM75(2,results);break;
      case LMT_TEMP_4: read_LM75(3,results);break;
      case LMT_TEMP_5: read_LM75(4,results);break;
      case LMT_TEMP_6: read_LM75(5,results);break;
      case LMT_TEMP_7: read_LM75(6,results);break;
      case LMT_TEMP_8: read_LM75(7,results);break;
      case LMT_TEMP_MB: read_temp(results);break;
      case LMT_IN_CONV: read_in_conv(results);break;
      case LMT_FAN_DIV: read_fan_div(results);break;
      case LMT_PRESENT: read_present(results);break;
      case LMT_VID: read_vid(results);break;
      case LMT_ALARM: read_alarm(results);break;
      default: /* Should never be called */ return -EINVAL;
    }
    
    /* Note the rounding factor! */
    if (nrels * sizeof(long) < oldlen)
      oldlen = nrels * sizeof(long);
    oldlen = (oldlen / sizeof(long)) * sizeof(long);
    copy_to_user(oldval,results,oldlen);
    put_user(oldlen,oldlenp);
  }

  /* Check to see whether we need to read the new values */
  if (newval && newlen) {
    if (nrels * sizeof(long) < newlen)
      newlen = nrels * sizeof(long);
    nrels = newlen / sizeof(long);
    newlen = (newlen / sizeof(long)) * sizeof(long);
    copy_from_user(results,newval,newlen);
    
    switch (table->ctl_name) {
      case LMT_IN0: write_in(0,nrels,results); break;
      case LMT_IN1: write_in(1,nrels,results); break;
      case LMT_IN2: write_in(2,nrels,results); break;
      case LMT_IN3: write_in(3,nrels,results); break;
      case LMT_IN4: write_in(4,nrels,results); break;
      case LMT_IN5: write_in(5,nrels,results); break;
      case LMT_IN6: write_in(6,nrels,results); break;
      case LMT_FAN1: write_fan(1,nrels,results); break;
      case LMT_FAN2: write_fan(2,nrels,results); break;
      case LMT_FAN3: write_fan(3,nrels,results); break;
      case LMT_TEMP_1: write_LM75(0,nrels,results);break;
      case LMT_TEMP_2: write_LM75(1,nrels,results);break;
      case LMT_TEMP_3: write_LM75(2,nrels,results);break;
      case LMT_TEMP_4: write_LM75(3,nrels,results);break;
      case LMT_TEMP_5: write_LM75(4,nrels,results);break;
      case LMT_TEMP_6: write_LM75(5,nrels,results);break;
      case LMT_TEMP_7: write_LM75(6,nrels,results);break;
      case LMT_TEMP_8: write_LM75(7,nrels,results);break;
      case LMT_TEMP_MB: write_temp(nrels,results); break;
      case LMT_IN_CONV: write_in_conv(nrels,results);break;
      case LMT_FAN_DIV: write_fan_div(nrels,results);break;
      case LMT_PRESENT: write_present(nrels,results);break;
      case LMT_VID: write_vid(nrels,results);break;
      case LMT_ALARM: write_alarm(nrels,results);break;
      default: /* Should never be called */ return -EINVAL; break;
    }
  }
  return 1; /* We have done all the work */
}

int lm_sysctl_str (ctl_table *table, int *name, int nlen, void *oldval, 
                   size_t *oldlenp, void *newval, size_t newlen,
                   void **context)
{
  char results[8][LABEL_LENGTH];
  int nrels,oldlen;

  switch (table->ctl_name) {
    case LMT_IN_LABELS:
      nrels = 7; break;
    case LMT_FAN_LABELS:
      nrels = 3; break;
    case LMT_TEMP_LABELS:
      nrels = 8; break;
    default: /* Should never happen */
      return -EINVAL;
  }
 
  /* Check if we need to output the old values */
  if (oldval && oldlenp && ! get_user_data(oldlen,oldlenp) && oldlen) {

    /* Read the strings to write in results */
    switch (table->ctl_name) {
      case LMT_IN_LABELS: read_in_labels(results); break;
      case LMT_FAN_LABELS: read_fan_labels(results); break;
      case LMT_TEMP_LABELS: read_lm75_labels(results); break;
      default: /* Should never be called */ return -EINVAL;
    }

    write_strings(nrels,oldval,&oldlen,results);
    
    /* Put the string terminator, but not beyond the end of oldvar! 
       We overwrite the \n here */
    put_user(0,(char *) oldval + oldlen - 1);
    put_user(oldlen,oldlenp);
  }

  /* Now use the new values */
  if (newval && newlen) {
    parse_strings(&nrels,newval,newlen,results);

    switch (table->ctl_name) {
      case LMT_IN_LABELS: write_in_labels(nrels,results); break;
      case LMT_FAN_LABELS: write_fan_labels(nrels,results); break;
      case LMT_TEMP_LABELS: write_lm75_labels(nrels,results); break;
      default:  /* Should never be called */ return -EINVAL; break;
    }
  }
  return 1;
}
 
void write_in(int nr, int nrels, long *results) 
{
  if (nrels >= 1)
    LM78_Set(LM_PROBE_IN_MIN,nr,results[0]);
  if (nrels >= 2)
    LM78_Set(LM_PROBE_IN_MAX,nr,results[1]);
}

void read_in(int nr, long *results)
{
  results[0]=LM78_Get(LM_PROBE_IN_MIN,nr);
  results[1]=LM78_Get(LM_PROBE_IN_MAX,nr);
  results[2]=LM78_Get(LM_PROBE_IN,nr);
}

void write_fan(int nr, int nrels, long *results)
{
  if (nrels >= 1)
    LM78_Set(LM_PROBE_FAN_MIN,nr,results[0]);
}

void read_fan(int nr, long *results)
{
  results[0]=LM78_Get(LM_PROBE_FAN_MIN,nr);
  results[1]=LM78_Get(LM_PROBE_FAN,nr);
}

void write_LM75(int nr, int nrels, long *results)
{
  if (nrels >= 1)
    LM75_Set(LM_PROBE_LM75_OS,nr,results[0]);
  if (nrels >= 2)
    LM75_Set(LM_PROBE_LM75_HYST,nr,results[1]);
}

void read_LM75(int nr, long *results)
{
  results[0]=LM75_Get(LM_PROBE_LM75_OS,nr);
  results[1]=LM75_Get(LM_PROBE_LM75_HYST,nr);
  results[2]=LM75_Get(LM_PROBE_LM75_VALUE,nr);
}

void write_temp(int nrels, long *results)
{
  if (nrels >= 1)
    LM78_Set(LM_PROBE_TEMP_MIN,0,results[0]);
  if (nrels >= 2)
    LM78_Set(LM_PROBE_TEMP_MAX,0,results[1]);
}

void read_temp(long *results)
{
  results[0]=LM78_Get(LM_PROBE_TEMP_MIN,0);
  results[1]=LM78_Get(LM_PROBE_TEMP_MAX,0);
  results[2]=LM78_Get(LM_PROBE_TEMP,0);
}

void write_fan_div(int nrels,long *results)
{
  if (nrels >= 1)
    LM78_Set(LM_PROBE_FAN_DIV,1,results[0]);
  if (nrels >= 2)
    LM78_Set(LM_PROBE_FAN_DIV,2,results[1]);
  if (nrels >= 3) /* Safe even if we have only 2 fans. */
    LM78_Set(LM_PROBE_FAN_DIV,3,results[2]);
}

void read_fan_div(long *results)
{
  results[0]=(LM_Sensor_Present.fan & 0x01)?LM78_Get(LM_PROBE_FAN_DIV,1):0;
  results[1]=(LM_Sensor_Present.fan & 0x02)?LM78_Get(LM_PROBE_FAN_DIV,2):0;
  results[2]=(LM_Sensor_Present.fan & 0x04)?LM78_Get(LM_PROBE_FAN_DIV,3):0;
}

void write_in_conv(int nrels,long *results)
{
  int i;

  /* Use sensible values */
  if (nrels >= 7) 
    nrels = 7;
  for (i = 0; i < nrels; i++)
    if (results[i] < 1000)
      results[i] = 1000;
    else if (results[i] > 100000)
      results[i] = 100000;
  
  if (nrels >= 1)
    LM_Sensor_Config.in_conv[0] = results[0];
  if (nrels >= 2)
    LM_Sensor_Config.in_conv[1] = results[1];
  if (nrels >= 3)
    LM_Sensor_Config.in_conv[2] = results[2];
  if (nrels >= 4)
    LM_Sensor_Config.in_conv[3] = results[3];
  if (nrels >= 5)
    LM_Sensor_Config.in_conv[4] = results[4];
  if (nrels >= 6)
    LM_Sensor_Config.in_conv[5] = results[5];
  if (nrels >= 7)
    LM_Sensor_Config.in_conv[6] = results[6];
}

/* Note: We always return 0 for IN sensors which are not available in 
   hardware! */
void read_in_conv(long *results)
{
  results[0]=(LM_Sensor_Present.in & 0x01)?LM_Sensor_Config.in_conv[0]:0;
  results[1]=(LM_Sensor_Present.in & 0x02)?LM_Sensor_Config.in_conv[1]:0;
  results[2]=(LM_Sensor_Present.in & 0x04)?LM_Sensor_Config.in_conv[2]:0;
  results[3]=(LM_Sensor_Present.in & 0x08)?LM_Sensor_Config.in_conv[3]:0;
  results[4]=(LM_Sensor_Present.in & 0x10)?LM_Sensor_Config.in_conv[4]:0;
  results[5]=(LM_Sensor_Present.in & 0x20)?LM_Sensor_Config.in_conv[5]:0;
  results[6]=(LM_Sensor_Present.in & 0x40)?LM_Sensor_Config.in_conv[6]:0;
  results[6]=LM_Sensor_Config.in_conv[6];
}

void write_present(int nrels, long *results)
{
  if (nrels >= 1)
    LM_Sensor_Config.selected.in = results[0] & LM_Sensor_Present.in;
  if (nrels >= 2)
    LM_Sensor_Config.selected.fan = results[1] & LM_Sensor_Present.fan;
  if (nrels >= 3)
    LM_Sensor_Config.selected.vid = results[2] & LM_Sensor_Present.vid;
  if (nrels >= 4)
    LM_Sensor_Config.selected.temp_mb = results[3] & LM_Sensor_Present.temp_mb;
  if (nrels >= 5)
    LM_Sensor_Config.selected.lm75 = results[4] & LM_Sensor_Present.lm75;
}

void read_present(long *results)
{
  results[0]=LM_Sensor_Config.selected.in;
  results[1]=LM_Sensor_Config.selected.fan;
  results[2]=LM_Sensor_Config.selected.vid;
  results[3]=LM_Sensor_Config.selected.temp_mb;
  results[4]=LM_Sensor_Config.selected.lm75;
}

/* Note: We always return <NA> for fans which are not available in hardware! */
void read_fan_labels(char (*results)[LABEL_LENGTH])
{
  strcpy(results[0],(LM_Sensor_Present.fan & 0x01)?
                     LM_Sensor_Config.fan_label[0]:"<NA>");
  strcpy(results[1],(LM_Sensor_Present.fan & 0x02)?
                     LM_Sensor_Config.fan_label[1]:"<NA>");
  strcpy(results[2],(LM_Sensor_Present.fan & 0x04)?
                     LM_Sensor_Config.fan_label[2]:"<NA>");
}

void write_fan_labels(int nrels, char (*results)[LABEL_LENGTH])
{
  if (nrels >= 1)
    strcpy(LM_Sensor_Config.fan_label[0],results[0]);
  if (nrels >= 2)
    strcpy(LM_Sensor_Config.fan_label[1],results[1]);
  if (nrels >= 3)
    strcpy(LM_Sensor_Config.fan_label[2],results[2]);
}

/* Note: We always return <NA> for lm75s which are not available in hardware! */
void read_lm75_labels(char (*results)[LABEL_LENGTH])
{
  strcpy(results[0],(LM_Sensor_Present.lm75 & 0x01)?
                     LM_Sensor_Config.lm75_label[0]:"<NA>");
  strcpy(results[1],(LM_Sensor_Present.lm75 & 0x02)?
                     LM_Sensor_Config.lm75_label[1]:"<NA>");
  strcpy(results[2],(LM_Sensor_Present.lm75 & 0x04)?
                     LM_Sensor_Config.lm75_label[2]:"<NA>");
  strcpy(results[3],(LM_Sensor_Present.lm75 & 0x08)?
                     LM_Sensor_Config.lm75_label[3]:"<NA>");
  strcpy(results[4],(LM_Sensor_Present.lm75 & 0x10)?
                     LM_Sensor_Config.lm75_label[4]:"<NA>");
  strcpy(results[5],(LM_Sensor_Present.lm75 & 0x20)?
                     LM_Sensor_Config.lm75_label[5]:"<NA>");
  strcpy(results[6],(LM_Sensor_Present.lm75 & 0x40)?
                     LM_Sensor_Config.lm75_label[6]:"<NA>");
  strcpy(results[7],(LM_Sensor_Present.lm75 & 0x80)?
                     LM_Sensor_Config.lm75_label[7]:"<NA>");
}

void write_lm75_labels(int nrels, char (*results)[LABEL_LENGTH])
{
  if (nrels >= 1)
    strcpy(LM_Sensor_Config.lm75_label[0],results[0]);
  if (nrels >= 2)
    strcpy(LM_Sensor_Config.lm75_label[1],results[1]);
  if (nrels >= 3)
    strcpy(LM_Sensor_Config.lm75_label[2],results[2]);
  if (nrels >= 4)
    strcpy(LM_Sensor_Config.lm75_label[3],results[3]);
  if (nrels >= 5)
    strcpy(LM_Sensor_Config.lm75_label[4],results[4]);
  if (nrels >= 6)
    strcpy(LM_Sensor_Config.lm75_label[5],results[5]);
  if (nrels >= 7)
    strcpy(LM_Sensor_Config.lm75_label[6],results[6]);
  if (nrels >= 8)
    strcpy(LM_Sensor_Config.lm75_label[7],results[7]);
}

void read_in_labels(char (*results)[LABEL_LENGTH])
{
  strcpy(results[0],(LM_Sensor_Present.in & 0x01)?
                    LM_Sensor_Config.in_label[0]:"<NA>");
  strcpy(results[1],(LM_Sensor_Present.in & 0x02)?
                    LM_Sensor_Config.in_label[1]:"<NA>");
  strcpy(results[2],(LM_Sensor_Present.in & 0x04)?
                    LM_Sensor_Config.in_label[2]:"<NA>");
  strcpy(results[3],(LM_Sensor_Present.in & 0x08)?
                    LM_Sensor_Config.in_label[3]:"<NA>");
  strcpy(results[4],(LM_Sensor_Present.in & 0x10)?
                    LM_Sensor_Config.in_label[4]:"<NA>");
  strcpy(results[5],(LM_Sensor_Present.in & 0x20)?
                    LM_Sensor_Config.in_label[5]:"<NA>");
  strcpy(results[6],(LM_Sensor_Present.in & 0x40)?
                    LM_Sensor_Config.in_label[6]:"<NA>");
}

void write_in_labels(int nrels, char (*results)[LABEL_LENGTH])
{
  if (nrels >= 1)
    strcpy(LM_Sensor_Config.in_label[0],results[0]);
  if (nrels >= 2)
    strcpy(LM_Sensor_Config.in_label[1],results[1]);
  if (nrels >= 3)
    strcpy(LM_Sensor_Config.in_label[2],results[2]);
  if (nrels >= 4)
    strcpy(LM_Sensor_Config.in_label[3],results[3]);
  if (nrels >= 5)
    strcpy(LM_Sensor_Config.in_label[4],results[4]);
  if (nrels >= 6)
    strcpy(LM_Sensor_Config.in_label[5],results[5]);
  if (nrels >= 7)
    strcpy(LM_Sensor_Config.in_label[6],results[6]);
}

void read_vid(long *results)
{
  results[0] = abs(LM78_Get(LM_PROBE_VID,0));
}

void write_vid(int nrels, long *results)
{ /* Do nothing */
}

void read_alarm(long *results)
{ 
  results[0] = LM78_Get(LM_PROBE_ALARM,0);
}

void write_alarm(int nrels, long *results)
{ /* Do nothing */
}

/* nrels contains initially the maximum number of elements which can be
   put in results, and finally the number of elements actually put there.
   A magnitude of 1 will multiply everything with 10; etc.
   buffer, bufsize is the character buffer we read from and its length.
   results will finally contain the parsed integers. 

   Buffer should contain several reals, separated by whitespace. A real
   has the following syntax:
     [ Minus ] Digit* [ Dot Digit* ] 
   (everything between [] is optional; * means zero or more).
   When the next character is unparsable, everything is skipped until the
   next whitespace.

   WARNING! This is tricky code. I have tested it, but there may still be
            hidden bugs in it, even leading to crashes and things!
*/

void parse_elements(int *nrels, void *buffer, int bufsize, 
                    long *results, int magnitude)
{
  int maxels,min,mag;
  long res;
  char nextchar=0;

  maxels = *nrels;
  *nrels = 0;

  while (bufsize && (*nrels < maxels)) {

    /* Skip spaces at the start */
    while (bufsize && ! get_user_data(nextchar,(char *) buffer) && 
           isspace((int) nextchar)) {
      bufsize --;
      ((char *) buffer)++;
    }

    /* Well, we may be done now */
    if (! bufsize)
      return;

    /* New defaults for our result */
    min = 0;
    res = 0;
    mag = magnitude;

    /* Check for a minus */
    if (! get_user_data(nextchar,(char *) buffer) && (nextchar == '-')) {
      min=1;
      bufsize--;
      ((char *) buffer)++;
    }

    /* Digits before a decimal dot */
    while (bufsize && !get_user_data(nextchar,(char *) buffer) && 
           isdigit((int) nextchar)) {
      res = res * 10 + nextchar - '0';
      bufsize--;
      ((char *) buffer)++;
    }

    /* If mag < 0, we must actually divide here! */
    while (mag < 0) {
      res = res / 10;
      mag++;
    }

    if (bufsize && (nextchar == '.')) {
      /* Skip the dot */
      bufsize--;
      ((char *) buffer)++;
  
      /* Read digits while they are significant */
      while(bufsize && (mag > 0) && 
            !get_user_data(nextchar,(char *) buffer) &&
            isdigit((int) nextchar)) {
        res = res * 10 + nextchar - '0';
        mag--;
        bufsize--;
        ((char *) buffer)++;
      }
    }
    /* If we are out of data, but mag > 0, we need to scale here */
    while (mag > 0) {
      res = res * 10;
      mag --;
    }

    /* Skip everything until we hit whitespace */
    while(bufsize && !get_user_data(nextchar,(char *) buffer) &&
          isspace ((int) nextchar)) {
      bufsize --;
      ((char *) buffer) ++;
    }

    /* Put res in results */
    results[*nrels] = (min?-1:1)*res;
    (*nrels)++;
  }    
  
  /* Well, there may be more in the buffer, but we need no more data. 
     Ignore anything that is left. */
  return;
}
    
void write_elements(int nrels,void *buffer,int *bufsize,long *results,
                    int magnitude)
{
  #define BUFLEN 20
  char BUF[BUFLEN+1]; /* An individual representation should fit in here! */
  char printfstr[10];
  int nr=0;
  int buflen,mag,times;
  int curbufsize=0;

  while ((nr < nrels) && (curbufsize < *bufsize)) {
    mag=magnitude;

    if (nr != 0) {
      put_user(' ', (char *) buffer);
      curbufsize ++;
      ((char *) buffer) ++;
    }

    /* Fill BUF with the representation of the next string */
    if (mag <= 0) {

      buflen=sprintf(BUF,"%ld",results[nr]);
      if (buflen < 0) { /* Oops, a sprintf error! */
        *bufsize=0;
        return;
      }
      while ((mag < 0) && (buflen < BUFLEN)) {
        BUF[buflen++]='0';
        mag++;
      }
      BUF[buflen]=0;
    } else {
      times=1;
      for (times=1; mag-- > 0; times *= 10);
      if (results[nr] < 0) {
        BUF[0] = '-';
        buflen = 1;
      } else
        buflen=0;
      strcpy(printfstr,"%ld.%0Xld");
      printfstr[6]=magnitude+'0';
      buflen+=sprintf(BUF+buflen,printfstr,abs(results[nr])/times,
                      abs(results[nr])%times);
      if (buflen < 0) { /* Oops, a sprintf error! */
        *bufsize=0;
        return;
      }
    }

    /* Now copy it to the user-space buffer */
    if (buflen + curbufsize > *bufsize)
      buflen=*bufsize-curbufsize;
    copy_to_user(buffer,BUF,buflen);
    curbufsize += buflen;
    (char *) buffer += buflen;

    nr ++;
  }
  if (curbufsize < *bufsize) {
    put_user('\n', (char *) buffer);
    curbufsize ++;
  }
  *bufsize=curbufsize;
}

/* Read whitespace-separated strings from buffer and put them in
   results. Each string may at most be stringlen characters long. 
   Note that a #0 also terminates the buffer. */
void parse_strings(int *nrels, void *buffer, int bufsize, 
                   char (*results)[LABEL_LENGTH])
{
  int thislen;
  char nextchar;

  int maxels = *nrels;

  *nrels = 0;

  while (bufsize && (*nrels < maxels)) {

    /* Skip spaces at the start */
    while (bufsize && ! get_user_data(nextchar,(char *) buffer) &&
           isspace((int) nextchar)) {
      bufsize --;
      ((char *) buffer)++;
    }
    
    /* Well, we may be done now */
    if (! bufsize || get_user_data(nextchar,(char *) buffer) || (nextchar == 0))
      return;

    /* Get the first stringlen characters */
    thislen = 0;
    while (bufsize && (thislen < LABEL_LENGTH - 1) && 
           ! get_user_data(nextchar,(char *) buffer) && 
           ! isspace((int) nextchar) && (nextchar != 0)) {
      results[*nrels][thislen] = nextchar;
      thislen ++;
      bufsize --;
      ((char *) buffer)++;
    }
 
    /* Skip until we meet another whitespace */
    while (bufsize && 
           ! get_user_data(nextchar,(char *) buffer) &&
           ! isspace((int) nextchar) && (nextchar != 0)) {
      bufsize --;
      ((char *) buffer)++;
    }

    /* Make results[*nrels] a real string */
    results[*nrels][thislen] = 0;

    (*nrels)++;
  }

  /* Well, there may be more in the buffer, but we need no more data.
     Ignore anything that is left. */
  return;
}

/* Write the strings from results to buffer */
void write_strings(int nrels,void *buffer,int *bufsize,
                   char (*results)[LABEL_LENGTH])
{
  int nr=0;
  int curbufsize=0;
  int tocopy;

  while ((nr < nrels) && (curbufsize < *bufsize)) {

    if (nr != 0) {
      put_user(' ', (char *) buffer);
      curbufsize ++;
      ((char *) buffer) ++;
    }

    tocopy = strlen(results[nr]);
    if (*bufsize-curbufsize < tocopy)
      tocopy = *bufsize-curbufsize;
    if (tocopy) {
      copy_to_user(buffer,results[nr],tocopy);
      curbufsize += tocopy;
      ((char *) buffer) += tocopy;
    }
    
    nr ++;
  
  }

  if (curbufsize < *bufsize) {
    put_user('\n', (char *) buffer);
    curbufsize ++;
  }

  *bufsize=curbufsize;
}

