/* -*- linux-c -*- */


/*
    lm75.c - Part of a Linux module for reading sensor data.
    Copyright (c) 1997, 1998  Alexander Larsson <alla@lysator.liu.se>,
    Frodo Looijaard <frodol@dds.nl> and Philip Edelbrock <phil@netroedge.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*
 *  LM75 sensor driver for LINUX
 *
 * Copyright (c) 1998 Philip Edelbrock
 * 
 * part of a module which provides a 
 * /proc/sensors file which shows
 * some information about your system.
 * Voltages, temperature and fan speeds
 * are currently reported.
 *
 *
 * please report any bugs/patches to me:
 *  phil@netroedge.com
 *
 *
 * Reference:
 *   National Semiconductor
 *   LM75 Digital Temperature Sensor and Thermal Watchdog with
 *   Two-Wire Interface, PDF, April 1997.
 *   http://www.national.com/ds/LM/LM75.pdf
 *
 *   Intel
 *   82371AB PCI-TO-ISA / IDE XCELERATOR (PIIX4), PDF, April 1997.
 *   ftp://ftp.intel.com
 *   
 *   System Management Bus Specification, Rev. 1.0, 1996.
 *   Contributors:
 *   Benchmarq Microelectronics Inc ., Duracell Inc.,
 *   Energizer Power Systems, Intel Corporation, Linear Technology Corporation,
 *   Maxim Integrated Products, Mitsubishi Electric Corporation,
 *   National Semiconductor Corporation, Toshiba Battery Co.,
 *   Varta Batterie AG, All rights reserved.
 *   http://www.sbs-forum.org/
 */

#include <linux/config.h>

/* The next define disables the redefinition of kernel_version. As it is
   already defined and initialized in lm78.c, this would result in twice
   defined symbols. We must include <linux/version.h> by hand now. */
#define __NO_VERSION__

/* Hack to allow the use of CONFIG_MODVERSIONS. In the kernel itself, this
   is arranged from the main Makefile. */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include "main.h"
#include "lm75.h"
#include "lm78.h"
#include "smbus.h"
#include "lmconfig.h"
#include "compat.h"

/* Functions used only internally */
static int calc_temp(u16 value);

static int winb_read_bank(u8 bank,u8 address);
static void winb_write_bank(u8 bank,u8 address, u8 value);
static int winb_read_value(u8 address);
static void winb_write_value(u8 address, u8 value);

static int Winbond_ISA=0;

static int winb_smbus = 0; /* 0=ISA, 1=SMBus */
MODULE_PARM(winb_smbus,"i");
MODULE_PARM_DESC(winb_smbus,"Use ISA (0) or SMBus (1) access for Winbond LM75 emulation");

static int lm75_addr[8] = {-1,-1,-1,-1,-1,-1,-1,-1};
MODULE_PARM(lm75_addr,"1-8i"); 
MODULE_PARM_DESC(lm75_addr,"A list of LM75 SMBus addresses to use");

static int SWITCH(u16 i) {
  return ((i & 0xff) << 8) | ((i & 0xff00) >> 8);
}

/* LM75 Initialization */
int LM75_Init () {

  int i,nr;

  if ((LM_Type == LMTYPE_WINB) && ! LM78_SMBus)
    Winbond_ISA = ! winb_smbus;

  if (! SMBus_Initialized && ! Winbond_ISA) {
    printk("Can't probe SMBus for LM75 availability, as it is not initialized.\n");
    return -ENODEV;
  }

/* Set the limits on LM75's and determine which once are available. */

  if (Winbond_ISA) {
    LM_Sensor_Present.lm75 = 0x03;
    printk("Winbond temperature sensors (2) accessed through the ISA bus\n");
  } else {
    if (lm75_addr[0] == -1) {
      nr = 0;
      for (i=0x48;i <= 0x4F; i++) { 
        if (!check_smbus(i) && (SMBus_Read_Word_Data(i,0) != -1)) {
          request_smbus(i,"lm75");
          lm75_addr[nr] = i;
          LM_Sensor_Present.lm75 |= (1 << nr);
          nr++;
        }
      }
    } else {
     nr = i = 0;
     while ((i < 8) && (lm75_addr[i] != -1))
       if ((lm75_addr[i] < 0) || (lm75_addr[i] > 0x7f)) {
         printk("Warning: SMBus address 0x%0x invalid!\n",lm75_addr[i]);
         i ++;
       }
       else if (!check_smbus(lm75_addr[i])) {
         request_smbus(lm75_addr[i],"lm75");
         LM_Sensor_Present.lm75 |= (1 << nr);
         i++;
         nr++;
       } else {
         printk("Warning: SMBus address 0x%0x already used!\n",lm75_addr[i]);
         i ++;
       }
    }

    if (!LM_Sensor_Present.lm75)
      printk("No LM75 temperature sensors found\n");
    else {
      printk("LM75 temperature sensors found at these SMBus addresses:");
      for (i=0; i <= 7; i++) 
        if (LM_Sensor_Present.lm75 & (1 << i))
          printk(" 0x%02x",lm75_addr[i]);
      printk("\n");
    }
  }

  if (LM_Sensor_Present.lm75 & 0x01) {
    LM75_Set(LM_PROBE_LM75_CONFIG,0,0);
    LM75_Set(LM_PROBE_LM75_OS,0,LM75_TEMP1_INIT_OS);
    LM75_Set(LM_PROBE_LM75_HYST,0,LM75_TEMP1_INIT_HYST);
  }

  if (LM_Sensor_Present.lm75 & 0x02) {
    LM75_Set(LM_PROBE_LM75_CONFIG,1,0);
    LM75_Set(LM_PROBE_LM75_OS,0,LM75_TEMP2_INIT_OS);
    LM75_Set(LM_PROBE_LM75_HYST,0,LM75_TEMP2_INIT_HYST);
  }

  if (LM_Sensor_Present.lm75 & 0x04) {
    LM75_Set(LM_PROBE_LM75_CONFIG,2,0);
    LM75_Set(LM_PROBE_LM75_OS,0,LM75_TEMP3_INIT_OS);
    LM75_Set(LM_PROBE_LM75_HYST,0,LM75_TEMP3_INIT_HYST);
  }

  if (LM_Sensor_Present.lm75 & 0x08) {
    LM75_Set(LM_PROBE_LM75_CONFIG,3,0);
    LM75_Set(LM_PROBE_LM75_OS,0,LM75_TEMP4_INIT_OS);
    LM75_Set(LM_PROBE_LM75_HYST,0,LM75_TEMP4_INIT_HYST);
  }

  if (LM_Sensor_Present.lm75 & 0x10) {
    LM75_Set(LM_PROBE_LM75_CONFIG,4,0);
    LM75_Set(LM_PROBE_LM75_OS,0,LM75_TEMP5_INIT_OS);
    LM75_Set(LM_PROBE_LM75_HYST,0,LM75_TEMP5_INIT_HYST);
  }

  if (LM_Sensor_Present.lm75 & 0x20) {
    LM75_Set(LM_PROBE_LM75_CONFIG,5,0);
    LM75_Set(LM_PROBE_LM75_OS,0,LM75_TEMP6_INIT_OS);
    LM75_Set(LM_PROBE_LM75_HYST,0,LM75_TEMP6_INIT_HYST);
  }

  if (LM_Sensor_Present.lm75 & 0x40) {
    LM75_Set(LM_PROBE_LM75_CONFIG,6,0);
    LM75_Set(LM_PROBE_LM75_OS,0,LM75_TEMP7_INIT_OS);
    LM75_Set(LM_PROBE_LM75_HYST,0,LM75_TEMP7_INIT_HYST);
  }

  if (LM_Sensor_Present.lm75 & 0x80) {
    LM75_Set(LM_PROBE_LM75_CONFIG,7,0);
    LM75_Set(LM_PROBE_LM75_OS,0,LM75_TEMP8_INIT_OS);
    LM75_Set(LM_PROBE_LM75_HYST,0,LM75_TEMP8_INIT_HYST);
  }

  return 0;
}

void LM75_Cleanup () {
  int i;
  if (!Winbond_ISA)
    for (i=0; i <= 7; i++) 
      if (LM_Sensor_Present.lm75 & (1 << i))
        release_smbus(lm75_addr[i]);
}

void LM75_Set(int what, int nr, long value) {

    if ( (nr < 0) || (nr > 7))  /* nr out of range */
      return;
    if (Winbond_ISA && (nr > 1))
      return;
    if (what == LM_PROBE_LM75_CONFIG) { /* Set config reg */
      if (Winbond_ISA) 
        winb_write_bank(nr+1,0x52,value & 0xff); /* Set Config reg to 0x00 */
      else
        SMBus_Write_Byte_Data(lm75_addr[nr],1,value);       
    } else {
      if (value < 0) /* Must be done here, to correct rounding for negative
                        values */
        value += 0x200 * 5;
      value = (value + 2) / 5 /* Now in half-degrees */;
      value = ((value & 0x1fe)>>1)+ ((value & 1)<<15); /* Now in SMBus format */
      if (what == LM_PROBE_LM75_HYST) { /* Set Thyst=value */
        if (Winbond_ISA) {
          winb_write_bank(nr+1,0x53,value & 0xff);
          winb_write_bank(nr+1,0x54,value >> 8);
        }
        else
          SMBus_Write_Word_Data(lm75_addr[nr],2,value); 
        LM_Sensor_Data.lm75[nr].temphyst = value;
      }
      else if (what == LM_PROBE_LM75_OS) { /* Set Tos=value */
        if (Winbond_ISA) {
          winb_write_bank(nr+1,0x55,value & 0xff);
          winb_write_bank(nr+1,0x56,value >> 8);
        }
        else
          SMBus_Write_Word_Data(lm75_addr[nr],3,value); 
        LM_Sensor_Data.lm75[nr].tempos = value;
      }
    }
}

long LM75_Get(int what, int nr) {
    if ( (nr < 0) || (nr > 7))  /* nr out of range */
      return 0;
    if (what == LM_PROBE_LM75_VALUE)
      return calc_temp(LM_Sensor_Data.lm75[nr].temp);
    else if (what == LM_PROBE_LM75_HYST)
      return calc_temp(LM_Sensor_Data.lm75[nr].temphyst);
    else if (what == LM_PROBE_LM75_OS)
      return calc_temp(LM_Sensor_Data.lm75[nr].tempos);
    else
      return 0; /* what out of range */
  
}

/* Input: a LM75 register value.
   Output: the temperature in tenth of degrees */
int calc_temp(u16 value) {
   int val;
   val=value >> 7;
   if (val > 0x0191)
      val= - ( 0x0200 - val );
   return val*5;
}

int LM75_Print_Values (char *buf) {
  int i,value,value2,value3;
  int len=0;

  if (LM_Sensor_Data.valid) {

    for (i=0; i < MAX_LM75 ; i++) 
      if (LM_Sensor_Config.selected.lm75 & (1 << i)) {
        value=LM75_Get(LM_PROBE_LM75_VALUE,i);
        value2=LM75_Get(LM_PROBE_LM75_OS,i);
        value3=LM75_Get(LM_PROBE_LM75_HYST,i);
        len += sprintf(buf+len,
                       "%s:\t%c%d.%d C\t(Tos = %c%d.%d C,\tThyst = %c%d.%d C)\n",
                      LM_Sensor_Config.lm75_label[i],
                      value<0?'-':'+', abs(value) / 10, abs(value) % 10,
                      value2<0?'-':'+', abs(value2) / 10, abs(value2) % 10, 
                      value3<0?'-':'+', abs(value3) / 10, abs(value3) % 10);
#ifdef DEBUG
        len += sprintf(buf+len, "%s reg 1 (configuration register) reports 0x%X\n",
               LM_Sensor_Config.lm75_label[i],LM_Sensor_Data.lm75[i].config);
#endif
      }
  }
  return len;
}

void LM75_Update_Values (void) {

  int i;
  int value;

  if (Winbond_ISA)
    for (i = 1; i <= 2; i++) {
      LM_Sensor_Data.lm75[i-1].temp=(winb_read_bank(i,0x50) << 8) | 
                                    winb_read_bank(i,0x51);
#ifdef DEBUG
      LM_Sensor_Data.lm75[i-1].config=(winb_read_bank(i,0x52));
#endif
      LM_Sensor_Data.lm75[i-1].temphyst=(winb_read_bank(i,0x53) << 8) |
                                        winb_read_bank(i,0x54);
      LM_Sensor_Data.lm75[i-1].tempos=(winb_read_bank(i,0x55) << 8) |
                                      winb_read_bank(i,0x56);
    }
  else
    for (i = 0 ; i < 8 ; i ++)
      if (LM_Sensor_Present.lm75 & (1 << i)) {
        /* Clear configuration reg 
        SMBus_Write_Byte_Data(lm75_addr[i],1,0); */
        value=SWITCH(SMBus_Read_Word_Data(lm75_addr[i],0));
        LM_Sensor_Data.lm75[i].temp=value;
#ifdef DEBUG
        LM_Sensor_Data.lm75[i].config=SWITCH(SMBus_Read_Byte_Data(lm75_addr[i],1));
#endif
        LM_Sensor_Data.lm75[i].temphyst=SWITCH(SMBus_Read_Word_Data(lm75_addr[i],2));
        LM_Sensor_Data.lm75[i].tempos=SWITCH(SMBus_Read_Word_Data(lm75_addr[i],3));
      }
}

/* When we change to bank 1, all registers are temporary nuked. So we need
   to put a semaphore around it. If you want to access a 'low' register
   (ie. something below the bank-specific region), you should still use
   this function, with bank=0! */
int winb_read_bank(u8 bank,u8 address)
{
  int res;
  down(&LM78_Sem); 
  if (bank)
/* When we change to bank 1, all registers are temporary nuked. So we need
   to put a semaphore around it. If you want to access a 'low' register
   (ie. something below the bank-specific region), you should still use
   this function! */
    winb_write_value(LM78_WINB_BANK,bank);
  res = winb_read_value(address);
  if (bank)
    winb_write_value(LM78_WINB_BANK,0);
  up(&LM78_Sem);
  return res;
}

/* When we change to bank 1, all registers are temporary nuked. So we need
   to put a semaphore around it. If you want to access a 'low' register
   (ie. something below the bank-specific region), you should still use
   this function, with bank=0! */
static void winb_write_bank(u8 bank, u8 address, u8 value)
{
  down(&LM78_Sem);
  if (bank)
    winb_write_value(LM78_WINB_BANK,bank);
  winb_write_value(address,value);
  if (bank)
    winb_write_value(LM78_WINB_BANK,0);
  up(&LM78_Sem);
}

/* DO NOT USE!
   If you use this function (except from winb_read_bank), you would circumvent
   the LM78_Sem lock! */
int winb_read_value(u8 address)
{
  while (LM78_BUSY) {
    current->state = TASK_INTERRUPTIBLE;
    current->timeout = jiffies + 10;
    schedule();
  }
  return LM78_READ(address);
}

/* DO NOT USE!
   If you use this function (except from winb_write_bank), you would circumvent
   the LM78_Sem lock! */
void winb_write_value(u8 address, u8 value)
{
  while (LM78_BUSY) {
    current->state = TASK_INTERRUPTIBLE;
    current->timeout = jiffies + 10;
    schedule();
  }
  LM78_WRITE(address,value);
}

