// wmgl518sm - A gl518sm monitor for Window Maker
// 04/29/99  Release 0.5
// Copyright (C) 1999 by Frederik Schler <fr.schueler@netsurf.de>
// based on wmlm7x
// Copyright (C) 1998  Sam Hawker <shawkie@geocities.com>
// This software comes with ABSOLUTELY NO WARRANTY
// This software is free software, and you are welcome to redistribute it
// under certain conditions
// See the README file for a more complete notice.


// Defines, includes and global variables
// --------------------------------------

// User defines - standard
#define WINDOWMAKER false
#define USESHAPE    false
#define AFTERSTEP   false
#define NORMSIZE    64
#define ASTEPSIZE   56
#define NAME        "wmgl518sm"
#define CLASS       "WMGL518SM"

// User defines - custom
#define BACKCOLOR   "#282828"
#define FORECOLOR   "#c7c3c7"
// #define VOLTAGEVIEW true
#define UINTERVAL   120           // 20ths of a second
#define TEMPMIN     30
#define TEMPMAX     60

// Includes - standard
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

// Includes - custom
#include <ctype.h>

// X-Windows includes - standard
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

// Pixmaps - standard
Pixmap pm_main;
Pixmap pm_tile;
Pixmap pm_disp;
Pixmap pm_mask;

// Pixmaps - custom
Pixmap pm_letters;

// Xpm images - standard
#include "XPM/wmgl518sm.xpm"
#include "XPM/tile.xpm"

// Xpm images - custom
#include "XPM/letters.xpm"

// Variables for command-line arguments - standard
bool wmaker=WINDOWMAKER;
bool ushape=USESHAPE;
bool astep=AFTERSTEP;
char display[256]="";
char position[256]="";
int winsize;

// Variables for rc file - custom
char backcolor[256]=BACKCOLOR;
char forecolor[256]=FORECOLOR;

// X-Windows basics - standard
Atom _XA_GNUSTEP_WM_FUNC;
Atom deleteWin;
Display *d_display;
Window w_icon;
Window w_main;
Window w_root;
Window w_activewin;

// X-Windows basics - custom
GC gc_gc;
unsigned long color[3];


// Misc custom global variables
// ----------------------------

// Current state information
int voltage=0;
float voltval=0;

int ucount=0;
int fanrpm[52];
int cputemp[52];

// Procedures and functions
// ------------------------

// Procedures and functions - standard
void initXWin(int argc, char **argv);
void freeXWin();
void createWin(Window *win, int x, int y);
unsigned long getColor(char *colorname);
unsigned long mixColor(char *colorname1, int prop1, char *colorname2, int prop2);

// Procedures and functions - custom
void scanArgs(int argc, char **argv);
void checkStats(bool forced=false);
// void checkVoltage(bool forced=true);
void repaint();
void drawText(int x, int y, char *text);


// Implementation
// --------------

int main(int argc, char **argv){
  scanArgs(argc, argv);
  initXWin(argc, argv);
  XGCValues gcv;
  unsigned long gcm;
  gcm=GCGraphicsExposures;
   gcv.graphics_exposures=false;
   gc_gc=XCreateGC(d_display, w_root, gcm, &gcv);

   color[0]=getColor(backcolor);
   color[1]=mixColor(forecolor, 60, backcolor, 40);
   color[2]=getColor(forecolor);

   XpmAttributes xpmattr;
   XpmColorSymbol xpmcsym[3]={{"back_color", NULL, color[0]},
                              {"mid_color", NULL, color[1]},
                              {"fore_color", NULL, color[2]}};
   xpmattr.numsymbols=3;
   xpmattr.colorsymbols=xpmcsym;
   xpmattr.exactColors=false;
   xpmattr.closeness=40000;
   xpmattr.valuemask=XpmColorSymbols | XpmExactColors | XpmCloseness;
   XpmCreatePixmapFromData(d_display, w_root, wmgl518sm_xpm, &pm_main, &pm_mask, &xpmattr);
   XpmCreatePixmapFromData(d_display, w_root, tile_xpm, &pm_tile, NULL, &xpmattr);
   XpmCreatePixmapFromData(d_display, w_root, letters_xpm, &pm_letters, NULL, &xpmattr);
   pm_disp=XCreatePixmap(d_display, w_root, 64, 64, DefaultDepth(d_display, DefaultScreen(d_display)));
   
   if(wmaker || ushape || astep)
     XShapeCombineMask(d_display, w_activewin, ShapeBounding, winsize/2-32, winsize/2-32, pm_mask, ShapeSet);
   else
     XCopyArea(d_display, pm_tile, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);
   
   XSetClipMask(d_display, gc_gc, pm_mask);
   XCopyArea(d_display, pm_main, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);
   XSetClipMask(d_display, gc_gc, None);
   for(int i=0;i<52;i++){
     fanrpm[i]=0;
     cputemp[i]=0;
     checkStats(true);
     XEvent xev;
     XSelectInput(d_display, w_activewin, ExposureMask | ButtonPressMask | ButtonReleaseMask);
     XMapWindow(d_display, w_main);
     bool done=false;
     while(!done){
       while(XPending(d_display)){
	 XNextEvent(d_display, &xev);
	 switch(xev.type){
	 case Expose:
	   repaint();
	    break;
	 case ClientMessage:
	   if(xev.xclient.data.l[0]==deleteWin)
	     done=true;
	   break;
	 }
       }
       
       ucount++;
       if(ucount>=UINTERVAL)
          checkStats(true);
       XFlush(d_display);
       usleep(50000);
     }     
   }
   XFreeGC(d_display, gc_gc);
   XFreePixmap(d_display, pm_main);
   XFreePixmap(d_display, pm_tile);
   XFreePixmap(d_display, pm_letters);
   XFreePixmap(d_display, pm_disp);
   XFreePixmap(d_display, pm_mask);
   freeXWin();
   return 0;
}

void initXWin(int argc, char **argv){
  winsize=astep ? ASTEPSIZE : NORMSIZE;
  
  if((d_display=XOpenDisplay(display))==NULL){
    fprintf(stderr,"%s : Unable to open X display '%s'.\n", NAME, XDisplayName(display));
    exit(1);
  }
  _XA_GNUSTEP_WM_FUNC=XInternAtom(d_display, "_GNUSTEP_WM_FUNCTION", false);
  deleteWin=XInternAtom(d_display, "WM_DELETE_WINDOW", false);
  
  w_root=DefaultRootWindow(d_display);
  
  XWMHints wmhints;
  XSizeHints shints;
  shints.x=0;
  shints.y=0;
  shints.flags=0;
  bool pos=(XWMGeometry(d_display, DefaultScreen(d_display), position, NULL, 0, &shints, &shints.x, &shints.y,
			&shints.width, &shints.height, &shints.win_gravity) & (XValue | YValue));
  shints.min_width=winsize;
  shints.min_height=winsize;
  shints.max_width=winsize;
   shints.max_height=winsize;
   shints.base_width=winsize;
   shints.base_height=winsize;
   shints.flags=PMinSize | PMaxSize | PBaseSize;

   createWin(&w_main, shints.x, shints.y);

   if(wmaker || astep || pos)
      shints.flags |= USPosition;
   if(wmaker){
      wmhints.initial_state=WithdrawnState;
      wmhints.flags=WindowGroupHint | StateHint | IconWindowHint;
      createWin(&w_icon, shints.x, shints.y);
      w_activewin=w_icon;
      wmhints.icon_window=w_icon;
   }
   else{
      wmhints.initial_state=NormalState;
      wmhints.flags=WindowGroupHint | StateHint;
      w_activewin=w_main;
   }
   wmhints.window_group=w_main;
   XSetWMHints(d_display, w_main, &wmhints);
   XSetWMNormalHints(d_display, w_main, &shints);
   XSetCommand(d_display, w_main, argv, argc);
   XStoreName(d_display, w_main, NAME);
   XSetIconName(d_display, w_main, NAME);
   XSetWMProtocols(d_display, w_activewin, &deleteWin, 1);
}

void freeXWin(){
   XDestroyWindow(d_display, w_main);
   if(wmaker)
      XDestroyWindow(d_display, w_icon);
   XCloseDisplay(d_display);
}

void createWin(Window *win, int x, int y){
   XClassHint classHint;
   *win=XCreateSimpleWindow(d_display, w_root, x, y, winsize, winsize, 0, 0, 0);
   classHint.res_name=NAME;
   classHint.res_class=CLASS;
   XSetClassHint(d_display, *win, &classHint);
}

unsigned long getColor(char *colorname){
   XColor color;
   XWindowAttributes winattr;
   XGetWindowAttributes(d_display, w_root, &winattr);
   color.pixel=0;
   XParseColor(d_display, winattr.colormap, colorname, &color);
   color.flags=DoRed | DoGreen | DoBlue;
   XAllocColor(d_display, winattr.colormap, &color);
   return color.pixel;
}

unsigned long mixColor(char *colorname1, int prop1, char *colorname2, int prop2){
   XColor color, color1, color2;
   XWindowAttributes winattr;
   XGetWindowAttributes(d_display, w_root, &winattr);
   XParseColor(d_display, winattr.colormap, colorname1, &color1);
   XParseColor(d_display, winattr.colormap, colorname2, &color2);
   color.pixel=0;
   color.red=(color1.red*prop1+color2.red*prop2)/(prop1+prop2);
   color.green=(color1.green*prop1+color2.green*prop2)/(prop1+prop2);
   color.blue=(color1.blue*prop1+color2.blue*prop2)/(prop1+prop2);
   color.flags=DoRed | DoGreen | DoBlue;
   XAllocColor(d_display, winattr.colormap, &color);
   return color.pixel;
}

void scanArgs(int argc, char **argv){
   for(int i=1;i<argc;i++){
      if(strcmp(argv[i], "-h")==0 || strcmp(argv[i], "--help")==0){
         fprintf(stderr, "wmgl518sm -- A gl518sm-r00 sensor chip monitor for Window Maker\nReleased 04/29/99 version 0.5\n");
         fprintf(stderr, "Copyright (C) 1998  Sam Hawker <shawkie@geocities.com>\n");
	 fprintf(stderr, "Copyright (C) 1999 Frederik Schler <fr.schueler@dnetsurf.de>\n");
         fprintf(stderr, "This software comes with ABSOLUTELY NO WARRANTY\n");
         fprintf(stderr, "This software is free software, and you are welcome to redistribute it\n");
         fprintf(stderr, "under certain conditions\n");
         fprintf(stderr, "See the README file for a more complete notice.\n\n");
         fprintf(stderr, "usage:\n\n   %s [options]\n\noptions:\n\n",argv[0]);
         fprintf(stderr, "   -h | --help            display this help screen\n");
         fprintf(stderr, "   -w | --withdrawm       use WithdrawnState    (for WindowMaker)\n");
         fprintf(stderr, "   -s | --shaped          shaped window\n");
         fprintf(stderr, "   -a | --afterstep       use smaller window    (for AfterStep Wharf)\n");
         fprintf(stderr, "   -position position     set window position   (see X manual pages)\n");
         fprintf(stderr, "   -display display       select target display (see X manual pages)\n");
         exit(0);
      }
      if(strcmp(argv[i], "-w")==0 || strcmp(argv[i], "--withdrawn")==0)
         wmaker=!wmaker;
      if(strcmp(argv[i], "-s")==0 || strcmp(argv[i], "--shaped")==0)
         ushape=!ushape;
      if(strcmp(argv[i], "-a")==0 || strcmp(argv[i], "--afterstep")==0)
         astep=!astep;
      if(strcmp(argv[i], "-position")==0){
         if(i<argc-1){
            i++;
            sprintf(position, "%s", argv[i]);
         }
         continue;
      }
      if(strcmp(argv[i],"-display")==0){
         if(i<argc-1){
            i++;
            sprintf(display, "%s", argv[i]);
         }
         continue;
      }
   }
}

void checkStats(bool forced=false) {
   ucount=0;
   memmove(&fanrpm[0], &fanrpm[1], sizeof(float)*51);
   memmove(&cputemp[0], &cputemp[1], sizeof(float)*51);
   float value1, value2, value3;
   FILE *f;
   if((f=fopen("/proc/sys/dev/sensors/gl518sm-r00-i2c-0-2d/fan1", "r"))==NULL)
      fprintf(stderr, "%s : Unable to read fan's rpm.\n", NAME);
   else{
      fscanf(f, "%f %f", &value1, &value2);
      fanrpm[51]=(int)value2 /2;
      fclose(f);
   }
   if((f=fopen("/proc/sys/dev/sensors/gl518sm-r00-i2c-0-2d/temp", "r"))==NULL)
      fprintf(stderr, "%s : Unable to read cpu temperature.\n", NAME);
   else{
      fscanf(f, "%f %f %f", &value1, &value2, &value3);
      cputemp[51]=(int)value3;
      fclose(f);
   }
   if((f=fopen("/proc/sys/dev/sensors/gl518sm-r00-i2c-0-2d/vin3", "r"))==NULL)
     fprintf(stderr, "%s : Unable to read voltage.\n", NAME);
   else{
     fscanf(f, "%f %f %f", &value1, &value2, &value3);
     voltval=value3;
     fclose(f);
   }
   char tval[10];
   sprintf(tval, "%3i", fanrpm[51]);
   drawText(34, 28, tval);
   sprintf(tval, "%3i", cputemp[51]);
   drawText(31, 8, tval);
   sprintf(tval, "%01i", (int)voltval);
   drawText(33, 48, tval);
   sprintf(tval, "%02i", abs((int)((voltval-(int)voltval)*100)));
   drawText(42, 48, tval);

   repaint();
}

void repaint(){
   XCopyArea(d_display, pm_disp, w_activewin, gc_gc, 0, 0, 64, 64, winsize/2-32, winsize/2-32);
   XEvent xev;
   while(XCheckTypedEvent(d_display, Expose, &xev));
}


void drawText(int x, int y, char *text){
   char letters[]="  .0123456789 ";
   for(int i=0;i<strlen(text);i++){
      char *letter=strchr(letters, toupper(text[i]));
      int lpos=(letter==NULL) ? 0 : (letter-letters);
      XCopyArea(d_display, pm_letters, pm_disp, gc_gc, lpos*6, 0, 5, 7, x+6*i, y);
   }
}
