/***********************************************************************
 *   Code is based on wmppp, wmload, wmtime, wmcp, and asbutton
 *   Author: Edward H. Flora <ehflora@ksu.edu>
 *   Ver 0 Rel 3    March 6, 1999
 *
 *   Contributors:
 *              Ben Cohen <buddog@aztec.asu.edu>
 *                  original author of wmcp (et al.)
 *              Thomas Nemeth <tnemeth@multimania.com>
 *                  contributor to wmcp
 *              Casey Harkins <charkins@cs.wisc.edu> 
 *                  Bug fix reading config file path - 3/6/99
 *                  Added button-presses, and other - denoted by *charkins*
 *              Michael Henderson <mghenderson@lanl.gov>
 *                  Application ideas, suggestions
 *              Ryan ?? <pancake@mindspring.com> 
 *                  Modified wmbutton to asbutton.
 *                  Note: asbutton is a seperate program, not associated 
 *                        with wmbutton (just as wmbutton is not associated
 *                        with wmcp)
 *              Jon Bruno
 *                  Web Page Development
 *    The contributors listed above are not necessarily involved with the 
 *    development of wmbutton.  I'm listing them here partially as thanks for 
 *    helping out, catching bugs in the code, etc.
 ***********************************************************************/
#include <Xlib.h>
#include <Xutil.h>
#include <xpm.h>
#include <extensions/shape.h>
#include <keysym.h>
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "backdrop.xpm"           // background graphic
#include "buttons.xpm"            // graphic of 9 buttons
#include "mask.xbm"

#define CONFIGFILEMAX 128
#define CONFFILENAME  "/.wmbutton"
#define VER           0
#define REL           4

// #define MIDMOUSE
#define LMASK         0
#define MMASK         10
#define RMASK         20

/***********************************************************************
 * 		Globals..    OK.. there's too many globals.
 *                           Feel free and fix it, if you'd like.
 ***********************************************************************/
Display *display;
int screen;
Window rootwin, win, iconwin;
GC gc;
int depth;
Pixel bg_pixel, fg_pixel;

XSizeHints xsizehints;
XWMHints * xwmhints;
XClassHint xclasshint;
char configfile[CONFIGFILEMAX]; 

typedef struct _XpmIcon {
  Pixmap pixmap;
  Pixmap mask;
  XpmAttributes attributes;
} XpmIcon;


Pixmap pixmask;

typedef struct _button_region {
  int x,y;
  int i,j;
} ButtonArea;

ButtonArea button_region[9];

XpmIcon template, visible, buttons;

int numb_of_apps = 9;
int N = 1;		/* Button number pressed to goto app #     */
int border  = 0;
int Verbose = 0;
int mmouse  = 0;
int button_pressed = -1;	/* button to be drawn pressed *charkins*/

char *app_name = "wmbutton";

/*************** Function Prototypes ***********************************/
void RunAppN(int app);             // function to run app N as found in conf file
char *Parse(int app);              // parse data in config file
void redraw(void);                 
void getPixmaps(void);
int  whichButton(int x, int y);    // determine which button has been pressed
int  flush_expose(Window w);
void show_usage(void);             // show usage message to stderr
char *readln(FILE *fp);            // read line from file, return pointer to it

/***********************************************************************
 * 		Main
 ***********************************************************************/
int main( int argc, char ** argv ) {
  XEvent report;
  XGCValues xgcValues;	
  XTextProperty app_name_atom;

  int dummy = 0;
  int i;
  char Geometry_str[64] = "64x64+0+0";
  char Display_str[64] = "";
  
  strcpy(configfile, getenv("HOME"));  // Added by Casey Harkin, 3/6/99
  strcat(configfile, CONFFILENAME);    // Fixed Bug - didn't look in home directory
                                       // but startup directory

  /* Parse Command Line Arguments */  
  for ( i=1; i<argc; i++ ) {
    if ( *argv[i] == '-' ) {
      switch ( *(argv[i]+1) ) {
      case 'v':                        // Turn on Verbose (debugging) Mode
	Verbose = 1;
	break;
      case 'g':                        // Set Geometry Options
	if ( ++i >= argc ) show_usage();
	sscanf(argv[i], "%s", Geometry_str);
	if ( Verbose ) printf("Geometry is: %s\n", Geometry_str);
	break;
      case 'd':                        // Set display 
	if ( ++i >= argc ) show_usage();
	sscanf(argv[i], "%s", Display_str);
	if ( Verbose ) printf("Display is: %s\n", Display_str);
	break;
      case 'h':                        // Show Help Message
	show_usage();
	break;
      case 'f':                        // use config file <filename>
	if ( ++i >= argc ) show_usage();
	sscanf(argv[i], "%s", configfile);
	if ( Verbose ) printf("Using Config File: %s\n", configfile);
	break;
      case 'm':                        // Enable Middle Mouse
	mmouse = 1;
	break;
      default:                         // other, unknown, parameters
	show_usage();
	break;
      }
    }
  }
    
  if ( (display = XOpenDisplay(Display_str)) == NULL ) {
    fprintf(stderr,"Fail: XOpenDisplay for %s\n", Display_str);	
    exit(-1);
  }
  
  screen  = DefaultScreen(display);
  rootwin = RootWindow(display,screen);
  depth   = DefaultDepth(display, screen);
  
  bg_pixel = WhitePixel(display, screen ); 
  fg_pixel = BlackPixel(display, screen ); 
  
  xsizehints.flags  = USSize | USPosition;
  xsizehints.width  = 64;
  xsizehints.height = 64;
  
  /* Parse Geometry string and fill in sizehints fields */
  XWMGeometry(display, screen, 
	      Geometry_str, 
	      NULL, 	
	      border, 
	      &xsizehints,
	      &xsizehints.x, 
	      &xsizehints.y,
	      &xsizehints.width,
	      &xsizehints.height, 
	      &dummy);
  
  if ( (win = XCreateSimpleWindow(display,
				  rootwin,
				  xsizehints.x,
				  xsizehints.y,
				  xsizehints.width,
				  xsizehints.height,
				  border,
				  fg_pixel, bg_pixel) ) == 0 ) {
    fprintf(stderr,"Fail: XCreateSimpleWindow\n");	
    exit(-1);
  }
  
  if ( (iconwin = XCreateSimpleWindow(display,
				      win,
				      xsizehints.x,
				      xsizehints.y,
				      xsizehints.width,
				      xsizehints.height,
				      border,
				      fg_pixel, bg_pixel) ) == 0 ) {
    fprintf(stderr,"Fail: XCreateSimpleWindow\n");	
    exit(-1);
  }

  /* Set up shaped windows */
  /*Gives the appicon a border so you can grab and move it. */

  if ( ( pixmask = XCreateBitmapFromData(display, 
					 win,
					 mask_bits,
					 mask_width,
					 mask_height) )  == 0 ) {
    fprintf(stderr,"Fail: XCreateBitmapFromData\n");
  }
  
  XShapeCombineMask(display, win, ShapeBounding, 0, 0, pixmask, ShapeSet );
  XShapeCombineMask(display, iconwin, ShapeBounding, 0, 0, pixmask, ShapeSet);

  /* Convert in pixmaps from .xpm includes. */
  getPixmaps();

  /* Interclient Communication stuff */
  /* Appicons don't work with out this stuff */
  xwmhints = XAllocWMHints();
  xwmhints->flags = WindowGroupHint | IconWindowHint | StateHint;	
  xwmhints->icon_window = iconwin;
  xwmhints->window_group = win;
  xwmhints->initial_state = WithdrawnState;  
  XSetWMHints( display, win, xwmhints );

  xclasshint.res_name = "wmbutton";
  xclasshint.res_class = "Wmbutton";
  XSetClassHint( display, win, &xclasshint );

  XSetWMNormalHints( display, win, &xsizehints );
  
  /* Tell window manager what the title bar name is. We never see */
  /* this anyways in the WithdrawnState      */
  if ( XStringListToTextProperty(&app_name, 1, &app_name_atom) == 0 ) {
    fprintf(stderr,"%s: Can't set up window name\n", app_name);
    exit(-1);
  }
  XSetWMName( display, win, &app_name_atom );
  
  /* Create Graphic Context */	
  if (( gc = XCreateGC(display, win,(GCForeground | GCBackground), &xgcValues))
       == NULL ) {
    fprintf(stderr,"Fail: XCreateGC\n");	
    exit(-1);
  }

  /* XEvent Masks. We want both window to process X events */
  XSelectInput(display, win,
	       ExposureMask | 
	       ButtonPressMask | 
		   ButtonReleaseMask |		/* added ButtonReleaseMask *charkins*/
	       PointerMotionMask |
	       StructureNotifyMask );  
  XSelectInput(display, iconwin,
	       ExposureMask | 
	       ButtonPressMask | 
		   ButtonReleaseMask |		/* added ButtonReleaseMask *charkins*/
	       PointerMotionMask |
	       StructureNotifyMask );
  
  /* Store the 'state' of the application for restarting */
  XSetCommand( display, win, argv, argc );	

  /* Window won't ever show up until it is mapped.. then drawn after a 	*/
  /* ConfigureNotify */
  XMapWindow( display, win );

  /* X Event Loop */
  while (1) {
    XNextEvent(display, &report );
    switch (report.type) {
    case Expose:
      if (report.xexpose.count != 0) {	
	break;
      }
      if ( Verbose ) fprintf(stdout,"Event: Expose\n");	
      redraw();
      break;      
    case ConfigureNotify:
      if ( Verbose ) fprintf(stdout,"Event: ConfigureNotify\n");	
      redraw();
      break;
    case ButtonPress:	/* draw button pressed, don't launch *charkins*/
      switch (report.xbutton.button) {
      case Button1:
	N = whichButton(report.xbutton.x, report.xbutton.y );
	if ( (N >= 0) && (N <= numb_of_apps) ) {
	  button_pressed = N + LMASK;
	  redraw();
	}
	if ( Verbose ) 
	  fprintf(stdout,"Button 1:x=%d y=%d N=%d\n", 
		  report.xbutton.x, report.xbutton.y, N+LMASK);	
	break;
      case Button2:
	if (mmouse) {
	  N = whichButton(report.xbutton.x, report.xbutton.y );
	  if ( (N >= 0) && (N <= numb_of_apps) ) {
	    button_pressed = N + MMASK;
	    redraw();
	  }
	  if ( Verbose ) 
	    fprintf(stdout,"Button 1:x=%d y=%d N=%d\n", 
		    report.xbutton.x, report.xbutton.y, N+MMASK);
	}
	break;
      case Button3:
	N = whichButton(report.xbutton.x, report.xbutton.y );
	if ( (N >= 0) && (N <= numb_of_apps) ) {
	  button_pressed = N + RMASK;
	  redraw();
	}
	if ( Verbose ) 
	  fprintf(stdout,"Button 3:x=%d y=%d N=%d\n", 
		  report.xbutton.x, report.xbutton.y, N+RMASK);
	break;
      }
      break;
    case ButtonRelease:	/* launch app here if still over button *charkins*/
      switch (report.xbutton.button) {
      case Button1:
	N = whichButton(report.xbutton.x, report.xbutton.y );
	if ( (N >= 0) && (N <= numb_of_apps) && (N == button_pressed))
	  RunAppN( N + LMASK);
	button_pressed=-1;
	redraw();
	if ( Verbose ) 
	  fprintf(stdout,"Button 1:x=%d y=%d N=%d\n", 
		  report.xbutton.x, report.xbutton.y, N+LMASK);	
	break;
      case Button2:
	if (mmouse) {
	  N = whichButton(report.xbutton.x, report.xbutton.y );
	  if ( (N >= 0) && (N <= numb_of_apps) && (N == button_pressed))
	    RunAppN( N + MMASK);
	  button_pressed=-1;
	  redraw();
	  if ( Verbose ) 
	    fprintf(stdout,"Button 1:x=%d y=%d N=%d\n", 
		    report.xbutton.x, report.xbutton.y, N+MMASK);
	}
	break;
      case Button3:
	N = whichButton(report.xbutton.x, report.xbutton.y );
	if ( (N >= 0) && (N <= numb_of_apps) && (N == button_pressed))
	  RunAppN( N + RMASK);
	button_pressed=-1;
	redraw();
	if ( Verbose ) 
	  fprintf(stdout,"Button 3:x=%d y=%d N=%d\n", 
		  report.xbutton.x, report.xbutton.y, N+RMASK);
	break;
      }
      break;
    case DestroyNotify:
      if ( Verbose ) fprintf(stdout, "Bye\n");
      XFreeGC(display, gc);
      XDestroyWindow(display,win);
      XDestroyWindow(display,iconwin);
      XCloseDisplay(display);
      if ( Verbose ) fprintf(stdout, "Bye\n");
      exit(0);
      break;
    }
  }
  return (0);
}

/***********************************************************************
 *   redraw
 *
 *	 Map the button region coordinates.
 *
 *   Draw the appropriate number of buttons on the 'visible' Pixmap 
 *   using data from the 'buttons' pixmap.
 *
 *   Then, copy the 'visible' pixmap to the two windows ( the withdrawn
 *   main window and the icon window which is the main window's icon image.)
 ***********************************************************************/
void redraw() {
  int n;
  int i,j;
  int dest_x, dest_y;
  int space;
  int offset;
  int bsize = 18;

  space = 0;
  offset = 5;
  XCopyArea(display, template.pixmap, visible.pixmap, gc, 0, 0,
	    template.attributes.width, template.attributes.height, 0, 0 ); 
 
  for ( j=0; j < 3; j++ ) {
    for ( i=0; i < 3; i++ ) {
      n = i + j * 3;
      dest_x = i*(bsize + space) + offset + space;
      dest_y = j*(bsize + space) + offset + space;
      
      /* Define button mouse coords */
      button_region[n].x = dest_x;
      button_region[n].y = dest_y;
      button_region[n].i = dest_x + bsize - 1;
      button_region[n].j = dest_y + bsize - 1; 
      
      /* Copy button images for valid apps */
      if (  (n + 1) <= numb_of_apps ) {
        XCopyArea(display, buttons.pixmap, visible.pixmap, gc, 
	          i * bsize, j * bsize, bsize, bsize, dest_x, dest_y);
      } 
    }
  }

  if ( button_pressed>0 ) {	/* draw pressed button *charkins*/
    if(button_pressed>RMASK) button_pressed-=RMASK;
    else if(button_pressed>MMASK) button_pressed-=MMASK;
    else if(button_pressed>LMASK) button_pressed-=LMASK;
    i = (button_pressed-1) % 3;	/* get col of button */
    j = (button_pressed-1) / 3;	/* get row of button */
    dest_x = i * (bsize + space) + offset + space;
    dest_y = j * (bsize + space) + offset + space;
    XSetForeground(display, gc, bg_pixel);
    XDrawLine(display, visible.pixmap, gc,
	      dest_x+1, dest_y+bsize-1, dest_x+bsize-1, dest_y+bsize-1);
    XDrawLine(display, visible.pixmap, gc,
	      dest_x+bsize-1, dest_y+bsize-1, dest_x+bsize-1, dest_y+1);
    XSetForeground(display, gc, fg_pixel);
    XDrawLine(display, visible.pixmap, gc,
	      dest_x, dest_y, dest_x+bsize-2, dest_y);
    XDrawLine(display, visible.pixmap, gc,
	      dest_x, dest_y, dest_x, dest_y+bsize-2);
  } /*charkins*/
  
  flush_expose( win ); 
  XCopyArea(display, visible.pixmap, win, gc, 0, 0,
	    visible.attributes.width, visible.attributes.height, 0, 0 ); 
  flush_expose( iconwin ); 
  XCopyArea(display, visible.pixmap, iconwin, gc, 0, 0,
	    visible.attributes.width, visible.attributes.height, 0, 0 );
  if ( Verbose ) fprintf(stdout,"In Redraw()\n");	
}

/***********************************************************************
 *  whichButton
 *
 *  Return the button that at the x,y coordinates. The button need not
 *  be visible ( drawn ). Return -1 if no button match.
 ***********************************************************************/
int whichButton( int x, int y ) {
  int index;

  for ( index=0; index < numb_of_apps; index++ ) {
    if ( x >= button_region[index].x &&
	 x <= button_region[index].i &&
	 y >= button_region[index].y &&
	 y <= button_region[index].j  ) {
      return( index + 1);
    }
  }
  return(-1);
}

/***********************************************************************
 * RunAppN(int app)
 *
 * Run the command given in the configuration file 'configfile'
 ***********************************************************************/
void RunAppN( int app ) {
  char *cmndstr;

  cmndstr = Parse(app);                  // Get command to pass to system
  if (Verbose) fprintf(stderr, "Command String: %s", cmndstr);
  if (cmndstr != NULL) system(cmndstr);  // if there's a command, run it
}

/***********************************************************************
 * Parse(int app)
 *
 * Parses the file 'configfile' for command to execute.
 ***********************************************************************/

char *Parse(int app) {
  FILE *fp;
  char *ptr  = NULL;
  char *cmnd = NULL;
  char *line = NULL;

  if ((fp = fopen(configfile, "r")) == NULL) {  // Can't find config file
    fprintf(stderr, "%s: Configuration File not found\n", configfile);
    exit(-1);
  }

  do {                               // Read Lines in config file
    line = readln(fp);
    if (line == NULL) break;         // if end of file, quit
    if ( (line[0] != '#') && (line[0] != '\n'))  // Ignore comments and blanks 
      if ( atoi(line) == app) break; // if we've found our app, quit looking
  } while (1);

  if (line == NULL) return(NULL);    // if we didn't find in, that just ignore

  ptr = strchr(line,'\t');                      // find first tab
  if (ptr == NULL)   ptr = strchr(line,' ');    // or space charater
  if (ptr == NULL)   return(NULL);              // if neither are there, its a badly
                                                // formatted line.  Ignore
  ptr++;
  cmnd = malloc((strlen(ptr)+1)*sizeof(char));  

  strcpy(cmnd, ptr);                 // Everything after number is passed to system

  free(line);
  fclose(fp);
  return(cmnd);
}

/***********************************************************************
 *   getPixmaps
 *    
 *   Load XPM data into X Pixmaps.
 *  
 *   Pixmap template contains the untouched window backdrop image.
 *   Pixmap visible is the template pixmap with buttons drawn on it.
 *          -- what is seen by the user.
 *   Pixmap buttons holds the images for individual buttons that are
 *          later copied onto Pixmap visible.
 ***********************************************************************/
void getPixmaps() {
  template.attributes.valuemask |= (XpmReturnPixels | XpmReturnExtensions);
  visible.attributes.valuemask  |= (XpmReturnPixels | XpmReturnExtensions);
  buttons.attributes.valuemask  |= (XpmReturnPixels | XpmReturnExtensions);
  
  /* Template Pixmap. Never Drawn To. */
  if ( XpmCreatePixmapFromData(	display, rootwin, backdrop_xpm,
				&template.pixmap, &template.mask, 
				&template.attributes) != XpmSuccess ) {
    fprintf(stderr, "Can't Create 'template' Pixmap\n");
    exit(1);
  }
  /* Visible Pixmap. Copied from template Pixmap and then drawn to. */
  if ( XpmCreatePixmapFromData(	display, rootwin, backdrop_xpm,
				&visible.pixmap, &visible.mask, 
				&visible.attributes) != XpmSuccess ) {
    fprintf(stderr, "Can't Create 'visible' Pixmap");
    exit(1);
  }
  
  /* Button Pixmap.  */
  if ( XpmCreatePixmapFromData(	display, rootwin, buttons_xpm,
				&buttons.pixmap, &buttons.mask, 
				&buttons.attributes) != XpmSuccess ) {
    fprintf(stderr, "Can't Create 'buttons' Pixmap");
    exit(1);
  }
}

/***********************************************************************
 *   flush_expose
 *
 *   Everyone else has one of these... Can't hurt to throw it in.
 ***********************************************************************/
int flush_expose(Window w) {
  XEvent      dummy;
  int         i=0;
  
  while (XCheckTypedWindowEvent(display, w, Expose, &dummy)) i++;
  return(i);
}

/***********************************************************************
 * 		show_usage
 ***********************************************************************/
void show_usage() {
  fprintf(stderr,"\n");
  fprintf(stderr,"usage: %s [-g geometry] [-d dpy] [-v] [-f <configfile>]\n",app_name);
  fprintf(stderr,"\n");
  fprintf(stderr," wmbutton Ver %d Rel %d\n",VER,REL);
  fprintf(stderr,"\n");
  fprintf(stderr,"-g  <geometry>   Window Geometry - ie: 64x64+10+10\n");
  fprintf(stderr,"-d  <display>    Display -  ie: 127.0.0.1:0.0\n"); 
  fprintf(stderr,"-f  <filename>   Full path to configuration file.\n");
  fprintf(stderr,"-v               Verbose Mode. \n");
  fprintf(stderr,"-h               Help. This message.\n");
  fprintf(stderr,"-m               Enable Middle Mouse functionality.\n\n");
  exit(-1);
}
