/*
 * WMAppl - Window Maker Application Launcher
 * Version:  0.1
 * Copyright (C) 1999 Casey Harkins
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <Xlib.h>
#include <Xutil.h>
#include <extensions/shape.h>
#include <xpm.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "interface.xpm"
#include "leftarr.xpm"
#include "rightarr.xpm"

#define APPNAME "WMAppLaunch"

typedef struct _XpmButton {
	Pixmap pixmap;
	int pressed;
	char *command;
	struct _XpmButton *last;
	struct _XpmButton  *next;
} XpmButton;


/* some globals */
Display *display;
int screen;
Window rootwin;
Window win;
Window iconwin;
GC gc;
Pixmap pixmap, pixmask;
XpmAttributes pixattr;
XpmButton *icons, sleft, sright;
char *appname;

int button_positions[] = {5, 5, 22, 22, 5, 23, 22, 40,
						23, 5, 40, 22, 23, 23, 40, 40,
						41, 5, 58, 22, 41, 23, 58, 40};

int scroll_positions[] = {5, 46, 22, 58, 42, 46, 59, 58};

/* x initialization crap */
void initx(int argc, char **argv) {
	int i;

	/* some x structs */
	XSizeHints xsh;
	XWMHints *xwmh;
	XClassHint xch;
	XGCValues xgcv;
	XTextProperty xtp;
	
	/* connect to default display */
	display = XOpenDisplay(getenv("DISPLAY"));
	if(!display) {
		fprintf(stderr, "Couldn't connect to display\n");
		exit(1);
	}

	/* get screen and root window */
	screen = DefaultScreen(display);
	rootwin = RootWindow(display, screen);

	/* set size hints 64 x 64 */
	xsh.flags = USSize | USPosition;
	xsh.width = 64;
	xsh.height = 64;
	XWMGeometry(display, screen, "64x64+0+0", NULL, 0,
		&xsh, &xsh.x, &xsh.y, &xsh.width, &xsh.height, &i);

	/* create the application window */
	win = XCreateSimpleWindow(display, rootwin,
		xsh.x, xsh.y, xsh.width, xsh.height, 0,
		BlackPixel(display, screen),
		WhitePixel(display, screen));

	if(!win) {
		fprintf(stderr, "Couldn't create window\n");
		exit(1);
	}

	/* create icon window */
	iconwin = XCreateSimpleWindow(display, rootwin,
		xsh.x, xsh.y, xsh.width, xsh.height, 0,
		BlackPixel(display, screen),
		WhitePixel(display, screen));

	if(!iconwin) {
		fprintf(stderr, "Couldn't create icon window\n");
		exit(1);
	}


	/* load interface pixmap */
	if(XpmCreatePixmapFromData(display, rootwin, interface_xpm,
		&pixmap, &pixmask, &pixattr)!=XpmSuccess) {
		fprintf(stderr, "Couldn't create interface\n");
		exit(1);
	}


	/* load scroll arrow pixmaps */
	if(XpmCreatePixmapFromData(display, rootwin, leftarr_xpm,
		&sleft.pixmap, NULL, NULL)!=XpmSuccess) {
		fprintf(stderr, "Couldn't create interface (left scroller) \n");
		exit(1);
	}
	sleft.pressed=0;
	if(XpmCreatePixmapFromData(display, rootwin, rightarr_xpm,
		&sright.pixmap, NULL, NULL)!=XpmSuccess) {
		fprintf(stderr, "Couldn't create interface (right scroller) \n");
		exit(1);
	}
	sright.pressed=0;

	/* set background pixmap for iconwin */
	XSetWindowBackgroundPixmap(display, win, pixmap);
	XSetWindowBackgroundPixmap(display, iconwin, pixmap);

	/* should query server to see if Shape extension is supported */

	/* setup shaped window */
	XShapeCombineMask(display, win, ShapeBounding, 
		0, 0, pixmask, ShapeSet);
	XShapeCombineMask(display, iconwin, ShapeBounding,
		0, 0, pixmask, ShapeSet);

	/* set window manager hints */
	xwmh = XAllocWMHints();
	xwmh->flags = WindowGroupHint | IconWindowHint | StateHint;
	xwmh->icon_window = iconwin;
	xwmh->window_group = win;
	xwmh->initial_state = WithdrawnState;
	XSetWMHints(display, win, xwmh);

	
	/* set class hints */
	appname =(char *)malloc(strlen(APPNAME));
	strcpy(appname, APPNAME);
	xch.res_name = appname; 
	xch.res_class = appname;
	XSetClassHint(display, win, &xch);

	/* set size hints */
	XSetWMNormalHints(display, win, &xsh);
	
	/* tell window manager app name */
	if(!XStringListToTextProperty(&appname, 1, &xtp)) {
		fprintf(stderr, "Couldn't create text property\n");
		exit(1);
	}
	XSetWMName(display, win, &xtp);


	/* create a graphics context */
	gc = XCreateGC(display, win, GCForeground | GCBackground, &xgcv);
	if(!gc) {
		fprintf(stderr, "Couldn't create graphics context\n");
		exit(1);
	}

	/* select events to catch */
	XSelectInput(display, win,
		ExposureMask |
		ButtonPressMask |
		ButtonReleaseMask |
		PointerMotionMask |
		StructureNotifyMask);
	XSelectInput(display, iconwin,
		ExposureMask |
		ButtonPressMask |
		ButtonReleaseMask |
		PointerMotionMask |
		StructureNotifyMask);

	/* set the command line for restarting */
	XSetCommand(display, win, argv, argc);

	/* map the main window */
	XMapWindow(display, win);

}


void draw_16button(XpmButton *b, int x, int y) {
		Pixel bl, wh;

		bl = BlackPixel(display, screen);
		wh = WhitePixel(display, screen);

		if(b->pressed) 
			XSetForeground(display, gc, bl);
		else
			XSetForeground(display, gc, wh);

		XDrawLine(display, win, gc, x, y, x+17, y);
		XDrawLine(display, win, gc, x, y, x, y+16);

		XDrawLine(display, iconwin, gc, x, y, x+17, y);
		XDrawLine(display, iconwin, gc, x, y, x, y+16);

		if(b->pressed)
			XSetForeground(display, gc, wh);
		else
			XSetForeground(display, gc, bl);

		XDrawLine(display, win, gc, x, y+17, x+17, y+17);
		XDrawLine(display, win, gc, x+17, y+1, x+17, y+17);

		XDrawLine(display, iconwin, gc, x, y+17, x+17, y+17);
		XDrawLine(display, iconwin, gc, x+17, y+1, x+17, y+17);

		XCopyArea(display, b->pixmap, win, gc, 0, 0, 16, 16, x+1, y+1);
		XCopyArea(display, b->pixmap, iconwin, gc, 0, 0, 16, 16, x+1, y+1);

}


void draw_1510button(XpmButton *b, int x, int y) {
		Pixel bl, wh;

		bl = BlackPixel(display, screen);
		wh = WhitePixel(display, screen);

		if(b->pressed) 
			XSetForeground(display, gc, bl);
		else
			XSetForeground(display, gc, wh);

		XDrawLine(display, win, gc, x, y, x+16, y);
		XDrawLine(display, win, gc, x, y, x, y+10);

		XDrawLine(display, iconwin, gc, x, y, x+16, y);
		XDrawLine(display, iconwin, gc, x, y, x, y+10);

		if(b->pressed)
			XSetForeground(display, gc, wh);
		else
			XSetForeground(display, gc, bl);

		XDrawLine(display, win, gc, x, y+11, x+16, y+11);
		XDrawLine(display, win, gc, x+16, y+1, x+16, y+11);

		XDrawLine(display, iconwin, gc, x, y+11, x+16, y+11);
		XDrawLine(display, iconwin, gc, x+16, y+1, x+16, y+11);

		XCopyArea(display, b->pixmap, win, gc, 0, 0, 15, 10, x+1, y+1);
		XCopyArea(display, b->pixmap, iconwin, gc, 0, 0, 15, 10, x+1, y+1);
}

void draw(void) {
	XpmButton *p;
	int i;

	for(i=0, p=icons; i<6 && p!=NULL; i++, p=p->next) 
		draw_16button(p, button_positions[i*4], button_positions[i*4+1]);

	draw_1510button(&sleft, scroll_positions[0], scroll_positions[1]);
	draw_1510button(&sright, scroll_positions[4], scroll_positions[5]);

}

XpmButton *which_button(int x, int y) {
		int i;
		XpmButton *p;

		for(i=0, p=icons; i<6 && p!=NULL; i++, p=p->next) {
			if(x>button_positions[i*4] && y>button_positions[i*4+1] &&
			   x<button_positions[i*4+2] && y<button_positions[i*4+3])
					return p;
		}

		if(x>scroll_positions[0] && y>scroll_positions[1] &&
			x<scroll_positions[2] && y<scroll_positions[3])
				return &sleft;
		
		if(x>scroll_positions[4] && y>scroll_positions[5] &&
			x<scroll_positions[6] && y<scroll_positions[7])
				return &sright;

		return NULL;
}

void event_loop(void) {

	XEvent e;
	XpmButton *b = NULL;
	int i=1;

	while(i) {

		/* get next event from queue */
		XNextEvent(display, &e);

		/* handle events */
		switch(e.type) {

			/* Expose event */
			case Expose:
				if(e.xexpose.count!=0) break;
				draw();
				break;

			/* Configure Notify event */
			case ConfigureNotify:
				draw();
				break;

			/* Destroy Notify event */
			case DestroyNotify:
				i=0;
				break;

			/* Button Press event */
			case ButtonPress:
				if(e.xbutton.button!=Button1) break;
				b = which_button(e.xbutton.x, e.xbutton.y);
				if(b!=NULL) b->pressed = 1; 
				draw();
				break;

			/* Button Release event */
			case ButtonRelease:
				if(e.xbutton.button!=Button1) break;
				if(b!=NULL && b == which_button(e.xbutton.x, e.xbutton.y)) {
				
					/* scroll left */
					if(b==&sleft) {
						if(icons->last!=NULL) {
							if(icons->last->last!=NULL)
								icons=icons->last->last;
							XClearWindow(display, win);
							XClearWindow(display, iconwin);
						}
					}
					/* scroll right */
					else if(b==&sright) {
						if(icons->next!=NULL) {
							if(icons->next->next!=NULL)
								icons=icons->next->next;
							XClearWindow(display, win);
							XClearWindow(display, iconwin);
						}
					}
					/* execute command */
					else {
						if(b->command!=NULL) system(b->command);
					}

				}
				if(b!=NULL) b->pressed = 0;
				draw();
				break;

		} // end event switch

	} // end while


} // end event_loop()


void loadIcons(char *filename, char *iconpath) {
	FILE *f;
	XpmButton *l, *c;
	char in, buf[1024], buf2[2048];
	int i=0, first=1;


	f=fopen(filename, "r");
	if(f==NULL) {
		fprintf(stderr, "Couldn't open file %s for reading.\n", filename);
		exit(1);
	}
	
	l=NULL;
	in=fgetc(f);
	while(in==' ' || in=='\t') in=fgetc(f);

	while(in!=EOF) {

		c=(XpmButton *)malloc(sizeof(XpmButton));
		if(first>0) {
			icons=c;
			first=0;
		}
		c->pressed=0;
		c->last=l;
		if(l!=NULL) l->next=c;
		c->next=NULL;
		
		i=0;
		while(in!=' ' && in!='\n' && in!='\t' && in!=EOF) { 
			buf[i++]=in;
			in=fgetc(f);
		}
		buf[i]='\0';

		/* prepend icon path */
		strcpy(buf2, iconpath);
		i=strlen(buf2);
		if(buf2[i-2]!='/') strcat(buf2,"/\0");
		strcat(buf2, buf);
		
		if(XpmReadFileToPixmap(display, win, buf2, &c->pixmap, NULL, NULL)
						!=XpmSuccess) {
			fprintf(stderr, "Error loading icon %s.\n", buf2);
			exit(1);
		}
		while(in==' ' || in=='\t') in=fgetc(f);


		i=0;
		while(in!='\n' && in!=EOF) { 

			buf[i++]=in;
			in=fgetc(f);
			

			/* backslash continues to next line */
			if(in=='\\') {
				while(in!='\n' && in!=EOF) in=fgetc(f);
				if(in!=EOF) in=fgetc(f);
				while(in==' ' || in=='\t') in=fgetc(f);
			}
			
		}

		buf[i]='\0';
		c->command=(char *)malloc(strlen(buf)*sizeof(char));
		strcpy(c->command, buf);

		l=c;

		in=fgetc(f);
	}
	fclose(f);
}

void show_usage() {
	fprintf(stderr, "\nusage\twmappl [-f configfile] [-i iconpath] [-V]\n");
	fprintf(stderr, "\t\t-f : reads configfile rather than ~/.wmapplrc\n");
	fprintf(stderr, "\t\t-i : prefixes icon filenames with iconpath\n");
	fprintf(stderr, "\t\t-V : show version information\n");
	exit(1);
}

void show_version() {
	fprintf(stderr, "\nwmappl v0.1 ");
	fprintf(stderr, "(http://www.pobox.com/~charkins/wmappl.html)\n");
	exit(1);
}



int main(int argc, char **argv) {
	int i;
	char *arg;
	char configfile[1024];	// config file to read from	
	char iconpath[1024];	// path to load icons from

	/* set default config file */
	strcpy(configfile, getenv("HOME"));
	strcat(configfile, "/.wmapplrc");

	/* set default icon path */
	iconpath[0]=0;
	
	/* parse command line arguments */
	for(i=1; i<argc; i++) {
		arg=argv[i];
		if(*arg=='-') {
			arg++;
			switch(*arg) {
			/* specify config file */
			case 'f':
				if(++i>=argc) show_usage();
				sscanf(argv[i], "%s", configfile);
				break;

			/* specify icon path */
			case 'i':
				if(++i>=argc) show_usage();
				sscanf(argv[i], "%s", iconpath);
				break;

			/* show version information */
			case 'V':
				show_version();
				break;

			/* anything else show usage information */
			default:
				show_usage();
				break;
				
			} // end switch
		} // end if
	} // end for
					

	initx(argc, argv);
	loadIcons(configfile, iconpath);
	event_loop();	
	return 0;
}
