/*
 ****************************************************************
 *
 * Copyright (C) 1998,1999 Atsushi Umeshima <ume@tka.att.ne.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************
 */

#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <math.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <fcntl.h>

#include <X11/xpm.h>
#include "wmgeneral/wmgeneral.h"
#include "wmamfm-master.xpm"
#include "wmamfm-mask.xbm"
#include "amfm.h"

// Defines
#define NAME "wmamfm"
#define CLASS "WMAmfm"
#define BACKCOLOR "#282828"
#define VERSION "1.1.2"
#define RCFILE "wmamfmrc"
#define FCODE(f)    (f-7600)/5
#define ACODE(f)    (f-531)/9
#define FMFREQ_MIN 7600
#define FMFREQ_MAX 10800
#define FMFREQ_STEP 5
#define AMFREQ_MIN 531
#define AMFREQ_MAX 1602
#define AMFREQ_STEP 9
#define VOL_MIN 0
#define VOL_MAX 30
#define VOLUME_MIN 0
#define VOLUME_MAX 100
#define INT_SHORT 10000
#define INT_LONG  50000
#define OFF_SLEEP_USEC 50000
#define ON_SLEEP_USEC 100000
#define TUNE_SLEEP_USEC 50000

// Global Variables
char *myname;
char temp[128];
XEvent Event;
int but_stat=-1;
int but_detail=-1;
int but_timer=0;
struct {
    unsigned int freq;
} *fm_presets, *am_presets;
char bckcol[256] = BACKCOLOR;
unsigned int fm_max_presets=0;
unsigned int am_max_presets=0;
unsigned int fm_preset_count=0;
unsigned int am_preset_count=0;
unsigned int preset_count=0;
int radio_status=OFF;
int radio_band=FM;
unsigned int volume=0;
unsigned int am_freq=AMFREQ_MIN;
unsigned int fm_freq=FMFREQ_MIN;
unsigned int cur_freq=FMFREQ_MIN;
char freqstr[20]="";
char volstr[20]="";
char *device=DEVICE;
int mixerfd;
int fd;
int interval = INT_LONG;
struct timeval interval_tv;
fd_set r_fds;
char *short_opts = "hvnap:";
struct option long_opts[] =
{
    {"help", 0, 0, 'h'},
    {"version", 0, 0, 'v'},
    {"on", 0, 0, 'n'},
    {"another", 0, 0, 'a'},
    {"path", 1, 0, 'p'},
    {NULL, 0, 0, 0}
};
struct sockaddr_un server;
int serv_sock, msg_sock;
int indx = 0;
char *rcfilepath = NULL;


// Functions
int probe_wmamfm(void);
void dummy_handler(void);
void exit_radio(void);
void wmamfm_routine(int argc, char *argv[]);
void parse_command_line(int argc, char *argv[]);
int parse_rc_file(char *filename);
unsigned int check_freq(unsigned int freq, int band);
void check_volume(void);
void usage(char *name);
void button_down(int button);
void button_up(int button);
void tune_up(void);
void tune_down(void);
void toggle_power(int status);
void toggle_band(void);
void preset_update(void);
void set_volume(void);
void general_freq_update(void);
void preset_handler(int preset_hint);
void draw_digital_preset(void);
void draw_digital_freq(void);
void draw_volume_bar(int vol);
void draw_digital_band(void);
void draw_power_led(void);
void version(void);
void make_freqstr(void);
void radio_on(void);
void radio_off(void);
void radio_tune(void);
unsigned int hundred_times_strtoint(char *str);

// Main
int main(int argc, char *argv[])
{
    myname = argv[0];
    parse_command_line(argc,argv);
    
    if(rcfilepath != NULL)
    {
	if (parse_rc_file(rcfilepath) == 1)
        {
	    perror("rcfile");
	    exit(1);
        }
    }
    else
    {
	strcpy(temp,(char*)getenv("HOME"));
	strcat(temp,"/.");
	strcat(temp,RCFILE);
	if (parse_rc_file(temp) == 1)
        {
	    strcpy(temp, "/usr/local/share/wmamfm/system.");
	    strcat(temp, RCFILE);
	    if (parse_rc_file(temp) == 1)
            {
		fprintf(stderr,"User(~/.%s) or global(/usr/local/share/wmamfm/system.%s) configuration files not found.\n",RCFILE,RCFILE);
		exit(1);
            }
        }
    }
    
    strcpy(server.sun_path, SOCK_NAME);
    if(indx == 1)
    {
        strcat(server.sun_path, "1");
    }
    server.sun_family = AF_UNIX;
    if(probe_wmamfm())
    {
        printf("wmamfm already running.\n");
        exit(1);
    }
    unlink(server.sun_path);
    if(bind(serv_sock, (struct sockaddr *) &server,
            strlen(server.sun_path)+sizeof(server.sun_family)) == -1)
    {
        perror("Error in bind");
        exit(1);
    }
    listen(serv_sock, 1);
    
    mixerfd = openmixer(indx);
    addrecsrc(mixerfd, SOUND_MIXER_LINE);
    
    fd = radioinit(device);
    if(radioprobe(fd))
    {
        radioclose(fd);
        fd = radioinit(device);
        if(radioprobe(fd))
        {
            radio_status = OFF;
        }
    }
    
    if(radio_status == ON)
    {
        radio_on();
    }
    else
    {
        radio_off();
    }
    
    signal(SIGINT, (void *)exit_radio);
    signal(SIGTERM, (void *)exit_radio);
    
    wmamfm_routine(argc, argv);
    exit(0);
}

int probe_wmamfm(void)
{
    char str[16];
    
    if((serv_sock = socket(AF_UNIX, SOCK_STREAM, 0)) == -1)
    {
        perror("Error in socket");
        exit(1);
    }
    
    if(connect(serv_sock, (struct sockaddr *) &server,
               strlen(server.sun_path)+sizeof(server.sun_family)) != -1)
    {
#ifdef DEBUG
        printf("<<< %s\n", STR_ENQ);
#endif
        if(write(serv_sock, STR_ENQ, sizeof(STR_ENQ)) != -1)
        {
            memset(str, 0, sizeof(str));
            if(read(serv_sock, str, 16) != -1)
            {
#ifdef DEBUG
                printf(">>> %s\n", str);
#endif
                ;
            }
            else
            {
                perror("Error in read");
                exit(1);
            }
        }
        else
        {
            perror("Error in write");
            exit(1);
        }
        close(serv_sock);
        return(1);
    }
    return(0);
}

void dummy_handler(void)
{
    signal(SIGALRM, SIG_IGN);
    return;
}

void exit_radio(void)
{
    radio_off();
    radioclose(fd);
    closemixer(mixerfd);
    unlink(server.sun_path);
    exit(0);
}

void radio_on()
{
    setvolume(mixerfd, 0);
    radioon(fd);
    if(cur_freq>0.0)
    {
        radiotune(fd, freqstr, radio_band);
    }
    usleep(ON_SLEEP_USEC);
    setvolume(mixerfd, volume);
}

void radio_off(void)
{
    setvolume(mixerfd, 0);
    radiooff(fd);
    usleep(OFF_SLEEP_USEC);
    setvolume(mixerfd, volume);
}

void radio_tune(void)
{
    setvolume(mixerfd, 0);
    radiotune(fd, freqstr, radio_band);
    usleep(TUNE_SLEEP_USEC);
    setvolume(mixerfd, volume);
}

void wmamfm_routine(int argc, char *argv[])
{
    int i=-1;
    int rval;
    char msg_buf[1024];
    char *p;
    char *tokens = " \t\n";
    int loop_timer=0;
    
    openXwindow(argc,argv,wmamfm_master_xpm,wmamfm_mask_bits,
                wmamfm_mask_width,wmamfm_mask_height);
    AddMouseRegion(0,4,4,12,13);		// FINE TUNE UP
    AddMouseRegion(1,4,15,12,24);		// FINE TUNE DOWN
    AddMouseRegion(2,4,27,12,41);		// LEFT PRESET SCAN BUTTON
    AddMouseRegion(3,39,27,47,41);		// RIGHT PRESET SCAN BUTTON
    AddMouseRegion(4,4,44,24,59);		// ON/OFF BUTTON
    AddMouseRegion(5,25,44,45,59);		// FM/AM BUTTON
    AddMouseRegion(6,50,29,58,59);		// VOLUME BAR
    AddMouseRegion(7,13,4,59,24);		// VOLUME BAR
    AddMouseRegion(8,15,27,36,41);		// VOLUME BAR
    
    set_volume();
    draw_power_led();
    general_freq_update();
    
    while (1)
    {
	while (XPending(display))
	{
            XNextEvent(display, &Event);
            switch (Event.type) 
            {
                case MapNotify:
                    RedrawWindow();
                    break;
                case Expose:
                    RedrawWindow();
                    break;
                case DestroyNotify:
                    XCloseDisplay(display);
                    exit_radio();
                    break;
                case ButtonPress:
                    i = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);
                    switch (i)
                    {
                        case 0:			// FINE TUNE UP
                            button_down(0);
                            but_timer = 0;
                            break;
                        case 1:			// FINE TUNE DOWN
                            button_down(1);
                            but_timer = 0;
                            break;
                        case 2:			// LEFT PRESET SCAN BUTTON
                            button_down(2);
                            break;
                        case 3:			// RIGHT PRESET SCAN BUTTON
                            button_down(3);
                            break;
                        case 4:			// ON/OFF BUTTON
                            button_down(4);
                            break;
                        case 5:			// FM/AM BUTTON
                            button_down(5);
                            break;
                        case 6:			// VOLUME BAR
                            button_down(6);
                            volume=(59-Event.xbutton.y)*10/3;
                            set_volume();
                        case 7:			// FINE TUNE UP
                            but_timer = 0;
                            break;
                    }
                    but_stat = i;
                    but_detail = Event.xbutton.button;
                    break;
                case ButtonRelease:
                    interval = INT_LONG;
                    switch (but_stat) 
                    {
                        case 0:			// FINE TUNE UP
                            button_up(0);
                            break;
                        case 1:			// FINE TUNE DOWN
                            button_up(1);
                            break;
                        case 2:			// LEFT PRESET SCAN BUTTON
                            button_up(2);
                            break;
                        case 3:			// RIGHT PRESET SCAN BUTTON
                            button_up(3);
                            break;
                        case 4:			// ON/OFF BUTTON
                            button_up(4);
                            break;
                        case 5:			// FM/AM BUTTON
                            button_up(5);
                            break;
                        case 6:			// VOLUME BAR
                            button_up(6);
                            break;
                    }
                    if (i == but_stat && but_stat >= 0)
                    {
                        switch (i)
                        {
                            case 0:			// FINE TUNE UP
                                if (!(but_timer >= 10))
                                {
                                    tune_up();
                                }
                                break;
                            case 1:			// FINE TUNE DOWN
                                if (!(but_timer >= 10))
                                {
                                    tune_down();
                                }
                                break;
                            case 2:			// LEFT PRESET SCAN BUTTON
                                preset_handler(-1);
                                break;
                            case 3:			// RIGHT PRESET SCAN BUTTON
                                preset_handler(1);
                                break;
                            case 4:			// ON/OFF BUTTON
                                toggle_power(radio_status);
                                break;
                            case 5:			// FM/AM BUTTON
                                toggle_band();
                                break;
                            case 6:			// VOLUME BAR
                                volume=(59-Event.xbutton.y)*10/3;
                                set_volume();
                                break;
                            case 7:			// DIGITAL FREQ BOX
                                switch (Event.xbutton.button)
                                {
                                    case Button3:		
                                    case Button2:		
                                        if (!(but_timer >= 10))
                                        {
                                            tune_up();
                                        }
                                        break;
                                    case Button1:		
                                        if (!(but_timer >= 10))
                                        {
                                            tune_down();
                                        }
                                        break;
                                }
                                break;
                            case 8:			// DIGITAL PRESETS BOX
                                switch (Event.xbutton.button)
                                {
                                    case Button3:		
                                    case Button2:		
                                        preset_handler(1);
                                        break;
                                    case Button1:		
                                        preset_handler(-1);
                                        break;
                                }
                                break;
                        }
                    }
                    but_stat = -1;
                    but_detail = -1;
                    i = -1;
            }
	}
        
	if (i == but_stat && but_stat >= 0)
	{
            but_timer++;
            if(but_timer == 50)
            {
                interval = INT_SHORT;
            }
            
            switch (i)
            {
                case 6:			// VOLUME BAR
                    volume=(59-Event.xbutton.y)*10/3;
                    set_volume();
                    break;
            }
            
            if(but_timer >= 10)
            {
                switch (i)
                {
                    case 0:			// FINE TUNE UP (+) AUTO-REPEAT
                        tune_up();
                        break;
                    case 1:			// FINE TUNE DOWN (-) AUTO-REPEAT
                        tune_down();
                        break;
                    case 7:			// FINE TUNE UP (+) AUTO-REPEAT
                        switch(but_detail)
                        {
                            case Button3:		
                            case Button2:		
                                tune_up();
                                break;
                            case Button1:
                                tune_down();
                                break;
                        }
                        break;
                }
            }
	}
#ifndef NOMIXER
        volume = getvolume(mixerfd);
#endif
        set_volume();
        
	loop_timer += interval;
	if(loop_timer > 1000000)
        {
	    loop_timer=0;
	    if(radio_status == ON && radioprobe(fd))
            {
		radio_status = OFF;
		draw_power_led();
            }
        }
        
        interval_tv.tv_sec = 0;
        interval_tv.tv_usec = interval;
        FD_ZERO(&r_fds);
        FD_SET(serv_sock, &r_fds);
        if(select(serv_sock+1,  &r_fds, (fd_set *)0, (fd_set *)0, &interval_tv)
	   == -1)
        {
            perror("Error in select\n");
            continue;
        }
        if(FD_ISSET(serv_sock, &r_fds))
        {
            if((msg_sock = accept(serv_sock, (struct sockaddr *)0, (int *)0))
               == -1)
                perror("Error in accept");
            else
            {
                memset(msg_buf, 0, sizeof(msg_buf));
                if((rval = read(msg_sock, msg_buf, 1024)) != -1)
                {
#ifdef DEBUG
                    printf(">>> %s\n", msg_buf);
#endif
                    if ((p = strtok(msg_buf,tokens)) != NULL)
		    {
                        if (strcmp(msg_buf, STR_ON) == 0)
			{
                            toggle_power(OFF);
                        }
                        else if (strcmp(msg_buf, STR_OFF) == 0)
			{
                            toggle_power(ON);
			}
                        else if (strcmp(msg_buf, STR_ENQ) == 0)
			{
                            ;
			}
                        else if (strcmp(msg_buf, STR_FM) == 0)
			{
                            if ((p = strtok(NULL,tokens)) != NULL)
			    {
                                radio_band = FM;
                                cur_freq = fm_freq = hundred_times_strtoint(p);
                                general_freq_update();
			    }
			}
                        else if (strcmp(msg_buf, STR_AM) == 0)
			{
                            if ((p = strtok(NULL,tokens)) != NULL)
			    {
                                radio_band = AM;
                                cur_freq = am_freq = atoi(p);
                                general_freq_update();
			    }
			}
                        else if (strcmp(msg_buf, STR_VOL) == 0)
			{
                            if ((p = strtok(NULL,tokens)) != NULL)
			    {
                                volume=atoi(p);
                                set_volume();
			    }
			}
                        else
			{
                            ;
			}
		    }
		}
                else
                    perror("Error in read");
                
		if(write(msg_sock, STR_ACK, sizeof(STR_ACK)) == -1)
                    perror("Error in write");
#ifdef DEBUG
                printf("<<< %s\n", STR_ACK);
#endif
                close(msg_sock);
            }
        }
    }
}

void parse_command_line(int argc, char *argv[])
{
    int opt_index = 0, c;
    
    while ((c = getopt_long(argc, argv, short_opts, long_opts, &opt_index)) != EOF) {
        switch( c ) {
            case 'h':
                usage( argv[0] );
                exit(0);
                break;
            case 'v':
                version();
                exit(0);
                break;
            case 'n':
                radio_status = ON;
                break;
	    case 'a':
                indx = 1;
                break;
	    case 'p':
		rcfilepath = (char *)malloc(strlen(optarg)+1);
                strcpy(rcfilepath, optarg);
                break;
            default:
                exit(1);
        }
    }
}

int parse_rc_file(char *filename)
{
    char temp[128];
    char *p;
    FILE *fp;
    int fm_count=1;
    int am_count=1;
    char *tokens = " \t\n";
    
    if ((fp = fopen(filename,"r")) == NULL)
    {
        return 1;
    }
    
    while(fgets(temp, 128, fp)) 
    {
        if ((p = strtok(temp,tokens)) != NULL)
        {
            if ((p = strchr(temp,'#')) != NULL) 
            {
                *p = '\0';
            }
            if ((strlen(temp)) != 0)
            {
                if (strcmp(temp, "FMStation:") == 0)
                {
                    fm_max_presets++;
                }
                else if (strcmp(temp, "AMStation:") == 0)
                {
                    am_max_presets++;
                }
            }
        }
    }
    
    fm_presets = malloc(sizeof(*fm_presets) * (fm_max_presets+1));
    am_presets = malloc(sizeof(*am_presets) * (am_max_presets+1));
    fm_presets[0].freq = FMFREQ_MIN;
    am_presets[0].freq = AMFREQ_MIN;
    
    fseek(fp, 0L, SEEK_SET);
    
    while(fgets(temp, 128, fp)) 
    {
        if ((p = strtok(temp,tokens)) != NULL)
        {
            if ((p = strchr(temp,'#')) != NULL) 
            {
                *p = '\0';
            }
            if ((strlen(temp)) != 0)
            {
                if ((p = strtok(NULL,tokens)) != NULL)
                {
                    if (strcmp(temp, "Device:") == 0)
                    {
                        device = (char *)malloc(strlen(p)+1);
                        strcpy(device, p);
#ifdef DEBUG
			printf("Device: %s\n", device);
#endif
                    }
                    else if (strcmp(temp, "FMStation:") == 0)
                    {
                        fm_presets[fm_count].freq =
                            check_freq(hundred_times_strtoint(p), FM);
#ifdef DEBUG
			printf("FMStation: %d\n", fm_presets[fm_count].freq);
#endif
                        fm_count++;
                    }
                    else if (strcmp(temp, "AMStation:") == 0)
                    {
                        am_presets[am_count].freq = check_freq(atoi(p), AM);
#ifdef DEBUG
			printf("AMStation: %d\n", am_presets[am_count].freq);
#endif
                        am_count++;
                    }
                    else if (strcmp(temp, "Startup:") == 0)
                    {
                        if (strcmp(p, "AM") == 0)
                        {
                            radio_band = AM;
                        }
                        if (strcmp(p, "FM") == 0)
                        {
                            radio_band = FM;
                        }
                        if ((p = strtok(NULL,tokens)) != NULL)
                        {
                            if (radio_band == FM)
                            {
                                cur_freq = fm_freq = 
                                    check_freq(hundred_times_strtoint(p), FM);
                            }
                            else
                            {
                                cur_freq = am_freq = check_freq(atoi(p), AM);
                            }
                        }
                        if ((p = strtok(NULL,tokens)) != NULL)
                        {
                            volume = atoi(p);
                            check_volume();
                        }
#ifdef DEBUG
			printf("Startup: %s, %d, %d\n", 
			       (radio_band==FM)?"FM":"AM",
			       cur_freq, volume);
#endif
                    }
                }
            }
        }
    }
    
    make_freqstr();
    
    fclose(fp);
    return 0;
}

void make_freqstr(void)
{
    if(radio_band == FM)
    {
        sprintf(freqstr, "%03d", FCODE(cur_freq));
#ifdef DEBUG
	printf("FM: %d, cmd: %s\n", cur_freq, freqstr);
#endif
    }
    else
    {
        sprintf(freqstr, "%03d", ACODE(cur_freq));
#ifdef DEBUG
	printf("AM: %d, cmd: %s\n", cur_freq, freqstr);
#endif
    }
}

unsigned int check_freq(unsigned int freq, int band)
{
    if (band == FM)
    {
        if(freq > FMFREQ_MAX)
        {
            freq = FMFREQ_MAX;
        }
        else if(freq < FMFREQ_MIN)
        {
            freq = FMFREQ_MIN;
        }
	freq -= freq % FMFREQ_STEP;
    }
    else
    {
        if(freq > AMFREQ_MAX)
        {
            freq = AMFREQ_MAX;
        }
        else if(freq < AMFREQ_MIN)
        {
            freq = AMFREQ_MIN;
        }
	freq -= freq % AMFREQ_STEP;
    }
    return freq;
}

void check_volume(void)
{
    if(volume > VOLUME_MAX)
    {
        volume = VOLUME_MAX;
    }
    else if(volume < VOLUME_MIN)
    {
        volume = VOLUME_MIN;
    }
}

void usage(char *name)
{
    version();
    
    printf("\n");
    printf("Usage: %s [OPTIONS]\n", myname);
    printf("\n");
    printf("  -n, --on                       "
           "turn radio on initially on startup\n");
    printf("  -a, --another                      "
           "run wmamfm as second instance\n");
    printf("  -p, --path                      "
           "specify the path of rcfile\n");
    printf("  -h, --help                     "
           "display this help and exit\n");
    printf("  -v, --version                  "
           "display the version and exit\n");
    printf("\n");
    printf("Author: Atsushi Umeshima <ume@tka.att.ne.jp>\n");
    printf("   Web: http://members.xoom.com/ume/amfm.html\n");
}

void button_down(int button)
{
    switch (button)
    {
        case 0:			// FINE TUNE UP
            copyXPMArea(50, 103, 7, 10, 4, 4);
            RedrawWindowXYWH(4, 4, 7, 10);
            break;
        case 1:			// FINE TUNE DOWN
            copyXPMArea(65, 103, 7, 10, 4, 15);
            RedrawWindowXYWH(4, 15, 7, 10);
            break;
        case 2:			// LEFT PRESET SCAN BUTTON
            copyXPMArea(52, 86, 9, 15, 4, 27);
            RedrawWindowXYWH(4, 27, 9, 15);
            break;
        case 3:			// RIGHT PRESET SCAN BUTTON
            copyXPMArea(71, 86, 9, 15, 39, 27);
            RedrawWindowXYWH(39, 27, 9, 15);
            break;
        case 4:			// ON/OFF BUTTON
            copyXPMArea(21, 103, 21, 16, 4, 44);
            RedrawWindowXYWH(4, 44, 21, 16);
            break;
        case 5:			// FM/AM BUTTON
            copyXPMArea(21, 86, 21, 16, 25, 44);
            RedrawWindowXYWH(25, 44, 21, 16);
            break;
        case 6:			// VOLUME BAR
            break;
    }
}

void button_up(int button)
{
    switch (button)
    {
        case 0:			// FINE TUNE UP
            copyXPMArea(43, 103, 7, 10, 4, 4);
            RedrawWindowXYWH(4, 4, 7, 10);
            break;
        case 1:			// FINE TUNE DOWN
            copyXPMArea(58, 103, 7, 10, 4, 15);
            RedrawWindowXYWH(4, 15, 7, 10);
            break;
        case 2:			// LEFT PRESET SCAN BUTTON
            copyXPMArea(43, 86, 9, 15, 4, 27);
            RedrawWindowXYWH(4, 27, 9, 15);
            break;
        case 3:			// RIGHT PRESET SCAN BUTTON
            copyXPMArea(62, 86, 9, 15, 39, 27);
            RedrawWindowXYWH(39, 27, 9, 15);
            break;
        case 4:			// ON/OFF BUTTON
            copyXPMArea(0, 103, 21, 16, 4, 44);
            RedrawWindowXYWH(4, 44, 21, 16);
            break;
        case 5:			// FM/AM BUTTON
            copyXPMArea(0, 86, 21, 16, 25, 44);
            RedrawWindowXYWH(25, 44, 21, 16);
            break;
        case 6:			// VOLUME BAR
            break;
    }
}

void tune_up(void)
{
    if(radio_band == FM)
    {
	if (cur_freq <= (FMFREQ_MAX-FMFREQ_STEP))
	{
            cur_freq += FMFREQ_STEP;
	}
	else if (cur_freq == FMFREQ_MAX)
	{
            cur_freq = FMFREQ_MIN;
	}
	else
	{
            cur_freq = FMFREQ_MAX;
	}
        fm_freq = cur_freq;
    }
    else
    {
	if (cur_freq <= (AMFREQ_MAX-AMFREQ_STEP))
	{
            cur_freq += AMFREQ_STEP;
	}
	else if (cur_freq == AMFREQ_MAX)
	{
            cur_freq = AMFREQ_MIN;
	}
	else
	{
            cur_freq = AMFREQ_MAX;
	}
        am_freq = cur_freq;
    }
    general_freq_update();
}

void tune_down(void)
{
    if(radio_band == FM)
    {
	if (cur_freq >= (FMFREQ_MIN+FMFREQ_STEP))
	{
            cur_freq -= FMFREQ_STEP;
	}
	else if (cur_freq == FMFREQ_MIN)
	{
            cur_freq = FMFREQ_MAX;
	}
	else
	{
            cur_freq = FMFREQ_MIN;
	}
        fm_freq = cur_freq;
    }
    else
    {
	if (cur_freq >= (AMFREQ_MIN+AMFREQ_STEP))
	{
            cur_freq -= AMFREQ_STEP;
	}
	else if (cur_freq == AMFREQ_MIN)
	{
            cur_freq = AMFREQ_MAX;
	}
	else
	{
            cur_freq = AMFREQ_MIN;
	}
        am_freq = cur_freq;
    }
    general_freq_update();
}

void toggle_power(int status)
{
    if(status == OFF)
    {
        if(!radioprobe(fd))
        {
            radio_status = ON;
            radio_on();
        }
    }
    else
    {
        radio_status = OFF;
        radio_off();
    }
    draw_power_led();
}

void toggle_band(void)
{
    if(radio_band == FM)
    {
        radio_band = AM;
        cur_freq = am_freq;
    }
    else
    {
        radio_band = FM;
        cur_freq = fm_freq;
    }
    general_freq_update();
}

void preset_update(void)
{
    int i;
    preset_count = 0;
    if(radio_band == FM)
    {
        for (i=1;i<=fm_max_presets;i++)
        {
            if (cur_freq == fm_presets[i].freq)
            {
                preset_count = i;
                fm_preset_count = preset_count;
            }
        }
    }
    else
    {
        for (i=1;i<=am_max_presets;i++)
        {
            if (cur_freq == am_presets[i].freq)
            {
                preset_count = i;
                am_preset_count = preset_count;
            }
        }
    }
}

void set_volume()
{
    int vol;
    
    check_volume();
    vol = volume*3/10;
#ifdef DEB
    printf("vol: %d, volume: %d\n", vol, volume);
#endif
    setvolume(mixerfd, (int)volume);
    draw_volume_bar(vol);
}

void general_freq_update()
{
    cur_freq = check_freq(cur_freq, radio_band);
    make_freqstr();
    if(radio_status == ON)
    {
        radio_tune();
    }
    preset_update();
    draw_digital_preset();
    draw_digital_freq();
    draw_digital_band();
}

void preset_handler(int preset_hint)
{
    if(radio_band == FM)
    {
        preset_count = fm_preset_count;
    }
    else
    {
        preset_count = am_preset_count;
    }
    
    if (preset_hint < 0)
    {
        if (preset_count <= 1)
        {
            if(radio_band == FM)
            {
                preset_count = fm_max_presets;
            }
            else
            {
                preset_count = am_max_presets;
            }
        }
        else
        {
            preset_count--;
        }
    } 
    else
    {
        if(radio_band == FM)
        {
            if(fm_max_presets > 0)
            {
                if(preset_count == fm_max_presets)
                {
                    preset_count = 1;
                }
                else
                {
                    preset_count++;
                }
            }
        }
        else
        {
            if(am_max_presets > 0)
            {
                if(preset_count == am_max_presets)
                {
                    preset_count = 1;
                }
                else
                {
                    preset_count++;
                }
            }
        }
    }
    
    if(radio_band == FM)
    {
        fm_preset_count = preset_count;
        cur_freq = fm_presets[fm_preset_count].freq;
        fm_freq = cur_freq;
    }
    else
    {
        am_preset_count = preset_count;
        cur_freq = am_presets[am_preset_count].freq;
        am_freq = cur_freq;
    }
    
    general_freq_update();
}

void draw_digital_preset(void)
{
    char temp[10];
    char *p = temp;
    int j=0,k=17;
    
    sprintf(temp, "%02d", preset_count);
    
    while (*p == '0' && j<2)
    {
        copyXPMArea(93, 64, 9, 11, k, 29);
        k += 9;
        j++;
        p++;
    }
    while (j<2) 
    {
        copyXPMArea((*p-'0')*9, 64, 9, 11, k, 29);
        k += 9;
        p++;
        j++;
    }
    RedrawWindowXYWH(17, 29, 18, 11);
}

void draw_digital_freq(void)
{
    char temp[10];
    char *p = temp;
    int j=0,k=15;
    
    if(radio_band == FM)
    {
        sprintf(temp,"%05d",cur_freq);
        
        if (*p == '1')
        {
            copyXPMArea(13, 64, 4, 11, k, 6);
            k += 4;
            j++;
            p++;
        }
        else
        {
            copyXPMArea(93, 64, 4, 11, k, 6);
            k += 4;
            j++;
            p++;
        }
        while (j<5) 
        {
            if (j == 3)
            {
                copyXPMArea(90, 64, 3, 11, k, 6);
                k += 3;
            }
	    copyXPMArea((*p-'0')*9, 64, 9, 11, k, 6);
	    k += 9;
            p++;
            j++;
        }
    }
    else
    {
        sprintf(temp,"%04d",cur_freq);
        
        copyXPMArea(93, 64, 4, 11, k, 6);
        k += 4;
        
        if (*p == '0')
        {
            copyXPMArea(93, 64, 9, 11, k, 6);
            k += 9;
            p++;
            j++;
        }
        while (j<4) 
        {
            copyXPMArea((*p-'0')*9, 64, 9, 11, k, 6);
            k += 9;
            p++;
            j++;
        }
        copyXPMArea(93, 64, 3, 11, k, 6);
    }
    RedrawWindowXYWH(15, 6, 43, 11);
}

void draw_volume_bar(int vol)
{
    if(vol < VOL_MAX)
    {
        copyXPMArea(90, 86, 9, VOL_MAX-vol, 50, 29);
    }
    if(vol > VOL_MIN)
    {
        copyXPMArea(81, 86+VOL_MAX-vol, 9, vol, 50, 29+VOL_MAX-vol); 
    }
    RedrawWindowXYWH(50, 29, 9, 30); 
}

void draw_digital_band(void)
{
    if(radio_band == FM)
    {
	copyXPMArea(97, 77, 12, 6, 29, 49); // FM
	copyXPMArea(103, 64, 13, 5, 45, 18); // MHz
    }
    else
    {
	copyXPMArea(110, 77, 12, 6, 29, 49); // AM
	copyXPMArea(103, 70, 13, 5, 45, 18); // KHz
    }
    RedrawWindowXYWH(29, 49, 12, 6); // FM/AM
    RedrawWindowXYWH(45, 18, 13, 5); // MHz/KHz
}

void draw_power_led(void)
{
    if(radio_status == ON)
    {
	copyXPMArea(58, 114, 15, 3, 7, 47); 
    }
    else
    {
	copyXPMArea(43, 114, 15, 3, 7, 47); 
    }
    RedrawWindowXYWH(7, 47, 15, 3);
}

void version(void)
{
    printf("This is " NAME " " VERSION "\n");
    printf("Copyright (C) 1998,1999 Atsushi Umeshima\n");
}
