/*
** Copyright (C) 1998 Steve Borho <borho@stlnet.com>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>

#include "mount.h"

void            printConfigHelp(int argc, char *argv[]);
void            configureMount();

static int      signalMountApp = FALSE;
static int      goingDown = FALSE;

static gchar   *iconName[] =
{
    "Compact Disk Icon",
    "Floppy Disk Icon",
    "Zip Disk Icon",
    "Linux (ext2) Icon",
    "DOS (FAT) Icon",
    "MAC (HFS) Icon",
    "Unknown Icon",
    "User Defined 1",
    "User Defined 2"
};

static gchar   *capType[] =
{
    "Do not show capacity",
    "Show percent full",
    "Show percent available",
    "Show amount full",
    "Show amount available"
};

static char    *helpFile = "online-help.txt";
static bool     showHelp = FALSE;
static char     initialMountPoint[50];

struct allInfo *aiPtr;

int             main(int argc, char *argv[])
{
    int             optchar;
    int             i;

    gtk_init(&argc, &argv);

    initialMountPoint[0] = 0;
    while ((optchar = getopt(argc, argv, "Hhism:")) != EOF)
    {
        switch (optchar)
        {
        case 'i':
            ignoreNFS = TRUE;
            break;
        case 's':
            signalMountApp = TRUE;
            break;
        case 'H':
            showHelp = TRUE;
            break;
        case 'm':
            safeCopy(initialMountPoint, optarg, sizeof(initialMountPoint));
            break;
        case 'h':
        default:
            printConfigHelp(argc, argv);
            break;
        }
    }

    aiPtr = (struct allInfo *)malloc(sizeof(allInfo));
    if (aiPtr == NULL)
    {
        fprintf(stderr, "Unable to allocate memory\n");
        exit(-1);
    }

    bzero((void *)aiPtr, sizeof(allInfo));

    parseFstab(aiPtr);
    readProperties(aiPtr);

    /* Any mount points to monitor? */
    if (aiPtr->numMounts == 0)
    {
        fprintf(stderr, "No mount points available to monitor\n");
        exit(0);
    }
    aiPtr->curMount = aiPtr->numMounts - 1;

    /* Assign default icons if they haven't been overwritten */
    for (i = 0; i < MAX_ICONS; i++)
        if (aiPtr->icoIsLoaded[i] == 0)
            setIcon(i, iconDefaultName[i], aiPtr);

    configureMount(aiPtr);
    return 0;
}

void            destroy(GtkWidget * widget, gpointer data)
{
    goingDown = TRUE;
    gtk_main_quit();
}

gint            delete_event(GtkWidget * widget, GdkEvent * event, gpointer data)
{
    goingDown = TRUE;
    return (FALSE);
}

void            saveConfiguration(GtkWidget * widget, gpointer data)
{
    goingDown = TRUE;
    saveProperties(aiPtr);
    if (signalMountApp)
        kill(getppid(), SIGHUP);
}

void            helpDialog(GtkWidget * widget, gpointer data)
{
    GtkWidget      *dialog;
    GtkWidget      *button;
    GtkWidget      *label;
    GtkWidget      *box;
    GtkWidget      *scrollbar;
    GtkWidget      *text;
    FILE           *olh;
    char            buf[256];

    sprintf(buf, "%s/%s", ICONDIR, helpFile);
    if ((olh = fopen(buf, "r")) == NULL)
        return;

    dialog = gtk_dialog_new();
    button = gtk_button_new_with_label("Ok, I've seen enough");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
        button, TRUE, TRUE, 0);
    gtk_widget_show(button);

    if (showHelp)
    {
        gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
            GTK_SIGNAL_FUNC(destroy), NULL);
        gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
            GTK_SIGNAL_FUNC(delete_event), NULL);
    }

    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(gtk_widget_destroy),
        GTK_OBJECT(dialog));

    label = gtk_label_new("mount.app Online help");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
        label, TRUE, TRUE, 0);
    gtk_widget_show(label);

    /*
     * Load online help text file into text area widget
     */
    box = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
        box, TRUE, TRUE, 0);

    text = gtk_text_new(NULL, NULL);
    gtk_widget_set_usize(text, 500, 500);
    gtk_text_set_editable(GTK_TEXT(text), FALSE);
    gtk_text_set_word_wrap(GTK_TEXT(text), FALSE);

    scrollbar = gtk_vscrollbar_new(GTK_TEXT(text)->vadj);
    gtk_box_pack_start(GTK_BOX(box), scrollbar, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(box), text, TRUE, TRUE, 0);

    /*
     * Work around differences in GTK+ 1.0.x
     */
#ifndef GTK_HAVE_FEATURES_1_1_0
    gtk_widget_realize(GTK_WIDGET(text));
#endif

    gtk_widget_show(scrollbar);
    gtk_widget_show(text);
    gtk_widget_show(box);

    while (fgets(buf, sizeof(buf), olh))
    {
        gtk_text_insert(GTK_TEXT(text), NULL, &text->style->black, NULL,
            buf, -1);
    }
    fclose(olh);

    /*
     * Run dialog
     */
    gtk_widget_show(dialog);
}

void            mountPointBox(GtkWidget *, GtkTooltips *, allInfo *);
void            externalBox(GtkWidget *, GtkTooltips *, allInfo *);
void            iconConfBox(GtkWidget *, GtkTooltips *, allInfo *);
void            fontBox(GtkWidget *, GtkTooltips *, allInfo *);
void            colorBox(GtkWidget *, GtkTooltips *, allInfo *);
void            extrasBox(GtkWidget *, GtkTooltips *, allInfo *);

void            configureMount(allInfo * ai)
{
    GtkWidget      *window;
    GtkTooltips    *toolTip;
    GtkWidget      *dialogBox;

    GtkWidget      *noteBook;
    GtkWidget      *frame;
    GtkWidget      *label;
    GtkWidget      *frameBox;

    GtkWidget      *align;
    GtkWidget      *buttonBox;
    GtkWidget      *saveButton;
    GtkWidget      *abortButton;
    GtkWidget      *helpButton;

    if (showHelp)
    {
        helpDialog(NULL, NULL);
        gtk_main();
        return;
    }

    /*
     * Create window widget, then connect delete events 
     */
    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_container_border_width(GTK_CONTAINER(window), 10);
    gtk_window_set_title(GTK_WINDOW(window), "mount.app Configurator");
    gtk_signal_connect(GTK_OBJECT(window), "destroy",
        GTK_SIGNAL_FUNC(destroy), NULL);
    gtk_signal_connect(GTK_OBJECT(window), "delete_event",
        GTK_SIGNAL_FUNC(delete_event), NULL);

    dialogBox = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(window), dialogBox);

    /*
     * Create a tool tip widget
     */
    toolTip = gtk_tooltips_new();

    /*
     * Create a notebook widget
     */
    noteBook = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(noteBook), GTK_POS_TOP);
    gtk_widget_show(noteBook);
    gtk_box_pack_start(GTK_BOX(dialogBox), noteBook, FALSE, TRUE, 0);

    /*
     * Create "mount point config" frame on notebook
     */
    frame = gtk_frame_new(NULL);
    gtk_container_border_width(GTK_CONTAINER(frame), 10);
    gtk_widget_show(frame);

    label = gtk_label_new("Mounts");
    gtk_notebook_append_page(GTK_NOTEBOOK(noteBook), frame, label);
    gtk_widget_show(label);

    frameBox = gtk_hbox_new(FALSE, 10);
    gtk_container_add(GTK_CONTAINER(frame), frameBox);
    gtk_widget_show(frameBox);
    gtk_container_border_width(GTK_CONTAINER(frameBox), 10);
    mountPointBox(frameBox, toolTip, ai);

    /*
     * Create "Edit external" frame on notebook
     */
    frame = gtk_frame_new(NULL);
    gtk_container_border_width(GTK_CONTAINER(frame), 10);
    gtk_widget_show(frame);

    label = gtk_label_new("Commands");
    gtk_notebook_append_page(GTK_NOTEBOOK(noteBook), frame, label);
    gtk_widget_show(label);

    frameBox = gtk_hbox_new(FALSE, 10);
    gtk_container_add(GTK_CONTAINER(frame), frameBox);
    gtk_widget_show(frameBox);
    gtk_container_border_width(GTK_CONTAINER(frameBox), 10);
    externalBox(frameBox, toolTip, ai);

    /*
     * Create "Icons" frame on notebook
     */
    frame = gtk_frame_new(NULL);
    gtk_container_border_width(GTK_CONTAINER(frame), 10);
    gtk_widget_show(frame);

    label = gtk_label_new("Icons");
    gtk_notebook_append_page(GTK_NOTEBOOK(noteBook), frame, label);
    gtk_widget_show(label);

    frameBox = gtk_vbox_new(FALSE, 10);
    gtk_container_add(GTK_CONTAINER(frame), frameBox);
    gtk_widget_show(frameBox);
    gtk_container_border_width(GTK_CONTAINER(frameBox), 10);
    iconConfBox(frameBox, toolTip, ai);

    /*
     * Create "Font" frame on notebook
#if GTK_MINOR_VERSION > 1
     frame = gtk_frame_new("Pick fonts");
     gtk_container_border_width(GTK_CONTAINER(frame), 10);
     gtk_widget_show(frame);

     label = gtk_label_new("Fonts");
     gtk_notebook_append_page(GTK_NOTEBOOK(noteBook), frame, label);
     gtk_widget_show(label);

     frameBox = gtk_hbox_new(FALSE, 10);
     gtk_container_add(GTK_CONTAINER(frame), frameBox);
     gtk_widget_show(frameBox);
     gtk_container_border_width(GTK_CONTAINER(frameBox), 10);
     fontBox(frameBox, toolTip, ai);
#endif
     */

    /*
     * Create "Color" frame on notebook
#if GTK_MINOR_VERSION > 1
     frame = gtk_frame_new("Pick colors");
     gtk_container_border_width(GTK_CONTAINER(frame), 10);
     gtk_widget_show(frame);

     label = gtk_label_new("Colors");
     gtk_notebook_append_page(GTK_NOTEBOOK(noteBook), frame, label);
     gtk_widget_show(label);

     frameBox = gtk_hbox_new(FALSE, 10);
     gtk_container_add(GTK_CONTAINER(frame), frameBox);
     gtk_widget_show(frameBox);
     gtk_container_border_width(GTK_CONTAINER(frameBox), 10);
     colorBox(frameBox, toolTip, ai);
#endif
     */

    /*
     * Create "Extras" frame on notebook
     */
    frame = gtk_frame_new("Stuff that doesn't fit anywhere else");
    gtk_container_border_width(GTK_CONTAINER(frame), 10);
    gtk_widget_show(frame);

    label = gtk_label_new("Extras");
    gtk_notebook_append_page(GTK_NOTEBOOK(noteBook), frame, label);
    gtk_widget_show(label);

    frameBox = gtk_vbox_new(FALSE, 10);
    gtk_container_add(GTK_CONTAINER(frame), frameBox);
    gtk_widget_show(frameBox);
    gtk_container_border_width(GTK_CONTAINER(frameBox), 10);
    extrasBox(frameBox, toolTip, ai);

    /*
     * Create save and abort buttons and connect click events
     */
    saveButton = gtk_button_new_with_label("Make it so");
    abortButton = gtk_button_new_with_label("Abort");
    helpButton = gtk_button_new_with_label("Help");
    gtk_signal_connect(GTK_OBJECT(saveButton), "clicked",
        GTK_SIGNAL_FUNC(saveConfiguration), NULL);
    gtk_signal_connect_object(GTK_OBJECT(saveButton), "clicked",
        GTK_SIGNAL_FUNC(gtk_widget_destroy),
        GTK_OBJECT(window));
    gtk_signal_connect_object(GTK_OBJECT(abortButton), "clicked",
        GTK_SIGNAL_FUNC(gtk_widget_destroy),
        GTK_OBJECT(window));
    gtk_signal_connect(GTK_OBJECT(helpButton), "clicked",
        GTK_SIGNAL_FUNC(helpDialog), NULL);

    /*
     * Pack buttons into buttonBox, and align them into the right-bottom
     * corner.
     */
    align = gtk_alignment_new(1.0, 1.0, 0.0, 0.0);
    gtk_widget_show(align);

    buttonBox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(buttonBox), saveButton, TRUE, TRUE, 5);
    gtk_widget_show(saveButton);
    gtk_box_pack_start(GTK_BOX(buttonBox), helpButton, TRUE, TRUE, 5);
    gtk_widget_show(helpButton);
    gtk_box_pack_start(GTK_BOX(buttonBox), abortButton, TRUE, TRUE, 5);
    gtk_widget_show(abortButton);
    gtk_widget_show(buttonBox);

    gtk_container_add(GTK_CONTAINER(align), buttonBox);
    gtk_box_pack_start(GTK_BOX(dialogBox), align, FALSE, TRUE, 5);

    /*
     * Pack button box into dialogBox
     */
    gtk_widget_show(dialogBox);
    gtk_widget_show(window);

    gtk_main();
}

/*
 *   EXTERNAL COMMANDS CONFIGURATION
 */

void            setString(GtkWidget * widget, gpointer data)
{
    safeCopy((gchar *) data, gtk_entry_get_text(GTK_ENTRY(widget)), STR);
}

void            externalBox(GtkWidget * page, GtkTooltips * toolTip,
        allInfo * ai)
{
    GtkWidget      *commandFrame;
    GtkWidget      *box;
    GtkWidget      *label;
    GtkWidget      *entry;

    commandFrame = gtk_frame_new("External Commands (Optional)");
    gtk_box_pack_start(GTK_BOX(page), commandFrame, FALSE, FALSE, 0);
    gtk_widget_show(commandFrame);

    box = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(box);
    gtk_container_set_border_width(GTK_CONTAINER(box), 10);
    gtk_container_add(GTK_CONTAINER(commandFrame), box);

    label = gtk_label_new("Mount");
    gtk_widget_show(label);
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    entry = gtk_entry_new_with_max_length(STR);
    gtk_entry_set_text(GTK_ENTRY(entry), mountcmd);
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
        GTK_SIGNAL_FUNC(setString), mountcmd);
    gtk_box_pack_start(GTK_BOX(box), entry, TRUE, TRUE, 0);
    gtk_widget_show(entry);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), entry,
        "This command will be used to mount a drive.  "
        "If this entry is left blank, a fast internal mount command "
        "will be used."
        "%s will be substituted with the mount point string. ",
        NULL);

    label = gtk_label_new("Unmount");
    gtk_widget_show(label);
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    entry = gtk_entry_new_with_max_length(STR);
    gtk_entry_set_text(GTK_ENTRY(entry), umountcmd);
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
        GTK_SIGNAL_FUNC(setString), umountcmd);
    gtk_box_pack_start(GTK_BOX(box), entry, TRUE, TRUE, 0);
    gtk_widget_show(entry);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), entry,
        "This command will be used to unmount a drive.  "
        "If this entry is left blank, a fast internal unmount command "
        "will be used."
        "%s will be substituted with the mount point string. ",
        NULL);

    label = gtk_label_new("Eject command");
    gtk_widget_show(label);
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    entry = gtk_entry_new_with_max_length(STR);
    gtk_entry_set_text(GTK_ENTRY(entry), ejectcmd);
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
        GTK_SIGNAL_FUNC(setString), ejectcmd);
    gtk_box_pack_start(GTK_BOX(box), entry, TRUE, TRUE, 0);
    gtk_widget_show(entry);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), entry,
        "This command will be used to eject a drive.  "
        "If this entry is left blank, a fast internal eject command "
        "will be used."
        "%s will be substituted with the mount point string. ",
        NULL);

    box = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(box);
    gtk_box_pack_start(GTK_BOX(page), box, FALSE, FALSE, 0);

    label = gtk_label_new("File browser command");
    gtk_widget_show(label);
    gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

    entry = gtk_entry_new_with_max_length(STR);
    gtk_entry_set_text(GTK_ENTRY(entry), opencmd);
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
        GTK_SIGNAL_FUNC(setString), opencmd);
    gtk_box_pack_start(GTK_BOX(box), entry, FALSE, FALSE, 0);
    gtk_widget_show(entry);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), entry,
        "This command will be issued when the user double-clicks "
        "on a drive icon.  It should start a file browser in the "
        "directory of that mount point.  %s will be substituted with "
        "the directory name",
        NULL);
}

/*
 *   FONTS AND COLORS CONFIGURATION
 */

GtkWidget      *fontSel;

void            colorSet(GtkWidget * widget, gpointer data)
{
    g_print("Color set\n");
}

void            fontSet(GtkWidget * widget, gpointer data)
{
       gtk_font_selection_dialog_get_font_name(
       GTK_FONT_SELECTION_DIALOG(fontSel));
       gtk_widget_destroy(fontSel);
}

void            changeFont(GtkWidget * widget, gpointer data)
{
    fontSel = gtk_font_selection_dialog_new ("Select a new font");
    gtk_window_position(GTK_WINDOW(fontSel), GTK_WIN_POS_MOUSE);
    gtk_window_set_modal (GTK_WINDOW(fontSel), TRUE);
    gtk_signal_connect(GTK_OBJECT(fontSel), "destroy",
            GTK_SIGNAL_FUNC(gtk_widget_destroyed),
            &fontSel);
    gtk_signal_connect(
            GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(fontSel)->ok_button),
            "clicked", GTK_SIGNAL_FUNC(fontSet), data);
    gtk_widget_show(fontSel);
}

void            changeColor(GtkWidget * widget, gpointer data)
{
    GtkWidget      *colorSelDlg;
    GtkWidget      *colorSel;

    colorSelDlg = gtk_color_selection_dialog_new("Select a new color");
    gtk_window_set_modal (GTK_WINDOW(colorSelDlg), TRUE);
    colorSel = GTK_COLOR_SELECTION_DIALOG(colorSelDlg)->colorsel;
    gtk_signal_connect(
            GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(colorSelDlg)->ok_button),
            "clicked", GTK_SIGNAL_FUNC(colorSet), data);
    gtk_signal_connect_object(
            GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(colorSelDlg)->ok_button),
            "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
            GTK_OBJECT (colorSelDlg));
    gtk_signal_connect_object(
            GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(colorSelDlg)->cancel_button),
            "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
            GTK_OBJECT (colorSelDlg));
    gtk_widget_show(colorSelDlg);
}

void            fontBox(
    GtkWidget * page,
    GtkTooltips * toolTip,
    allInfo * ai)
{
    GtkWidget      *box;
    GtkWidget      *label;
    GtkWidget      *button;
    GtkWidget      *sep;

    box = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(box);
    gtk_container_add(GTK_CONTAINER(page), box);

    button = gtk_button_new_with_label("Mount name text font");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(changeFont), txtfont);
    gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 10);
    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), button,
        "Set the font used in the mount name text.",
        NULL);
    gtk_widget_show(button);

    button = gtk_button_new_with_label("Capacity text font");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(changeFont), capfont);
    gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 10);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), button,
        "Set the font used in the capacity text.", NULL);
    gtk_widget_show(button);
}

void            colorBox(
    GtkWidget * page,
    GtkTooltips * toolTip,
    allInfo * ai)
{
    GtkWidget      *box;
    GtkWidget      *button;

    box = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(box);
    gtk_container_add(GTK_CONTAINER(page), box);

    button = gtk_button_new_with_label("Mount name text color");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(changeColor), txtcolor);
    gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 10);
    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), button,
        "Open a color selection dialog for the mount name text",
        NULL);
    gtk_widget_show(button);

    button = gtk_button_new_with_label("Capacity text color");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(changeColor), capcolor);
    gtk_box_pack_start(GTK_BOX(box), button, TRUE, TRUE, 10);
    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), button,
        "Open a color selection dialog for the capacity text",
        NULL);
    gtk_widget_show(button);
}

/*
 *   ICON CONFIGURATION
 */

static gint     curIcon;
GtkWidget      *fileNameEntry;
GtkWidget      *pixmapList;
const gchar    *fileNameKey = "file_name_key";
static gint     listNo = 0;

void            iconNameChanged(GtkWidget * filenameList, gpointer funcData)
{
    GList          *dlist;
    GtkObject      *listItem;
    gchar          *str;

    if (goingDown)
        return;

    dlist = GTK_LIST(filenameList)->selection;
    if (!dlist)
    {
        /*
         * If the user has unselected a name, then revert to default
         */
        safeCopy(aiPtr->iconFileName[curIcon], iconDefaultName[curIcon], STR);
        gtk_entry_set_text(GTK_ENTRY(fileNameEntry),
            aiPtr->iconFileName[curIcon]);
        return;
    }

    /*
     * Extract filename and store
     */
    listItem = GTK_OBJECT(dlist->data);
    str = gtk_object_get_data(listItem, fileNameKey);

    safeCopy(aiPtr->iconFileName[curIcon], str, STR);
    gtk_entry_set_text(GTK_ENTRY(fileNameEntry), aiPtr->iconFileName[curIcon]);
}

void            refreshPixmaps(GtkWidget * widget, gpointer data)
{
    GtkWidget      *item;
    gint            len, i;
    DIR            *dir;
    struct dirent  *dirent;

    if (listNo)
        gtk_list_clear_items(GTK_LIST(pixmapList), 0, listNo);
    listNo = 0;

    dir = opendir(icondir);
    if (dir == 0)
    {
        perror("Invalid icon directory");
        return;
    }

    while ((dirent = readdir(dir)) != NULL)
    {
        static char     match_string[] = ".xpm";

        /*
         * Screen out non-xpm files
         */
        len = strlen(dirent->d_name);
        if (len < 4)
            continue;
        for (i = 0; i < 4; i++)
            if (dirent->d_name[len - i] != match_string[4 - i])
                break;
        if (i < 4)
            continue;

        item = gtk_list_item_new_with_label(dirent->d_name);
        gtk_container_add(GTK_CONTAINER(pixmapList), item);
        gtk_widget_show(item);
        gtk_object_set_data(GTK_OBJECT(item), fileNameKey,
            strdup(dirent->d_name));
        listNo++;
    }

    closedir(dir);
}

#ifdef GTK_HAVE_FEATURES_1_1_7
#define newIconChosenWrapper(w,d)    newIconChosen(d,w)
void            newIconChosen(gpointer data, GtkWidget * widget)
#else
#define newIconChosenWrapper(w,d)    newIconChosen(w,d)
void            newIconChosen(GtkWidget * widget, gpointer data)
#endif
{
    /*
     * A new mount point has been selected.  Update the file name
     */
    curIcon = (int)(long int)data;
    gtk_entry_set_text(GTK_ENTRY(fileNameEntry), aiPtr->iconFileName[curIcon]);
}

void            iconConfBox(
    GtkWidget * page,
    GtkTooltips * toolTip,
    allInfo * ai)
{
    GtkWidget      *selFrame;
    GtkWidget      *hbox;
    GtkWidget      *label;
    GtkWidget      *entry;

    GtkWidget      *pickBox;

    GtkWidget      *iconList;
    GtkWidget      *menu;
    GtkWidget      *item;
    gint            i;

    GtkWidget      *scrolledWindow;
    GtkWidget      *vbox;

    /* Create hbox for left side + icon sel frame */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(page), hbox, FALSE, FALSE, 0);
    gtk_widget_show(hbox);

    /* Create vbox for left side */
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);
    gtk_widget_show(vbox);

    /*
     * Make option menu of icon types
     */
    menu = gtk_menu_new();
    for (i = 0; i < sizeof(iconName) / sizeof(gchar *); i++)
    {
        item = gtk_menu_item_new_with_label(iconName[i]);
        gtk_widget_show(item);
        gtk_signal_connect_object(GTK_OBJECT(item), "activate",
            GTK_SIGNAL_FUNC(newIconChosen), (gpointer) (long int)i);
        gtk_menu_append(GTK_MENU(menu), item);
    }
    iconList = gtk_option_menu_new();
    gtk_widget_show(iconList);
    gtk_option_menu_set_menu(GTK_OPTION_MENU(iconList), GTK_WIDGET(menu));
    gtk_box_pack_start(GTK_BOX(vbox), iconList, FALSE, FALSE, 10);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), iconList,
        "Select an icon type whose pixmap you want to replace",
        NULL);

    /*
     * Icon filename entry field
     */
    label = gtk_label_new("Filename");
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
    gtk_widget_show(label);

    fileNameEntry = gtk_entry_new_with_max_length(STR);
    gtk_entry_set_editable(GTK_ENTRY(fileNameEntry), FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), fileNameEntry, FALSE, FALSE, 0);
    gtk_widget_show(fileNameEntry);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), fileNameEntry,
        "The filename of the pixmap which will be used for this "
        "icon type.  "
        "Any changes you make will not take effect until mount.app "
        "is restarted.",
        NULL);

    /*
     * Create pixmap selection frame
     */
    selFrame = gtk_frame_new("Find Pixmap");
    gtk_box_pack_start(GTK_BOX(hbox), selFrame, FALSE, FALSE, 10);
    gtk_widget_show(selFrame);

    pickBox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(selFrame), pickBox);
    gtk_container_set_border_width(GTK_CONTAINER(pickBox), 10);
    gtk_widget_show(pickBox);

    label = gtk_label_new("Path to pixmaps");
    gtk_widget_show(label);
    gtk_box_pack_start(GTK_BOX(pickBox), label, FALSE, FALSE, 0);

    entry = gtk_entry_new_with_max_length(STR);
    gtk_entry_set_text(GTK_ENTRY(entry), icondir);
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
        GTK_SIGNAL_FUNC(setString), icondir);
    gtk_signal_connect(GTK_OBJECT(entry), "activate",
        GTK_SIGNAL_FUNC(refreshPixmaps), icondir);
    gtk_entry_select_region(GTK_ENTRY(entry), 0, STR);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(pickBox), entry, TRUE, TRUE, 0);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), entry,
        "This is the directory where mount.app will look for all "
        "of it's pixmaps.", NULL);

    /*
     * Show list of pixmaps in icondir
     */
    scrolledWindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_set_border_width(GTK_CONTAINER(scrolledWindow), 10);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledWindow),
        GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

    gtk_widget_set_usize(scrolledWindow, 90, 150);
    gtk_box_pack_start(GTK_BOX(pickBox), scrolledWindow, FALSE, FALSE, 0);
    gtk_widget_show(scrolledWindow);

    pixmapList = gtk_list_new();
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolledWindow),
            pixmapList);
    gtk_widget_show(pixmapList);
    gtk_signal_connect(GTK_OBJECT(pixmapList), "selection_changed",
        GTK_SIGNAL_FUNC(iconNameChanged), NULL);

    refreshPixmaps(page, icondir);
    newIconChosenWrapper(page, 0);
}

/*
 *   EXTRAS CONFIGURATION
 */

void            cycleClicked(GtkWidget * widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON(widget)->active != aiPtr->idleCycle)
        aiPtr->idleCycle = !aiPtr->idleCycle;
}

void            newbieClicked(GtkWidget * widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON(widget)->active != aiPtr->newbieMessages)
        aiPtr->newbieMessages = !aiPtr->newbieMessages;
}

void            showCharClicked(GtkWidget * widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON(widget)->active != aiPtr->showErrorChars)
        aiPtr->showErrorChars = !aiPtr->showErrorChars;
}

void            descriptClicked(GtkWidget * widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON(widget)->active != aiPtr->descriptChar)
        aiPtr->descriptChar = !aiPtr->descriptChar;
}

void            extrasBox(GtkWidget * page, GtkTooltips * toolTip, allInfo * ai)
{
    GtkWidget      *button;
    char           *cycleName = "Cycle through mount points when idle";
    char           *newbieName = "Show friendly error messages";
    char           *descriptName = "Use descriptive capacity indicators";
    char           *ErrorCharName = "Use single characters to indicate errors";

    button = gtk_check_button_new_with_label(cycleName);
    gtk_box_pack_start(GTK_BOX(page), button, FALSE, FALSE, 5);
    gtk_widget_show(button);

    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button),
        aiPtr->idleCycle);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(cycleClicked), NULL);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), button,
        "Make mount.app cycle "
        "through all the mount points when it is idle "
        "(ie. no events for 5 seconds).",
        NULL);

    button = gtk_check_button_new_with_label(newbieName);
    gtk_box_pack_start(GTK_BOX(page), button, FALSE, FALSE, 5);
    gtk_widget_show(button);

    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button),
        aiPtr->newbieMessages);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(newbieClicked), NULL);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), button,
        "Display friendly messages when common errors occur while "
        "mounting or unmounting devices.  "
        "Can be annoying after the first or second time you see "
        "them.",
        NULL);

    button = gtk_check_button_new_with_label(ErrorCharName);
    gtk_box_pack_start(GTK_BOX(page), button, FALSE, FALSE, 5);
    gtk_widget_show(button);

    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button),
        aiPtr->showErrorChars);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(showCharClicked), NULL);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), button,
        "If the message boxes begin to annoy you, you can enable "
        "this feature which will place a character underneath the "
        "mount point name describing any errors which occur. "
        "P = not enough permisions. B = device busy. R = drive not ready.",
        NULL);

    button = gtk_check_button_new_with_label(descriptName);
    gtk_box_pack_start(GTK_BOX(page), button, FALSE, FALSE, 5);
    gtk_widget_show(button);

    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button),
        aiPtr->descriptChar);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
        GTK_SIGNAL_FUNC(descriptClicked), NULL);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), button,
        "When displaying capacity in terms of MB, this button "
        "toggles the use of < > characters to differentiate between "
        "disk space used and disk space remaining.",
        NULL);
}

/*
 *   MOUNT POINT CONFIGURATION
 */

GtkWidget      *firstCB;
GtkWidget      *ignoreCB;
GtkWidget      *ejectCB;
GtkWidget      *showCapList;
GtkWidget      *nameEntry;
GtkWidget      *iconList;
MPInfo         *mp;

void            firstEvent(GtkWidget * widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON(widget)->active)
        strcpy(initialMnt, mp->mountStr);
    else if (strcmp(initialMnt, mp->mountStr) == 0)
        *initialMnt = 0;
}

void            ignoreEvent(GtkWidget * widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON(widget)->active != mp->ignored)
        mp->ignored = !mp->ignored;
}

void            ejectChange(GtkWidget * widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON(widget)->active != mp->ejectFlag)
        mp->ejectFlag = !mp->ejectFlag;
}

void            showCapEvent(GtkWidget * widget, gpointer data)
{
    aiPtr->mpi[aiPtr->curMount]->showCap = (int)(long int)widget;
}

void            nameChange(GtkWidget * widget, gpointer data)
{
    safeCopy(aiPtr->mpi[aiPtr->curMount]->name,
        gtk_entry_get_text(GTK_ENTRY(widget)), NAME_LEN);
}

void            iconChosen(GtkWidget * widget, gpointer data)
{
    aiPtr->mpi[aiPtr->curMount]->icon = (int)(long int)widget;
}

void            mpChosen(gpointer data, GtkWidget * widget)
{
    int             newMp = (int)(long int)data;

    /*
     * A new mount point has been selected.  Update the configurables
     */
    aiPtr->curMount = newMp;
    mp = aiPtr->mpi[newMp];

    gtk_entry_set_text(GTK_ENTRY(nameEntry), mp->name);
    gtk_entry_select_region(GTK_ENTRY(nameEntry), 0, NAME_LEN);
    gtk_option_menu_set_history(GTK_OPTION_MENU(iconList),
        mp->icon);

    gtk_option_menu_set_history(GTK_OPTION_MENU(showCapList),
        mp->showCap);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(ignoreCB),
        aiPtr->mpi[newMp]->ignored);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(ejectCB),
        aiPtr->mpi[newMp]->ejectFlag);

    if (strcmp(initialMnt, mp->mountStr) == 0)
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(firstCB), 1);
    else
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(firstCB), 0);
}

void            mountPointBox(
    GtkWidget * page,
    GtkTooltips * toolTip,
    allInfo * ai)
{
    GtkWidget      *configBox;
    GtkWidget      *attributeFrame;
    GtkWidget      *mpList;
    GtkWidget      *menu;
    GtkWidget      *item;
    GtkWidget      *label;
    GtkWidget      *vbox;
    gint            i;
    char            buf[30];

    /*
     * Make option menu of mount points
     */
    label = gtk_label_new("Mount Point");
    gtk_widget_show(label);

    menu = gtk_menu_new();
    for (i = 0; i < ai->numMounts; i++)
    {
        item = gtk_menu_item_new_with_label(ai->mpi[i]->mountStr);
        gtk_widget_show(item);
        gtk_signal_connect_object(GTK_OBJECT(item), "activate",
            GTK_SIGNAL_FUNC(mpChosen), (gpointer) (long int)i);
        gtk_menu_append(GTK_MENU(menu), item);
    }
    mpList = gtk_option_menu_new();
    gtk_option_menu_set_menu(GTK_OPTION_MENU(mpList), GTK_WIDGET(menu));
    gtk_widget_show(mpList);
    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), mpList,
        "Select a mount point from this drop down list "
        "and then edit it's attributes to the right",
        NULL);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), mpList, FALSE, TRUE, 0);
    gtk_container_add(GTK_CONTAINER(page), vbox);

    /*
     * Create button for "ignore"
     */
    ignoreCB = gtk_check_button_new_with_label("Ignore");
    gtk_signal_connect(GTK_OBJECT(ignoreCB), "clicked",
        GTK_SIGNAL_FUNC(ignoreEvent), NULL);
    gtk_box_pack_start(GTK_BOX(vbox), ignoreCB, FALSE, FALSE, 5);
    gtk_widget_show(ignoreCB);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), ignoreCB,
        "When this button is down, mount.app will not display this "
        "mount point",
        NULL);

    /*
     * Create a frame widget
     */
    attributeFrame = gtk_frame_new("Attributes");
    gtk_box_pack_start(GTK_BOX(page), attributeFrame, FALSE, FALSE, 0);

    configBox = gtk_vbox_new(FALSE, 0);

    /* TODO: assertion `widget->parent == NULL' failed */
    gtk_container_add(GTK_CONTAINER(attributeFrame), configBox);
    gtk_container_set_border_width(GTK_CONTAINER(configBox), 10);
    gtk_widget_show(attributeFrame);

    /*
     * Make option menu of icon types
     */
    menu = gtk_menu_new();
    for (i = 0; i < sizeof(iconName) / sizeof(gchar *); i++)
    {
        item = gtk_menu_item_new_with_label(iconName[i]);
        gtk_widget_show(item);
        gtk_signal_connect_object(GTK_OBJECT(item), "activate",
            GTK_SIGNAL_FUNC(iconChosen), (gpointer) (long int)i);
        gtk_menu_append(GTK_MENU(menu), item);
    }
    iconList = gtk_option_menu_new();
    gtk_option_menu_set_menu(GTK_OPTION_MENU(iconList), GTK_WIDGET(menu));
    gtk_widget_show(iconList);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), iconList,
        "Pick an icon type from the drop down list which "
        "corresponds to the selected mount point",
        NULL);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox);

    gtk_box_pack_start(GTK_BOX(vbox), iconList, FALSE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(configBox), vbox, FALSE, TRUE, 5);

    /*
     * Create button for "Show first"
     */
    firstCB = gtk_check_button_new_with_label("Show first");
    gtk_widget_show(firstCB);
    gtk_signal_connect(GTK_OBJECT(firstCB), "clicked",
        GTK_SIGNAL_FUNC(firstEvent), NULL);
    gtk_box_pack_start(GTK_BOX(configBox), firstCB, TRUE, TRUE, 5);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), firstCB,
        "When this button is down, mount.app will display this "
        "mount point first",
        NULL);

    /*
     * Create button for "eject"
     */
    ejectCB = gtk_check_button_new_with_label("Eject disk after unmounting");
    gtk_widget_show(ejectCB);
    gtk_signal_connect(GTK_OBJECT(ejectCB), "clicked",
        GTK_SIGNAL_FUNC(ejectChange), NULL);
    gtk_box_pack_start(GTK_BOX(configBox), ejectCB, TRUE, TRUE, 5);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), ejectCB,
        "Should this drive be ejected after it is unmounted?",
        NULL);

    /*
     * Make option menu of capacity display types
     */
    menu = gtk_menu_new();
    for (i = 0; i < sizeof(capType) / sizeof(gchar *); i++)
    {
        item = gtk_menu_item_new_with_label(capType[i]);
        gtk_signal_connect_object(GTK_OBJECT(item), "activate",
            GTK_SIGNAL_FUNC(showCapEvent), (gpointer) (long int)i);
        gtk_widget_show(item);
        gtk_menu_append(GTK_MENU(menu), item);
    }
    showCapList = gtk_option_menu_new();
    gtk_option_menu_set_menu(GTK_OPTION_MENU(showCapList), GTK_WIDGET(menu));
    gtk_widget_show(showCapList);
    gtk_box_pack_start(GTK_BOX(configBox), showCapList, FALSE, FALSE, 5);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), showCapList,
        "Pick how you want the capacity of this device displayed",
        NULL);

    /*
     * Mount point name entry field and text label
     */
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox);

    sprintf(buf, "Name displayed (max %d): ", NAME_LEN - 1);
    label = gtk_label_new(buf);
    gtk_widget_show(label);

    nameEntry = gtk_entry_new_with_max_length(NAME_LEN - 1);
    gtk_entry_set_text(GTK_ENTRY(nameEntry), "Mount name");
    gtk_signal_connect(GTK_OBJECT(nameEntry), "changed",
        GTK_SIGNAL_FUNC(nameChange), NULL);
    gtk_widget_show(nameEntry);

    gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), nameEntry, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(configBox), vbox, TRUE, TRUE, 0);

    gtk_tooltips_set_tip(GTK_TOOLTIPS(toolTip), nameEntry,
        "This is the name which will be displayed by mount.app "
        "when viewing this mount point",
        NULL);

    /*
     * Pack the config box onto the page
     */
    /* gtk_container_add(GTK_CONTAINER(page), configBox); */
    gtk_widget_show(configBox);

    if (initialMountPoint[0] == 0)
        mpChosen((gpointer) 0, page);
    else
    {
        int             j;

        for (j = 0; j < ai->numMounts; j++)
        {
            if (strcmp(initialMountPoint, ai->mpi[j]->mountStr) == 0)
                break;
        }
        if (j >= ai->numMounts)
            j = 0;
        mpChosen((gpointer)(long int) j, page);
        gtk_option_menu_set_history(GTK_OPTION_MENU(mpList), j);
    }
}

void            printConfigHelp(int argc, char *argv[])
{
    fprintf(stderr, "mount.conf %s - Configurator for mount.app\n", VERSION);
    fprintf(stderr, "Copyright (C) 1998  Steve Borho <borho@stlnet.com>\n");
    fprintf(stderr, "This software comes with ABSOLUTELY NO WARRANTY\n");
    fprintf(stderr, "This software is free software, and you are welcome "
        "to redistribute it\n");
    fprintf(stderr, "under certain conditions\n");
    fprintf(stderr, "See the COPYING file for a more complete notice.\n\n");
    fprintf(stderr, "usage:\n\n   %s [options]\n", argv[0]);
    fprintf(stderr, "\noptions:\n\n");
    fprintf(stderr, "   -h | -help             display this help screen\n");
    fprintf(stderr, "   -H                     display FAQ window\n");
    fprintf(stderr, "   -i                     ignore NFS mounts\n");
    fprintf(stderr, "   -s                     SIGHUP parent when saving\n");
    fprintf(stderr, "   -m                     specify initial mount point\n");
    fprintf(stderr, "\nThis is intended to be called internally by mount.app\n");
    exit(0);
}

void            setIcon(int num, char *filename, allInfo * ai)
{
    safeCopy(ai->iconFileName[num], filename, STR);
    ai->icoIsLoaded[num] = 1;
}
